package jp.sourceforge.sxdbutils.query;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import jp.sourceforge.sxdbutils.TypeMappings;
import jp.sourceforge.sxdbutils.bean.BeanColumn;
import jp.sourceforge.sxdbutils.bean.ColumnNameMapping;
import jp.sourceforge.sxdbutils.bean.ColumnProperty;
import jp.sourceforge.sxdbutils.bean.NameMapping;
import jp.sourceforge.sxdbutils.bean.VersionColumnProperty;
import jp.sourceforge.sxdbutils.util.OthersUtils;
import jp.sourceforge.sxdbutils.util.ReflectionUtil;

/**
 * BeanQueryFactoryのビルダーです。
 * 
 * @author chinpei
 * 
 */
public class BeanQueryFactoryBuilder {

	private static final NameMapping DEFAULT_NAME_MAPPING = new ColumnNameMapping();

	protected String tableName;
	protected final Class beanClass;
	protected final NameMapping nameMapping;
	protected Set updateKeyColumnSet;
	protected String versionColumnName;

	/**
	 * KEY：カラム名、VALUE：SQLタイプ
	 */
	protected Map columnTypeMap = new HashMap();

	// List<String>
	protected Set excludeColumns = new HashSet();

	public BeanQueryFactoryBuilder(Class beanClass) {
		this(beanClass, DEFAULT_NAME_MAPPING);
	}

	public BeanQueryFactoryBuilder(Class beanClass, NameMapping nameMapping) {
		this.beanClass = beanClass;
		this.nameMapping = nameMapping;
		this.tableName = OthersUtils.getShortClassName(beanClass);
	}

	public BeanQueryFactoryBuilder excludeColumn(String excludeColumn) {
		excludeColumns.add(excludeColumn.toLowerCase());
		return this;
	}

	public BeanQueryFactoryBuilder excludeColumns(String[] excludeColumns) {
		for (int i = 0; i < excludeColumns.length; i++)
			excludeColumn(excludeColumns[i]);
		return this;
	}

	public BeanQueryFactoryBuilder excludeColumns(List excludeColumns) {
		for (int i = 0; i < excludeColumns.size(); i++)
			excludeColumn((String) excludeColumns.get(i));
		return this;
	}

	public BeanQueryFactoryBuilder tableName(String tableName) {
		this.tableName = tableName;
		return this;
	}

	public BeanQueryFactoryBuilder versionColumnName(String versionColumnName) {
		this.versionColumnName = versionColumnName;
		return this;
	}

	public BeanQueryFactoryBuilder columnType(String columnName, int sqlType) {
		columnTypeMap.put(columnName.toLowerCase(), new Integer(sqlType));
		return this;
	}

	public BeanQueryFactoryBuilder updateKeyColumn(String columnNames) {
		if (this.updateKeyColumnSet == null) {
			this.updateKeyColumnSet = new LinkedHashSet();
		}
		this.updateKeyColumnSet.add(columnNames.toLowerCase());
		return this;
	}

	private boolean isUpdateKey(String columnName) {
		if (this.updateKeyColumnSet == null)
			return false;
		return this.updateKeyColumnSet.contains(columnName.toLowerCase());
	}

	public QueryFactory buildUpdate() {
		BeanColumn[] beanColumns = this.toBeanColumns();

		StringBuffer columnBuffer = new StringBuffer(beanColumns.length * 7);

		List updateColumnProperties = new ArrayList();
		List whereColumnProperties = new ArrayList();

		columnBuffer.append("update ").append(this.tableName).append(" set ");
		boolean flg = false;
		for (int i = 0; i < beanColumns.length; i++) {

			BeanColumn beanColumn = beanColumns[i];
			if (isUpdateKey(beanColumn.getColumnName())) {
				whereColumnProperties.add(beanColumn);
				continue;
			}
			if (flg) {
				columnBuffer.append(',');
			} else {
				flg = true;
			}
			columnBuffer.append(beanColumn.getColumnName()).append("=?");
			if (versionColumnName != null
					&& versionColumnName.equalsIgnoreCase(beanColumn
							.getColumnName())) {
				whereColumnProperties.add(beanColumn);
				updateColumnProperties
						.add(new VersionColumnProperty(beanColumn));
			} else {
				updateColumnProperties.add(beanColumn);
			}
		}
		StringBuffer whereBuffer = null;
		if (!whereColumnProperties.isEmpty()) {
			for (Iterator iterator = whereColumnProperties.iterator(); iterator
					.hasNext();) {
				if (whereBuffer == null)
					whereBuffer = new StringBuffer(" where ");
				else
					whereBuffer.append(" and ");
				ColumnProperty columnProperty = (ColumnProperty) iterator
						.next();
				whereBuffer.append(columnProperty.getColumnName()).append("=?");
			}
		}

		List list = new ArrayList(updateColumnProperties);
		list.addAll(whereColumnProperties);
		if (whereBuffer != null)
			columnBuffer.append(whereBuffer);
		String sql = columnBuffer.toString();
		return new BeanQueryFactory(sql, (BeanColumn[]) list
				.toArray(new BeanColumn[list.size()]));
	}

	public QueryFactory buildInsert() {
		BeanColumn[] columnProperties = this.toBeanColumns();
		StringBuffer columnBuffer = new StringBuffer(
				columnProperties.length * 7);
		StringBuffer valueBuffer = new StringBuffer(columnProperties.length * 2);

		columnBuffer.append("insert into ").append(this.tableName).append("(");
		valueBuffer.append(" values ( ");
		boolean flg = false;
		for (int i = 0; i < columnProperties.length; i++) {

			if (flg) {
				columnBuffer.append(',');
				valueBuffer.append(',');
			} else {
				flg = true;
			}
			BeanColumn columnProperty = columnProperties[i];
			columnBuffer.append(columnProperty.getColumnName());
			valueBuffer.append('?');
		}
		columnBuffer.append(" ) ");
		valueBuffer.append(" ) ");
		String sql = columnBuffer.append(valueBuffer).toString();

		return new BeanQueryFactory(sql, columnProperties);

	}

	protected BeanColumn[] toBeanColumns() {

		PropertyDescriptor[] propertyDescriptors = ReflectionUtil
				.propertyDescriptors(beanClass);

		List columnProperties = new ArrayList();
		for (int i = 0; i < propertyDescriptors.length; i++) {
			PropertyDescriptor propertyDescriptor = propertyDescriptors[i];

			Method getter = propertyDescriptor.getReadMethod();
			if (getter == null || propertyDescriptor.getWriteMethod() == null
					|| "class".equals(propertyDescriptor.getName()))
				continue;

			String columnName = this.nameMapping
					.toColumnName(propertyDescriptor);
			if (excludeColumns.contains(columnName.toLowerCase()))
				continue;
			// ValueType取得
			Integer sqlType = (Integer) columnTypeMap.get(columnName
					.toLowerCase());

			columnProperties.add(new ColumnProperty(columnName, getter,
					(sqlType == null ? TypeMappings
							.getValueType(propertyDescriptor.getPropertyType())
							: TypeMappings.getValueType(propertyDescriptor
									.getPropertyType(), sqlType.intValue()))));
		}
		return (BeanColumn[]) columnProperties
				.toArray(new BeanColumn[columnProperties.size()]);
	}

	/**
	 * @deprecated
	 * @return
	 */
	protected ColumnProperty[] toColumnProperties() {

		PropertyDescriptor[] propertyDescriptors = ReflectionUtil
				.propertyDescriptors(beanClass);

		List columnProperties = new ArrayList();
		for (int i = 0; i < propertyDescriptors.length; i++) {
			PropertyDescriptor propertyDescriptor = propertyDescriptors[i];

			Method getter = propertyDescriptor.getReadMethod();
			if (getter == null || propertyDescriptor.getWriteMethod() == null
					|| "class".equals(propertyDescriptor.getName()))
				continue;

			String columnName = this.nameMapping
					.toColumnName(propertyDescriptor);
			if (excludeColumns.contains(columnName.toLowerCase()))
				continue;
			// ValueType取得
			Integer sqlType = (Integer) columnTypeMap.get(columnName
					.toLowerCase());

			columnProperties.add(new ColumnProperty(columnName, getter,
					(sqlType == null ? TypeMappings
							.getValueType(propertyDescriptor.getPropertyType())
							: TypeMappings.getValueType(propertyDescriptor
									.getPropertyType(), sqlType.intValue()))));
		}
		return (ColumnProperty[]) columnProperties
				.toArray(new ColumnProperty[columnProperties.size()]);
	}

}
