<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_ConfigActions.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * アクションクラス
 */
require_once SYL_FRAMEWORK_DIR . '/core/SyL_Action.php';

/**
 * アクション設定ファイルを読み込むクラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_ConfigActions.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_ConfigActions extends SyL_Config
{
    /**
     * 設定ファイル名
     * 
     * @access private
     * @var string
     */
    var $config_file_name = 'actions.xml';
    /**
     * イベント名
     * 
     * @access protected
     * @var string
     */
    var $event = 'executeStream';
    /**
     * デフォルト実行順序
     * 
     * @access protected
     * @param int
     */
    var $default_priority = 5;
    /**
     * デフォルトメソッド名
     * 
     * @access private
     * @param string
     */
    var $default_method = 'execute';
    /**
     * pre実行メソッド名
     * 
     * @access private
     * @param string
     */
    var $pre_method = 'preExecute';
    /**
     * 検証実行メソッド名
     * 
     * @access private
     * @param string
     */
    var $validate_method = 'validate';
    /**
     * post実行メソッド名
     * 
     * @access private
     * @param string
     */
    var $post_method = 'postExecute';
    /**
     * コンテナアクション名
     * 
     * @access private
     * @var object
     */
    var $action_key = SYL_ACTION_KEY;

    /**
     * 設定ファイルをセット
     *
     * @access protected
     */
    function setProjectConfigFiles()
    {
        $this->file_names[] = SYL_APP_CONFIG_DIR . "/{$this->config_file_name}";
    }

    /**
     * XMLファイルの解析処理
     *
     * @access public
     * @param bool キャッシュ有効フラグ
     */
    function parseXml($enable_cache=true)
    {
        // アクションベースクラスロード
        SyL_Loader::userLib(SYL_ACTION_BASE_CLASS);

        if (SYL_CACHE && $enable_cache) {
            // キャッシュを使用する
            $key = SYL_APP_DIR . $this->router->getActionFile() . $this->router->getActionName();
            $cache =& $this->getCache($key);
            if ($cache) {
                // キャッシュが取得できなかった場合（キャッシュオブジェクトを取得した場合）
                parent::parseXml();
                // アクション実行順序変更
                $this->changeActionOrder();
                // アクションイベントメソッド定義
                $this->setEvents();
                // アクション設定をキャッシュ化
                $cache->write($this->config);
            }
        } else {
            // キャッシュを使用しない
            parent::parseXml();
            // アクション実行順序変更
            $this->changeActionOrder();
            // アクションイベントメソッド定義
            $this->setEvents();
        }

        if (!isset($this->config[$this->action_key])) {
            // 対応するアクションマッピングが存在しない場合
            trigger_error("[SyL error] Action mapping not found (" . $this->router->getActionFile() . ")", E_USER_ERROR);
        }
    }

    /**
     * XMLデータ内容取得ハンドラメソッド
     * ※オーバーライド
     *
     * @access protected
     * @param string 現在XML解析中のパス
     * @param string XMLのタグの属性値
     * @param string XMLのタグの値
     */
    function getElement($current_path, $attribute, $text)
    {
        static $name             = '';
        static $pre_name         = '';
        static $post_name        = '';

        static $method_name      = '';
        static $pre_method_name  = '';
        static $post_method_name = '';

        static $component_name   = '';
        static $load             = false;

//echo $current_path . "<br>";
        // 1アクションロード済みの場合以降スキップ
        if ($load) return;

        // XML解析結果を取得
        switch ($current_path) {
        case '/syl-actions/action':
            if ($name != '') {
                $name = '';
                $load = true;
                return;
            }
            if (isset($attribute['enable']) && ($attribute['enable'] === 'false')) {
                return;
            }

            // アクセスURL確認
            $reg = isset($attribute['path']) ? str_replace('/', '\\/', $attribute['path']) : '(.*)';
            if (preg_match('/^' . $reg . '$/', '/' . $this->router->getActionFile(), $matches)) {
                $name  = $this->action_key;
                // デフォルトアクション
                $class  = $this->router->getClassName();
                $file   = $this->router->getActionDir() . $this->router->getActionFile();
                // デフォルト存在判定
                if (!is_file($file)) {
                    $class = SyL_Loader::convertClass(SYL_ACTION_BASE_CLASS);
                    $file  = SYL_APP_LIB_DIR . SyL_Loader::convertPath(SYL_ACTION_BASE_CLASS);
                }
                // コンポーネントパラメータ
                $this->config[$name] = array(
                    'class'       => $class,
                    'type'        => 'action',
                    'file'        => $file,
                    'reference'   => true,
                    'priority'    => $this->default_priority, // 実行順序
                    'constructor' => false, // コンストラクタインジェクションフラグ
                    'force'       => (SYL_ENV_TYPE == 'cmd'), // ファイルが存在しないとエラーのフラグ コマンドラインのみファイル存在チェック
                    'args'        => array() // 引数初期化
                );

                // イベントタグ（オブジェクト作成時）
                $this->config[$name]['event'][$this->event][] = $name;

                $method_name = '';
                if (isset($attribute['method'])) {
                    $method_name = 'method:' . str_replace('{action}', $this->router->getActionName(), $attribute['method']);
                } else {
                    // アクションメソッドのデフォルト
                    $method_name = 'method:' . $this->default_method;
                }
                // アクションメソッドの引数は固定
                $this->config[$name]['args'][$method_name] = array();
                $this->config[$name]['args'][$method_name][] = 'component:true:data';
                $this->config[$name]['args'][$method_name][] = 'component:true:context';

                $this->config[$name]['event'][$this->event][] = $method_name; // イベントタグ
            }
            break;

        default:
            if (!$name) return;

            switch ($current_path) {
            case '/syl-actions/action/setter':
                // ダイレクトセットタグ
                if (isset($attribute['direct']) && ($attribute['direct'] === 'true')) {
                    $method_name = 'setter:' . $attribute['name'] . ':true';
                } else {
                    $method_name = 'setter:set' . ucfirst($attribute['name']) . ':false';
                }
                $this->config[$name]['event'][$this->event][] = $method_name;
                $this->config[$name]['args'][$method_name]    = array();
                break;

            case '/syl-actions/action/method':
                $method_name = 'method:' . $attribute['name'];
                $this->config[$name]['event'][$this->event][] = $method_name; // イベントタグ
                $this->config[$name]['args'][$method_name] = array();
                break;

            case '/syl-actions/action/setter/arg':
            case '/syl-actions/action/method/arg':
                $type = isset($attribute['type']) ? $attribute['type'] : 'value';
                $ref  = (isset($attribute['reference']) && ($attribute['reference'] === 'true')) ? 'true' : 'false';
                $arg  = "{$type}:{$ref}:{$text}";
                $this->config[$name]['args'][$method_name][] = $arg;
                break;

           case '/syl-actions/action/components/component':
                if (isset($attribute['enable']) && ($attribute['enable'] === 'false')) {
                    $component_name = '';
                    return;
                }
                $component_name = $attribute['name'];
                $this->config[$component_name]['type']      = 'component';
                $this->config[$component_name]['class']     = isset($attribute['class'])  ? $attribute['class'] : $component_name;
                $this->config[$component_name]['file']      = $attribute['file'];
                $this->config[$component_name]['reference'] = (isset($attribute['reference']) && ($attribute['reference'] === 'true'));
                // 実行順序(アクションより2段階早い)
                $default_priority = $this->default_priority - 2;
                $component_priority = isset($attribute['priority']) ? floatval($default_priority . '.' . $attribute['priority']) : $default_priority;
                $this->config[$component_name]['priority'] = $component_priority;
                // コンストラクタインジェクションフラグ
                $this->config[$component_name]['constructor'] = false;
                // ファイルが存在しないとエラーのフラグ
                $this->config[$component_name]['force'] = false;
                // イベントタグ
                $this->config[$component_name]['event'][$this->event][] = $component_name;
                // 引数
                $this->config[$component_name]['args'] = array();
                break;

            default:
                if (!$component_name) return;

                switch ($current_path) {
                case '/syl-actions/action/components/component/constructor':
                    $method_name = 'constructor:' . (isset($attribute['static']) ? $attribute['static'] : '');
                    // コンストラクタインジェクションフラグ
                    $this->config[$component_name]['constructor'] = true;
                    // コンストラクタインジェクションの場合、元のメソッドは削除
                    $index = array_search($component_name, $this->config[$component_name]['event'][$this->event]);
                    if ($index !== false) {
                        unset($this->config[$component_name]['event'][$this->event][$index]);
                    }
                    $this->config[$component_name]['event'][$this->event][] = $method_name;
                    $this->config[$component_name]['args'][$method_name] = array();
                    break;

                case '/syl-actions/action/components/component/setter':
                    // ダイレクトセットタグ
                    if (isset($attribute['direct']) && ($attribute['direct'] === 'true')) {
                        $method_name = 'setter:' . $attribute['name'] . ':true';
                    } else {
                        $method_name = 'setter:set' . ucfirst(strtolower($attribute['name'])) . ':false';
                    }
                    $this->config[$component_name]['event'][$this->event][] = $method_name;
                    $this->config[$component_name]['args'][$method_name] = array();
                    break;

                case '/syl-actions/action/components/component/method':
                    $method_name = 'method:' . $attribute['name'];
                    $this->config[$component_name]['event'][$this->event][] = $method_name;
                    $this->config[$component_name]['args'][$method_name] = array();
                    break;

                case '/syl-actions/action/components/component/setter/arg':
                case '/syl-actions/action/components/component/constructor/arg':
                case '/syl-actions/action/components/component/method/arg':
                    $type = isset($attribute['type']) ? $attribute['type'] : 'value';
                    $ref  = (isset($attribute['reference']) && ($attribute['reference'] === 'true')) ? 'true' : 'false';
                    $this->config[$component_name]['args'][$method_name][] = $type . ':' . $ref . ':' . $text;
                    break;
                }
            }
        }
    }

    /**
     * アクション実行メソッドの順番を変更する
     * ※主にコンポーネントのインジェクションがある場合
     *
     * @access private
     */
    function changeActionOrder()
    {
        $name = $this->action_key;

        // インジェクションが無い場合はスルー
        if (count($this->config[$name]['event'][$this->event]) <= 2) {
            return;
        }

        $method = array_splice($this->config[$name]['event'][$this->event], 1, 1);
        array_push($this->config[$name]['event'][$this->event], $method[0]);

        // 連想配列の順序変更のため先頭の値を取得
        $main_args = array();
        foreach ($this->config[$name]['args'] as $key => $arg) {
            $main_args[$key] = $arg;
            break;
        }

        array_shift($this->config[$name]['args']);
        foreach ($main_args as $key => $arg) {
            $this->config[$name]['args'][$key] = $arg;
        } 
    }


    /**
     * アクションイベントメソッドを定義する
     *
     * @access private
     */
    function setEvents()
    {
        $name = $this->action_key;

        // メソッド引数
        $args = array();
        $args[0] = 'component:true:data';
        $args[1] = 'component:true:context';

        // 連想配列の順序変更のため一時保存
        $main_args = $this->config[$name]['args'];

        // 初期化
        $this->config[$name]['args'] = array();

        // preExecute 定義
        $method = 'method:' . $this->pre_method;
        array_splice($this->config[$name]['event'][$this->event], 1, 0, $method);
        $this->config[$name]['args'][$method] = $args;

        // validate 定義
        $method = 'method:' . $this->validate_method;
        array_splice($this->config[$name]['event'][$this->event], 2, 0, $method);
        $this->config[$name]['args'][$method] = $args;

        // execute 定義
        $this->config[$name]['args'] += $main_args;

        // postExecute 定義
        $method = 'method:' . $this->post_method;
        array_push($this->config[$name]['event'][$this->event], $method);
        $this->config[$name]['args'][$method] = $args;
    }
}
