<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_Error.php,v 1.1 2009/01/11 05:34:30 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * エラークラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_Error.php,v 1.1 2009/01/11 05:34:30 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_Error
{
    /**
     * エラーメッセージ配列
     * 
     * @access public
     * @var array
     */
    var $error_messages = array();
    /**
     * 改行コード
     * 
     * @access public
     * @var string
     */
    var $eol = "\r\n";
    /**
     * コントローラ
     * 
     * @access protected
     * @var object
     */
    var $controller = null;

    /**
     * コンストラクタ
     *
     * @access public
     * @param object コントローラ
     */
    function SyL_Error()
    {
    }

    /**
     * SyL_Errorクラスのインスタンス取得
     *
     * singletonモデルを採用しているので、
     * 常に1つのインスタンスの参照が取得される。
     *
     * ※取得方法
     *   $SyL_Error =& SyL_Error::getInstance();
     *
     * @access public
     * @param object コントローラ or データ管理オブジェクト
     */
    function &getInstance()
    {
        static $singleton;
        if (!is_object($singleton)) {
            $classname = 'SyL_Error' . ucfirst(SYL_ENV_TYPE);
            include_once SYL_FRAMEWORK_DIR . "/core/Error/{$classname}.php";
            // 子クラスのインスタンス生成
            $singleton = new $classname();
        }
        return $singleton;
    }

    /**
     * コントローラをセット
     *
     * @access public
     * @param object コントローラ
     */
    function setController(&$controller)
    {
        $this->controller =& $controller;
    }

    /**
     * エラーハンドラを設定
     * 
     * @static
     * @access public
     */
    function setErrorHandler()
    {
        $error =& SyL_Error::getInstance();
        if ($error->controller->error_handler_method) {
            if (version_compare(PHP_VERSION, '5.0.0', '>=')) {
                set_error_handler(array(&$error->controller, $error->controller->error_handler_method), error_reporting());
                if ($error->controller->exception_handler_method) {
                    set_exception_handler(array(&$error->controller, $error->controller->exception_handler_method));
                }
            } else {
                set_error_handler(array(&$error->controller, $error->controller->error_handler_method));
            }
        }
    }

    /**
     * エラーハンドラを初期化
     * 
     * @static
     * @access public
     */
    function resetErrorHandler()
    {
        restore_error_handler();
    }

    /**
     * エラーメッセージをセット
     * 
     * @static
     * @access public
     * @param string エラーメッセージ
     */
    function setErrorMessage($error_message)
    {
        $error =& SyL_Error::getInstance();
        $error->error_messages[] = $error_message;
    }

    /**
     * 全エラーメッセージを取得
     * 
     * @access protected
     */
    function getErrorMessages()
    {
        $error_message = $this->controller->getErrorMessage();
        if ($error_message) {
            $error_message = '[' . $this->getErrorType() . '] ' . $error_message;
            return array_merge($this->error_messages, (array)$error_message);
        } else {
            return $this->error_messages;
        }
    }

    /**
     * 処理を中断し、共通エラー画面を表示
     * 
     * @static
     * @access public
     * @param string エラーメッセージ
     * @param int エラーレベル
     */
    function trigger($error_string='', $error_type=E_USER_ERROR)
    {
        trigger_error($error_string, $error_type);
    }

    /**
     * 404 Not Found を表示する
     *
     * @static
     * @access public
     */
    function forwardFileNotFound()
    {
        // バッファリングクリア
        if (!headers_sent() && (ob_get_level() > 0)) {
            ob_clean();
        }

        $error =& SyL_Error::getInstance();
        $error->forwardFileNotFoundDetail();
        exit;
    }

    /**
     * 404 Not Found の表示内容
     *
     * @access abstract
     */
    function forwardFileNotFoundDetail()
    {
    }

    /**
     * エラーハンドラーの実体
     *
     * ※テンプレートメソッドパターンのテンプレート部
     * 
     * @access public
     */
    function errorHandler()
    {
        // バッファリングクリア
        if (!headers_sent() && (ob_get_level() > 0)) {
            ob_clean();
        }

        // エラーメッセージ取得
        $error_messages = SYL_ERROR ? $this->getErrorMessages() : array(SYL_ERROR_MESSAGE);
        // エラートレースの取得
        $error_trace = SYL_ERROR ? $this->getTrace() : array();

        // サブクラスへ
        $this->displayError($error_messages, $error_trace);

        // ロギング
        $this->log();
        // エラーメール送信
        $this->sendMail();
    }

    /**
     * 環境別エラー処理
     * 
     * @access public
     * @param array エラーメッセージ配列
     * @param array エラートレース
     */
    function displayError($error_messages, $error_trace)
    {
    }

    /**
     * エラータイプを返す
     * ※ただし、実際にプログラムでハンドリングできるのは
     *   E_WARNING
     *   E_NOTICE
     *   E_USER_ERROR
     *   E_USER_WARNING
     *   E_USER_NOTICE
     * のみ
     *
     * @access protected
     * @param  int    エラーNo
     * @return string エラータイプ名
     */
    function getErrorType()
    {
        switch ($this->controller->getErrorNo()) {
        case E_ERROR           : return 'Error';
        case E_WARNING         : return 'Warning';
        case E_PARSE           : return 'Parsing Error';
        case E_NOTICE          : return 'Notice';
        case E_CORE_ERROR      : return 'Core Error';
        case E_CORE_WARNING    : return 'Core Warning';
        case E_COMPILE_ERROR   : return 'Compile Error';
        case E_COMPILE_WARNING : return 'Compile Warning';
        case E_USER_ERROR      : return 'User Error';
        case E_USER_WARNING    : return 'User Warning';
        case E_USER_NOTICE     : return 'User Notice';
        default:
            if (defined('E_STRICT') && ($errorNo == E_STRICT)) {
                return 'Strict';
            }
            if (defined('E_RECOVERABLE_ERROR') && ($errorNo == E_RECOVERABLE_ERROR)) {
                return 'Recoverable Error';
            }
        }

        return 'unknonwn';
    }

    /**
     * トレース情報を取得
     *
     * @access private
     * @return array トレース情報
     */
    function getTrace()
    {
        // エラートレース取得
        $error_trace = array();
        $i = 1;
        foreach (array_reverse($this->controller->getErrorTrace()) as $values) {
            $tmp = array();
            $tmp['no']   = $i++;
            $tmp['file'] = isset($values['file']) ? $values['file'] : '-';
            $tmp['line'] = isset($values['line']) ? $values['line'] : '-';
            if (isset($values['class']) && isset($values['type'])) {
                $tmp['function'] = $values['class'] . $values['type'] . $values['function'];
            } else {
                $tmp['function'] = (isset($values['function']) && ($values['function'] != 'unknown')) ? $values['function'] : '-';
            }
            $error_trace[] = $tmp;

            // ユーザー定義エラー時は、「trigger_error」関数以降を排除
            // その他のエラー時は、「SyL_Error::trigger」メソッド以降を排除
            if (isset($values['class']) && (strtolower($values['class']) == 'syl_error')) {
                break;
            } else if (isset($values['function']) && ($values['function'] == 'trigger_error')) {
                break;
            }
        }

        return array_reverse($error_trace);
    }

    /**
     * エラートレース付きエラーメッセージを取得する
     * ※ロギング、メール用
     *
     * @access prvate
     * @return string エラートレース付きエラーメッセージ
     */
    function getErrorMessageTrace()
    {
        $error_message = implode($this->eol, $this->getErrorMessages()) . $this->eol;
        foreach ($this->getTrace() as $values) {
            $error_message .= "[file] "     . $values['file']     . " ";
            $error_message .= "[line] "     . $values['line']     . " ";
            $error_message .= "[function] " . $values['function'] . $this->eol;
        }
        return $error_message;
    }

    /**
     * エラーファイルを取得する
     *
     * @access protected
     * @return string エラーファイル
     */
    function getErrorFile()
    {
        return $this->controller->getErrorFile();
    }

    /**
     * エラー行数を取得する
     *
     * @access protected
     * @return int エラー行数
     */
    function getErrorLine()
    {
        return $this->controller->getErrorLine();
    }

    /**
     * ログを保存
     *
     * @access public
     */
    function log()
    {
        // トレース付きエラーメッセージ取得
        $error_message  = $this->getErrorMessageTrace();
        $error_message .= $this->extraLog();

        // エラーレベル毎にログ保存
        switch ($this->controller->getErrorNo()) {
        case E_ERROR:
        case E_USER_ERROR:
            SyL_Loggers::error($error_message);
            break;
        case E_WARNING:
        case E_USER_WARNING:
            SyL_Loggers::warn($error_message);
            break;
        case E_NOTICE:
        case E_USER_NOTICE:
            SyL_Loggers::notice($error_message);
            break;
        }
    }

    /**
     * ログ保存その他情報
     *
     * @access protected
     * @return string その他情報
     */
    function extraLog()
    {
        return '';
    }

    /**
     * エラーメールを送信
     *
     * @access public
     */
    function sendMail()
    {
        if (($this->controller->getErrorNo() <= SYL_LOG) && SYL_ERROR_MAIL && SYL_MAIL_ADMIN_ADDRESS) {
            // 日付取得
            $date = date('Y-m-d H:i:s');
            // サーバー名取得
            $server = isset($_SERVER['HTTP_HOST']) ? $_SERVER['HTTP_HOST'] : '';
            if ($server == '') {
                $server = isset($_SERVER['SERVER_NAME']) ? $_SERVER['SERVER_NAME'] : '';
            }

            // トレース付きエラーメッセージ取得
            $error_message  = $this->getErrorMessageTrace();
            // SyLのバージョン
            $version     = SYL_VERSION;
            $version_php = PHP_VERSION;
            $uname       = php_uname();

            $error_mail_subject = "SyL Framework Error Report - {$server} - {$date}";
            $error_mail_body = <<< EOF
SyL Framework Error info. {$date}
----------------------------------------

{$error_message}

----------------------------------------
SyL Framework ver. {$version} on PHP {$version_php}
{$uname}
EOF;
            // 送信前ログ
            SyL_Loggers::info("Error mail Type: "    . SYL_MAIL_TYPE);
            SyL_Loggers::info("Error mail Subject: " . $error_mail_subject);
            SyL_Loggers::info("Error mail From: "    . SYL_MAIL_ADMIN_ADDRESS);
            SyL_Loggers::info("Error mail To: "      . SYL_MAIL_ADMIN_ADDRESS);

            // エラーメール送信実行
            include_once SYL_FRAMEWORK_DIR . '/lib/SyL_Mail.php';
            $mail =& SyL_Mail::factorySend(SYL_MAIL_TYPE);
            $message =& $mail->createMessage();
            $message->setSubject($error_mail_subject);
            $message->setFrom(SYL_MAIL_ADMIN_ADDRESS);
            $message->addTo(SYL_MAIL_ADMIN_ADDRESS);
            $message->setBody($error_mail_body);
            if (!$mail->send($message)) {
                SyL_Loggers::error($mail->getErrorMessage());
            }
        }
    }
}
