<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_UtilProgress.php,v 1.1 2009/01/11 05:34:30 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * リクエストクラス
 */
require_once SYL_FRAMEWORK_DIR . '/core/SyL_Request.php';

/**
 * プログレスバー用ユーティリティクラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_UtilProgress.php,v 1.1 2009/01/11 05:34:30 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_UtilProgress
{
    /**
     * プログレス名
     *
     * @access private
     * @var string
     */
    var $name = '';
    /**
     * 経過値接頭辞
     *
     * @access private
     * @var string
     */
    var $prefix = '__syl_ps_';
    /**
     * MAX値接頭辞
     *
     * @access private
     * @var string
     */
    var $prefix_max = '__syl_ps_max_';
    /**
     * セッションオブジェクト
     *
     * @access private
     * @var object
     */
    var $session = null;

    /**
     * コンストラクタ
     *
     * @access public
     */
    function SyL_UtilProgress()
    {
        $request =& SyL_Request::singleton();
        $this->session =& $request->getSessionObject();
        $this->session->closeSession();
    }

    /**
     * プログレス名をセット
     * 
     * @access public
     * @param string プログレス名
     */
    function setProgressName($name)
    {
        $this->name = $name;
    }

    /**
     * 経過値をセット
     * 
     * @access public
     * @param int 経過値
     */
    function setProgressValue($value)
    {
        if (!is_numeric($value) || ($value < 0)) {
            trigger_error("[SyL error] Progress parameter is not integer ($value)", E_USER_ERROR);
        }
        // 接頭辞追加
        $name = $this->prefix . $this->name;
        $this->session->writeCloseSession(array($name => $value));
    }

    /**
     * 最大値をセット
     * 
     * @access public
     * @param int 最大値
     */
    function setProgressMaxValue($value)
    {
        if (!is_numeric($value) || ($value < 0)) {
            trigger_error("[SyL error] Progress max parameter is not integer ($value)", E_USER_ERROR);
        }
        // 接頭辞追加
        $name = $this->prefix_max . $this->name;
        $this->session->writeCloseSession(array($name => $value));
    }

    /**
     * 経過値を%でを取得
     * 
     * @access public
     * @param string 経過値
     */
    function getProgressValue()
    {
        $name     = $this->prefix . $this->name;
        $name_max = $this->prefix_max . $this->name;
        $this->session->startSession();
        $value = $this->session->get($name);
        $value_max = $this->session->get($name_max);
        $this->session->closeSession();
        
        if (($value === null) || ($value_max === null)) {
            // パラメータが存在しない場合
            return 0;
        }
        if (($value == 100) || ($value_max == 0)) {
            // MAX値が0 or valueが100の場合
            return 100;
        }

        return ceil(($value/$value_max)*100);
    }

    /**
     * パラメータをセット
     * 
     * @access public
     * @param string パラメータ名
     * @param string パラメータ値
     */
    function setParameter($name, $value)
    {
        $name = $this->prefix . $this->name . $name;
        $this->session->writeCloseSession(array($name => $value));
    }

    /**
     * パラメータを取得
     * 
     * @access public
     * @param string パラメータ名
     * @return string パラメータ値
     */
    function getParameter($name)
    {
        $name = $this->prefix . $this->name . $name;
        $this->session->startSession();
        $value = $this->session->get($name);
        $this->session->closeSession();

        return $value;
    }

    /**
     * パラメータ削除処理
     * 
     * @access public
     */
    function finish()
    {
        $name = $this->prefix . $this->name;
        $name_max = $this->prefix_max . $this->name;

        $this->session->startSession();
        $this->session->delete($name);
        $this->session->delete($name_max);
        $this->session->closeSession();
    }
}
