<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_CacheDb.php,v 1.1 2009/01/11 05:34:31 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * DB接続文字列設定
 */
if (!defined('SYL_DB_DSN_CACHE')) {
    if (!defined('SYL_DB_DSN')) {
        trigger_error("[SyL error] Undefined constant `SYL_DB_DSN' or `SYL_DB_DSN_CACHE'", E_USER_ERROR);
    }
    define('SYL_DB_DSN_CACHE', SYL_DB_DSN);
}

/**
 * DB操作クラス
 */
require_once dirname(__FILE__) . '/../SyL_DB.php';

/**
 * DBキャッシュクラス
 *
 * 例）
 * CREATE TABLE SYL_CACHE ( 
 *   cache_id      varchar(50) NOT NULL, 
 *   cache_expires int NOT NULL, 
 *   cache_data    text,
 *   PRIMARY KEY  (cache_id)
 * );
 * のようなテーブルを想定
 *
 * SYL_CACHE_DB_TABLE = 'SYL_CACHE:cache_id:cache_data:cache_expires'
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_CacheDb.php,v 1.1 2009/01/11 05:34:31 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_CacheDb extends SyL_Cache
{
    /**
     * DBコネクション
     *
     * @access private
     * @var object
     */
    var $conn = null; 
    /**
     * セッションテーブル名
     *
     * @access private
     * @var string
     */
    var $table;
    /**
     * セッションIDカラム名
     *
     * @access private
     * @var string
     */
    var $id;
    /**
     * セッションデータカラム名
     *
     * @access private
     * @var string
     */
    var $data;
    /**
     * セッションタイムスタンプカラム名
     *
     * @access private
     * @var string
     */
    var $timestamp;

    /**
     * コンストラクタ
     *
     * @access public
     */
    function SyL_CacheDb()
    {
        parent::SyL_Cache();

        if (!defined('SYL_CACHE_DB_TABLE')) {
            trigger_error("[SyL error] Undefined constant `SYL_CACHE_DB_TABLE'", E_USER_ERROR);
        }
        // テーブルプロパティ取得
        list($this->table, $this->id, $this->data, $this->timestamp) = explode(':', SYL_CACHE_DB_TABLE);
        // DB接続取得
        $this->conn =& SyL_DB::getConnection(SYL_DB_DSN_CACHE);
        // カラムの調整
        switch ($this->conn->field_case) {
        case CASE_UPPER:
            $this->id        = strtoupper($this->id);
            $this->data      = strtoupper($this->data);
            $this->timestamp = strtoupper($this->timestamp);
            break;
        case CASE_LOWER:
            $this->id        = strtolower($this->id);
            $this->data      = strtolower($this->data);
            $this->timestamp = strtolower($this->timestamp);
            break;
        }
    }

    /**
     * DBとの接続断
     *
     * @access public
     */
    function close()
    {
        if ($this->conn != null) {
            $this->conn->close();
            $this->conn = null;
        }
    }

    /**
     * キャッシュの更新時間を更新する
     *
     * @access public
     * @param int 更新時間(UNIX Time)
     * @return bool true: 更新OK、false: 更新エラー
     */
    function setModifyTime($mtime=0)
    {
        if (!$mtime) {
            $mtime = time();
        }

        $sql  = "";
        $sql .= "UPDATE ";
        $sql .=   "{$this->table} ";
        $sql .= "SET ";
        $sql .=   "{$this->timestamp} = {$mtime} ";
        $sql .= "WHERE ";
        $sql .=   "{$this->id} = '" . $this->conn->escape($this->getKey()) . "' ";
        return $this->conn->exec($sql);
    }

    /**
     * キャッシュファイルの更新時間を取得する
     *
     * @access public
     * @return int 更新時間(UNIX Time)
     */
    function getModifyTime()
    {
        $sql  = "";
        $sql .= "SELECT ";
        $sql .=   "{$this->timestamp} ";
        $sql .= "FROM ";
        $sql .=   "{$this->table} ";
        $sql .= "WHERE ";
        $sql .=   "{$this->id} = '" . $this->conn->escape($this->getKey()) . "' ";

        $this->conn->queryRef($sql, $result, 'one');
        return $result;
    }

    /**
     * キャッシュを読み込む
     *
     * @access public
     * @param bool キャッシュ有効期間を更新
     * @return mixed キャッシュデータ
     */
    function read($modify_time_update=false)
    {
        $sql  = "";
        $sql .= "SELECT ";
        $sql .=   "{$this->data}, ";
        $sql .=   "{$this->timestamp} ";
        $sql .= "FROM ";
        $sql .=   "{$this->table} ";
        $sql .= "WHERE ";
        $sql .=   "{$this->id} = '" . $this->conn->escape($this->getKey()) . "' ";

        $this->conn->queryRef($sql, $result, 'record');

        if ($result) {
            // キャッシュファイルOK
            $data  = $result[$this->data];
            $mtime = $result[$this->timestamp];

            // 更新時間の最小値判定
            if ($this->min_mtime > 0) {
                if ($mtime < $this->min_mtime) {
                    $this->delete();
                    return false;
                }
            }
            // 更新時間の最大値判定
            if ($this->max_mtime > 0) {
                if ($mtime > $this->max_mtime) {
                    $this->delete();
                    return false;
                }
            }

            // 有効時間を取得
            $life_time = ($this->life_time > 0) ? $mtime + $this->life_time : null;

            if (($life_time === null) || ($life_time >= time())) {
                $hash = null;
                if ($this->isCrc()) {
                    $hash = substr($data, 0, 32);
                    $data = substr($data, 32);
                }

                if (($hash === null) || ($hash == $this->getCrc($data))) {
                    if ($modify_time_update) {
                        $this->setModifyTime();
                    }
                    return $this->isSerialize() ? unserialize($data) : $data;
                } else {
                    $this->delete();
                    return false;
                }
            } else {
                // キャッシュ有効期間切れ
                $this->delete();
                return false;
            }
        } else {
            // キャッシュファイル無し
            return false;
        }
    }

    /**
     * キャッシュを保存する
     *
     * @access public
     * @param mixed キャッシュデータ
     * @return bool true: 保存成功、false: 保存エラー
     */
    function write($data)
    {
        // データのシリアル化判定
        if ($this->isSerialize()) {
            $data = serialize($data);
        }
        // CRC追加判定
        if ($this->isCrc()) {
            $data = $this->getCrc($data) . $data;
        }

        $sql  = "";
        $sql .= "INSERT INTO {$this->table} (";
        $sql .=   "{$this->id}, ";
        $sql .=   "{$this->timestamp}, ";
        $sql .=   "{$this->data} ";
        $sql .= ") VALUES ( ";
        $sql .=   "'" . $this->conn->escape($this->getKey()) . "', ";
        $sql .=   " " . time() . ", ";
        $sql .=   "'" . $this->conn->escape($data) . "' ";
        $sql .= ") ";

        return $this->conn->exec($sql);
    }

    /**
     * キャッシュを削除する
     * 
     * @access public
     * @return bool true: 削除成功、false: 削除エラー
     */
    function delete()
    {
        $sql  = "";
        $sql .= "DELETE FROM ";
        $sql .=   "{$this->table} ";
        $sql .= "WHERE ";
        $sql .=   "{$this->id} = '" . $this->conn->escape($this->getKey()) . "' ";
        return $this->conn->exec($sql);
    }

    /**
     * キャッシュ中の期限切れキャッシュを削除する
     *
     * @access public
     */
    function clean()
    {
        if ($this->life_time == 0) {
            return;
        }

        $sql  = "";
        $sql .= "DELETE FROM ";
        $sql .=   "{$this->table} ";
        $sql .= "WHERE ";
        $sql .= "(";
        $sql .= "(" . time() . " > {$this->timestamp} + {$this->life_time}) ";
        if ($this->min_mtime > 0) {
            $sql .= "OR ({$this->timestamp} < {$this->min_mtime}) ";
        }
        if ($this->max_mtime > 0) {
            $sql .= "OR ({$this->timestamp} > {$this->max_mtime}) ";
        }
        $sql .= ") ";

        if ($this->prefix) {
            $sql .= "AND ({$this->id} like '" . $this->conn->escape($this->prefix) . "%') ";
        }
        $this->conn->exec($sql);
    }

    /**
     * キャッシュを全て削除する
     *
     * @access public
     */
    function cleanAll()
    {
        $sql  = "";
        $sql .= "DELETE FROM ";
        $sql .=   "{$this->table} ";
        $this->conn->exec($sql);
    }
}
