<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDaoTable.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * テーブルスキーマ定義クラス
 * 
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDaoTable.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_DBDaoTable
{
    /**
     * テーブル名
     * 
     * @access protected
     * @var string
     */
    var $table = '';
    /**
     * テーブル別名
     * 
     * @access protected
     * @var string
     */
    var $alias = '';
    /**
     * デフォルト全カラム取得フラグ
     *
     * @access protected
     * @var array
     */
    var $is_defaul_all = true;
    /**
     * プライマリキーカラム
     * 
     * @access protected
     * @var array
     */
    var $primary = array();
    /**
     * 一意キーカラム
     * 
     * @access protected
     * @var array
     */
    var $uniques = array();
    /**
     * 外部キーカラム
     * 
     * @access protected
     * @var array
     */
    var $foreigns = array();
    /**
     * カラム定義
     *
     * @access protected
     * @var array
     */
    var $columns = array();
    /**
     * 対象カラム
     * 
     * @access private
     * @var array
     */
    var $data_columns = array();
    /**
     * 対象カラムのSQL関数
     * 
     * @access private
     * @var array
     */
    var $data_functions = array();
    /**
     * 条件オブジェクト
     * 
     * @access protected
     * @var object
     */
    var $conditions = null;
    /**
     * ソートカラム
     * 
     * @access protected
     * @var array
     */
    var $order_bys = array();
    /**
     * グループ化カラム
     * 
     * @access protected
     * @var array
     */
    var $group_bys = array();

    /**
     * コンストラクタ
     *
     * @access public
     * @param string テーブル名
     */
    function SyL_DBDaoTable($table='', $columns=array())
    {
        if ($this->table == '') {
            $this->table = $table;
        }
        if (count($this->columns) == 0) {
            $this->columns = $columns;
        }
        // カラム名を大文字に変換
        $this->primary = array_map('strtoupper', $this->primary);
        $this->columns = array_change_key_case($this->columns, CASE_UPPER);

        // PHP4.xのオーバーロード
        if (version_compare(PHP_VERSION, '5.0.0', '<')) {
            //overload(get_class($this));
        }
    }

    /**
     * カラム値の追加のオーバーロードメソッド
     *
     * @access public
     * @param string メソッド名
     * @param array 引数
     * @param mixed 戻り値
     */
/*
    function __call($method, $args, &$return)
    {
        if (preg_match('/^set(.+)/', $method, $matches)) {
            $column_name = strtoupper($matches[1]);
            $this->set($column_name, $args[0]);
            return true;
        } else {
            return false;
        }
    }
*/
    /**
     * カラム値を追加
     *
     * @access public
     * @param string カラム名
     * @param mixed カラム値
     * @param array SQL関数
     */
    function set($name, $value=null, $function=null)
    {
        // 大文字に統一
        $name = strtoupper($name);
        if (isset($this->columns[$name])) {
            if (is_array($function)) {
                $this->data_functions[$name] = $function;
            }
            $this->data_columns[$name] = $value;
        } else {
            trigger_error("[SyL error] Column name not found in `" . get_class($this) . "' class ({$name})", E_USER_ERROR);
        }
    }

    /**
     * テーブル名を取得
     *
     * @access public
     * @return string テーブル名
     */
    function getName()
    {
        return $this->table;
    }

    /**
     * テーブル別名をセット
     *
     * @access public
     * @param string テーブル別名
     */
    function setAliasName($alias)
    {
        $this->alias = $alias;
    }

    /**
     * テーブル別名を取得
     *
     * @access public
     * @return string テーブル別名
     */
    function getAliasName()
    {
        return ($this->alias) ? $this->alias : $this->table;
    }

    /**
     * SELECT項目用テーブル名を取得
     *
     * @access public
     * @return array SELECT項目配列
     */
    function getSelectName()
    {
        if ($this->alias) {
            return $this->getName() . ' ' . $this->alias;
        } else {
            return $this->getName();
        }
    }

    /**
     * デフォルト全カラム取得フラグをセット
     *
     * @access public
     * @param bool デフォルト全カラム取得フラグ
     */
    function setDefaultAll($is_defaul_all)
    {
        $this->is_defaul_all = (bool)$is_defaul_all;
    }

    /**
     * カラム定義を取得
     *
     * @access public
     * @return array カラム定義
     */
    function getColumn($name)
    {
        // 大文字に統一
        $name = strtoupper($name);
        return $this->columns[$name];
    }

    /**
     * カラム定義を取得
     *
     * @access public
     * @return array カラム定義
     */
    function getColumns()
    {
        return $this->columns;
    }

    /**
     * 主キー配列を取得
     *
     * @access public
     * @param bool 別名追加フラグ
     * @return array 主キー配列
     */
    function getPrimary($alias_flag=true)
    {
        if ($alias_flag) {
            $alias = $this->getAliasName();
            $primary = array();
            foreach ($this->primary as $column) {
                $primary[] = "{$alias}.{$column}";
            }
            return $primary;
        } else {
            return $this->primary;
        }
    }

    /**
     * 一意キー配列を取得
     *
     * @access public
     * @param bool 別名追加フラグ
     * @return array 一意キー配列
     */
    function getUniques($alias_flag=true)
    {
        if ($alias_flag) {
            $alias = $this->getAliasName();
            $uniques = array();
            foreach ($this->uniques as $unique) {
                $tmp = array();
                foreach ($unique as $column) {
                    $tmp[] = "{$alias}.{$column}";
                }
                $uniques[] = $tmp;
            }
            return $uniques;
        } else {
            return $this->uniques;
        }
    }

    /**
     * 外部キー配列を取得
     *
     * @access public
     * @param bool 別名追加フラグ
     * @return array 外部キー配列
     */
    function getForeigns($alias_flag=true)
    {
        if ($alias_flag) {
            $alias = $this->getAliasName();
            $foreigns = array();
            foreach ($this->foreigns as $name => $foreign) {
                $tmp = array();
                foreach ($foreign as $column1 => $column2) {
                    $tmp["{$alias}.{$column1}"] = $column2;
                }
                $foreigns[$name] = $tmp;
            }
            return $foreigns;
        } else {
            return $this->foreigns;
        }
    }

    /**
     * 条件オブジェクトをセット
     *
     * @access public
     * @param object 条件オブジェクト
     */
    function setConditions($conditions)
    {
        $this->conditions = $conditions;
    }

    /**
     * 条件オブジェクトを取得
     *
     * @access public
     * @return object 条件オブジェクト
     */
    function getConditions($alias_flag=true)
    {
        $conditions = $this->conditions;
        if ($alias_flag && ($conditions !== null)) {
            $conditions->setAlias($this->getAliasName());
        }
        return $conditions;
    }

    /**
     * ソートカラムを追加
     *
     * @access public
     * @param string ソートカラム
     * @param bool ソート順
     */
    function addSort($name, $asc=true)
    {
        // 大文字に統一
        $name = strtoupper(trim($name));
        if (isset($this->columns[$name])) {
            $this->order_bys[] = array($name, $asc, false);
        } else if (array_search($name, $this->data_columns) !== false) {
            $this->order_bys[] = array($name, $asc, true);
        } else {
            trigger_error("[SyL error] Sort column name not found in " . get_class($this) . " class ({$name})", E_USER_ERROR);
        }
    }

    /**
     * ソートリンクからソートカラムを追加
     *
     * @access public
     * @param string ソートカラム
     */
    function addSortLink($linkname)
    {
        $sorts = array_map('trim', explode('.', $linkname, 3));
        if (count($sorts) == 3) {
            if ($this->getAliasName() == $sorts[0]) {
                $this->addSort($sorts[1], ($sorts[2] == 'ASC'));
            }
        }
    }

    /**
     * ソートカラムを取得
     *
     * @access public
     * @param bool 主キーソートフラグ
     * @return array ソートカラム
     */
    function getSorts($primary_sort=false)
    {
        $order_bys = $this->order_bys;
        if ($primary_sort) {
            foreach ($this->primary as $primary) {
                $exist = false;
                foreach ($order_bys as $order_by) {
                    if ($primary == $order_by[0]) {
                        $exist = true;
                        break;
                    }
                }
                if (!$exist) {
                    $order_bys[] = array($primary, true, false);
                }
            }
        }

        $result = array();
        for ($i=0; $i<count($order_bys); $i++) {
            $sort = ($order_bys[$i][1]) ? 'ASC' : 'DESC';
            if ($order_bys[$i][2]) {
                $result[$i] = $order_bys[$i][0] . ' ' . $sort;
            } else {
                $result[$i] = $this->getAliasName() . '.' . $order_bys[$i][0] . ' ' . $sort;
            }
        }
        return $result;
    }

    /**
     * グループ化カラムを追加
     *
     * @access public
     * @param string グループ化カラム
     */
    function addGroup($name)
    {
        // 大文字に統一
        $name = strtoupper(trim($name));
        if (isset($this->columns[$name])) {
            $this->group_bys[] = array($name, false);
        } else if (array_search($name, $this->data_columns) !== false) {
            $this->group_bys[] = array($name, true);
        } else {
            trigger_error("[SyL error] Group column name not found in " . get_class($this) . " class ({$name})", E_USER_ERROR);
        }
    }

    /**
     * グループ化カラムを取得
     *
     * @access public
     * @return array グループ化カラム
     */
    function getGroups()
    {
        $result = array();
        for ($i=0; $i<count($this->group_bys); $i++) {
            if ($this->group_bys[$i][1]) {
                $result[$i] = $this->group_bys[$i][0];
            } else {
                $result[$i] = $this->getAliasName() . '.' . $this->group_bys[$i][0];
            }
        }
        return $result;
    }

    /**
     * SELECT項目を取得
     *
     * @access public
     * @return array SELECT項目配列
     */
    function getSelectColumns(&$conn)
    {
        $columns_tmp = array();
        if (count($this->data_columns) > 0) {
            $columns_tmp = $this->data_columns;
        } else {
            if ($this->is_defaul_all) {
                foreach ($this->columns as $name => $value) {
                    $columns_tmp[$name] = null;
                }
            } else {
                return array();
            }
        }

        $columns = array();
        foreach ($columns_tmp as $name => $column_alias) {
            $column_tmp = '';
            if (isset($this->data_functions[$name])) {
                $funcs    = $this->data_functions[$name];
                $funcname = array_shift($funcs);
                $func = implode(',', array_map(array(&$conn, 'quote'), $funcs));
                if ($func != '') {
                    $func = ',' . $func;
                }
                $column_tmp = $funcname . '(' . $this->getAliasName() . '.' . $name . $func . ')';
            } else {
                $column_tmp = $this->getAliasName() . '.' . $name;
            }
            $columns[] = ($column_alias) ? "{$column_tmp} AS {$column_alias}" : $column_tmp;
        }
        return $columns;
    }

    /**
     * SELECT項目名を取得
     *
     * @access public
     * @return array SELECT項目名配列
     */
    function getSelectHeaders()
    {
        $alias = $this->getAliasName() . '.';
        $tmp_columns    = array();
        $column_aliases = array();
        if (count($this->data_columns) > 0) {
            foreach (array_keys($this->data_columns) as $name) {
                if ($this->data_columns[$name]) {
                    $column_aliases[$name] = $this->data_columns[$name];
                }
                $tmp_columns[$name] = $this->columns[$name];
            }
        } else {
            if ($this->is_defaul_all) {
                $tmp_columns =& $this->columns;
            } else {
                return array();
            }
        }

        $columns = array();
        foreach (array_keys($tmp_columns) as $name) {
            if (isset($column_aliases[$name])) {
                $column_name = $name = $column_aliases[$name];
            } else {
                $column_name = $alias . $name;
            }
            $exist = false;
            foreach ($this->getSorts() as $order_by) {
                list($sort, $order) = explode(' ', $order_by, 2);
                if ($column_name == $sort) {
                    $columns[$name]['order'] = ($order == 'ASC') ? 'DESC' : 'ASC'; // リンクなのでソートが逆
                    $exist = true;
                    break;
                }
            }
            if (!$exist) {
                $columns[$name]['order'] = 'ASC';
            }
            $columns[$name]['sort'] = $alias . $name;
        }
        return $columns;
    }

    /**
     * DML用項目を取得
     *
     * @access public
     * @param bool テーブル別名追加フラグ
     * @return array DML用項目配列
     */
    function getDataColumns($is_alias=false)
    {
        $columns = array();
        foreach ($this->data_columns as $name => $value) {
            if (isset($this->data_functions[$name])) {
                $funcs    = $this->data_functions[$name];
                $funcname = array_shift($funcs);
                $func     = implode(',', array_map(array(&$conn, 'quote'), $funcs));
                $value    = "{$funcname}({$func})";
            }
            if ($is_alias && $this->alias) {
                $columns[$this->getAliasName() . '.' . $name] = $value;
            } else {
                $columns[$name] = $value;
            }
        }
        return $columns;
    }

    /**
     * データ項目配列を初期化する
     *
     * @access public
     */
    function initDataColumns()
    {
        $this->data_columns = array();
    }
}
