<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDriverOci8.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 *  DB操作クラス（Oracle）
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDriverOci8.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_DBDriverOci8 extends SyL_DB
{
    /**
     * ステートメント
     *
     * @access private
     * @var resource
     */
    var $stmt = null;

    /**
     * エンコード変換テーブル
     *
     * ※全て小文字で記述
     *
     * @access protected
     * @var string
     */
    var $encode_table = array(
      // PHP, DB
      array('eucjp-win', 'japanese_japan.ja16euc'),
      array('euc-jp',    'japanese_japan.ja16euc'),
      array('euc',       'japanese_japan.ja16euc'),
      array('sjis-win',  'japanese_japan.ja16sjis'),
      array('sjis',      'japanese_japan.ja16sjis'),
      array('utf-8',     'japanese_japan.utf8')
    );

    /**
     * DB接続実行
     * 
     * @access public
     * @param string データベース名
     * @param string ユーザー名
     * @param string パスワード
     * @param string ホスト名
     * @param string ポート番号
     * @param bool 持続的接続を行うか
     * @return boolean 接続OK: true, 接続NG: false
     */
    function openConnection($dbname, $user, $passwd, $host, $port, $persistent)
    {
        $dbname_tmp = $dbname;
        if ($host) {
            $dbname_tmp = ($port) ? "{$host}:{$port}/{$dbname_tmp}" : "{$host}/{$dbname_tmp}";
        }
        if ($this->client_encode) {
            $this->connection = $persistent ? ociplogon($user, $passwd, $dbname_tmp, $this->client_encode) : ocilogon($user, $passwd, $dbname_tmp, $this->client_encode);
        } else {
            $this->connection = $persistent ? ociplogon($user, $passwd, $dbname_tmp) : ocilogon($user, $passwd, $dbname_tmp);
        }
        return is_resource($this->connection);
    }

    /**
     * DB接続終了
     * 
     * @access public
     */
    function closeConnection()
    {
        if (is_resource($this->connection)) {
            ocilogoff($this->connection);
            $this->connection = null;
        }
    }

    /**
     * クライアント側文字エンコーティング設定
     * 
     * @access public
     * @param string DB文字コード
     * @param string PHP側エンコード
     */
    function setClientEncoding($client_encode, $server_encode='')
    {
        $client_encode2 = strtolower($client_encode);
        foreach ($this->encode_table as $encode_table) {
            if ($encode_table[0] == $client_encode2) {
                $client_encode2 = $encode_table[1];
                break;
            }
        }
        putenv("NLS_LANG={$client_encode2}");
        parent::setClientEncoding($client_encode, $server_encode);
    }

    /**
     * 最後に起こったエラーメッセージをセット
     *
     * @access public
     * @param string エラーコード
     * @param string エラーメッセージ
     */
    function setErrorMessage($code='', $message='')
    {
        if ($code) {
            parent::setErrorMessage($code, $message);
        } else {
            $error = false;
            if ($this->stmt) {
                $error = $this->stmt ? ocierror($this->stmt) : ocierror();
            } else {
                $error = $this->connection ? ocierror($this->connection) : ocierror();
            }
            if ($error) {
                $this->last_error_code    = $error['code'];
                $this->last_error_message = $error['message'];
            }
        }
    }

    /**
     * パラメータサニタイズ（無効化）処理
     * 
     * @access public
     * @param string サニタイズ対象文字列
     * @return string サニタイズ後文字列
     */
    function escape($parameter)
    {
        return str_replace("'", "''", $parameter);
    }

    /**
     * トランザクション開始
     *
     * @access public
     */
    function beginTransaction()
    {
        parent::beginTransaction();
    }

    /**
     * トランザクション破棄
     *
     * @access public
     */
    function rollBack()
    {
        @ocirollback($this->connection);
        parent::rollBack();
    }

    /**
     * トランザクション確定
     *
     * @access public
     */
    function commit()
    {
        @ocicommit($this->connection);
        parent::commit();
    }

    /**
     * SQLを実行し、DBリソースを取得
     *
     * @access public
     * @param string SQL文
     * @return mixed 実行OK: DBリソース, 実行NG: false
     */
    function execSelect($sql)
    {
        if ($this->createStatement($sql)) {
            return $this->stmt;
        } else {
            $this->setErrorMessage();
            $this->closeStatement();
            $this->triggerError();
            return false;
        }
    }

    /**
     * SQLを実行し、実行結果影響件数を取得
     *
     * @access public
     * @param string SQL文
     * @return mixed 実行OK: 実行結果影響件数, 実行NG: false
     */
    function execUpdate($sql)
    {
        if ($this->createStatement($sql)) {
            $num = ocirowcount($this->stmt);
            $this->closeStatement();
            return $num;
        } else {
            $this->setErrorMessage();
            $this->closeStatement();
            $this->triggerError();
            return false;
        }
    }

    /**
     * SQL実行のみ
     *
     * @access public
     * @param string SQL文
     * @return boolean 実行OK: true, 実行NG: false
     */
    function execNoReturn($sql)
    {
        if ($this->createStatement($sql)) {
            return true;
        } else {
            $this->setErrorMessage();
            $this->closeStatement();
            $this->triggerError();
            return false;
        }
    }

    /**
     * SQL実行し、結果データを取得
     *
     * ・第3パラメータが、'one'の場合
     *    最初の1カラムを文字列として取得
     * ・第3パラメータが、'record'の場合
     *    最初の1レコードを配列として取得
     * ・第3パラメータが、'all'の場合
     *    全レコードを配列として取得
     *
     * @access public
     * @param string SQL文
     * @param mixed  実行結果を取得
     * @param string 結果取得フラグ
     * @return boolean 実行OK: true, 実行NG: false
     */
    function queryRef($sql, &$data, $get='all')
    {
        // SQL実行
        $this->execRef($sql, $query);
        // 取得判定
        if (!is_resource($query)) {
            $data = $query;
            return ($query !== false);
        }
        switch ($get) {
        // 1フィールド
        case 'one':
            $tmp = array();
            if (ocifetchinto($this->stmt, $tmp, OCI_NUM + OCI_RETURN_NULLS) > 0) {
                $data = $tmp[0];
            } else {
                $data = '';
            }
            break;
        // 1レコード
        case 'record':
            $data = array();
            if (ocifetchinto($this->stmt, $data, OCI_ASSOC + OCI_RETURN_NULLS)) {
                $data = $this->caseFieldName($data);
            } else {
                $data = array();
            }
            break;
        // 全レコード
        case 'all':
            $data = array();
            $tmp  = array();
            while (ocifetchinto($this->stmt, $tmp, OCI_ASSOC + OCI_RETURN_NULLS)) {
                $data[] = $tmp;
            }
            break;
        }
        $this->closeStatement();
        return true;
    }

    /**
     * SQLを実行し、ページ毎にデータを取得
     *
     * @access public
     * @param string SQL文
     * @param array 実行結果格納配列
     * @param object ページオブジェクト
     * @param int 1ページの表示件数
     * @param int 表示対象ページ数 
     * @return bool 実行結果
     */
    function queryPageRef($sql, &$result, &$pager, $limit=20, $page=1)
    {
        // ページングオブジェクト
        $pager = $this->getPager($limit, $page);
        // 総件数取得SQL作成
        $sql_count = "SELECT COUNT(*) FROM (" . $sql . ") SyL$";
        // 件数取得SQL実行
        if (!$this->queryRef($sql_count, $record, 'one')) return false;
        $pager->setCountAll($record);

        // ページ単位レコード取得
        $sql_page  =  "";
        $sql_page .= "SELECT ";
        $sql_page .=   "SyL2$.* ";
        $sql_page .= "FROM ";
        $sql_page .=   "(";
        $sql_page .=     "SELECT ";
        $sql_page .=       "SyL$.*, ";
        $sql_page .=       "ROWNUM as ROWNUM$ ";
        $sql_page .=     "FROM ";
        $sql_page .=       "(" . $sql . ") SyL$ ";
        $sql_page .=   ") SyL2$ ";
        $sql_page .= "WHERE ";
        $sql_page .=   "ROWNUM$ BETWEEN " . ($pager->getStartOffset() + 1) . " AND " . ($pager->getStartOffset() + $pager->getCount());

        // データ取得SQL実行
        return $this->queryRef($sql_page, $result, 'all');
    }

    /**
     * SQLをパースし、ステートメントリソースを取得する
     *
     * @access private
     * @param string SQL文
     * @return bool true: SQL実行OK、false: SQLパース、実行エラー
     */
    function createStatement($sql)
    {
        $this->closeStatement();
        $this->stmt = ociparse($this->connection, $sql);
        if (!$this->stmt) {
            return false;
        }
        return ($this->transaction) ? @ociexecute($this->stmt, OCI_DEFAULT)
                                    : @ociexecute($this->stmt, OCI_COMMIT_ON_SUCCESS);
    }

    /**
     * ステートメントリソースを開放する
     *
     * @access private
     */
    function closeStatement()
    {
        if (is_resource($this->stmt)) {
            ocifreestatement($this->stmt);
            $this->stmt = null;
        }
    }

    /**
     * 最後に挿入された行の ID あるいはシーケンスの値を取得
     *
     * @access public
     * @return int 最後に挿入された行のID
     */
    function lastInsertId($seq='')
    {
        $sql = "SELECT CURRVAL('{$seq}') FROM DUAL";
        $this->queryRef($sql, $data, 'one');
        return $data;
    }

    /**
     * 接続しているDBサーバーのバージョンを取得する
     * 
     * @access public
     * @return string DBのバージョン
     */
    function getVersion()
    {
        $sql = 'SELECT BANNER FROM V$VERSION';
        $this->queryRef($sql, $versions, 'one');
        $version = false;
        if (preg_match ('/ ([\d\.]+) \-/', $versions, $matches)) {
            $version = $matches[1];
        }
        return $version;
    }
}
