<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDriverOdbc.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 *  DB操作クラス（ODBC）
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDriverOdbc.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_DBDriverOdbc extends SyL_DB
{
    /**
     * 接続カーソルタイプ
     *
     * @access private
     * @var int
     */
    var $cursor_type = SQL_CUR_USE_ODBC;

    /**
     * DB接続実行
     * 
     * @access public
     * @param string データベース名
     * @param string ユーザー名
     * @param string パスワード
     * @param string ホスト名
     * @param string ポート番号
     * @param bool 持続的接続を行うか
     * @return boolean 接続OK: true, 接続NG: false
     */
    function openConnection($dbname, $user, $passwd, $host, $port, $persistent)
    {
        $this->connection = $persistent ? odbc_pconnect($dbname, $user, $passwd, $this->cursor_type)
                                        : odbc_connect ($dbname, $user, $passwd, $this->cursor_type);
        return is_resource($this->connection);
    }

    /**
     * DB接続終了
     * 
     * @access public
     */
    function closeConnection()
    {
        if (is_resource($this->connection)) {
            odbc_close($this->connection);
            $this->connection = null;
        }
    }

    /**
     * 最後に起こったエラーメッセージをセット
     *
     * @access public
     * @param string エラーコード
     * @param string エラーメッセージ
     */
    function setErrorMessage($code='', $message='')
    {
        if ($code) {
            parent::setErrorMessage($code, $message);
        } else {
            $code = $this->connection ? odbc_error($this->connection) : odbc_error();
            if ($code) {
                $this->last_error_code    = $code;
                $this->last_error_message = $this->connection ? odbc_errormsg($this->connection) : odbc_errormsg();
            }
        }
    }

    /**
     * トランザクション開始
     *
     * @access public
     */
    function beginTransaction()
    {
        parent::beginTransaction();
        odbc_autocommit($this->connection, false);
    }

    /**
     * トランザクション破棄
     *
     * @access public
     */
    function rollBack()
    {
        odbc_rollback($this->connection);
        parent::rollBack();
    }

    /**
     * トランザクション確定
     *
     * @access public
     */
    function commit()
    {
        odbc_commit($this->connection);
        parent::commit();
    }

    /**
     * SQLを実行し、結果取得
     * 
     * ・SQL文が、select句の場合
     *   実行結果をリソースとして取得
     * ・SQL文が、insert, update, delete句の場合
     *   実行結果影響件数を取得
     * ・SQL文が、上記以外の場合
     *   実行後、true or falseを返却
     *
     * @access public
     * @param string SQL文
     * @param mixed 実行結果件数、またはDBリソース
     * @return boolean 実行OK: true, 実行NG: false
     */
    function execRef($sql, &$result)
    {
        $this->convertEncoding($sql);
        parent::execRef($sql, $result);
    }

    /**
     * SQLを実行し、DBリソースを取得
     *
     * @access public
     * @param string SQL文
     * @return mixed 実行OK: DBリソース, 実行NG: false
     */
    function execSelect($sql)
    {
        $result = @odbc_exec($this->connection, $sql);
        if ($result !== false) {
            return $result;
        } else {
            $this->setErrorMessage();
            $this->triggerError();
            return false;
        }
    }

    /**
     * SQLを実行し、実行結果影響件数を取得
     *
     * @access public
     * @param string SQL文
     * @return mixed 実行OK: 実行結果影響件数, 実行NG: false
     */
    function execUpdate($sql)
    {
        // 実行時エラー判定
        $result = @odbc_exec($this->connection, $sql);
        if ($result !== false) {
            return odbc_num_rows($result);
        } else {
            $this->setErrorMessage();
            $this->triggerError();
            return false;
        }
    }

    /**
     * SQL実行のみ
     *
     * @access public
     * @param string SQL文
     * @return boolean 実行OK: true, 実行NG: false
     */
    function execNoReturn($sql)
    {
        // 実行時エラー判定
        $result = @odbc_exec($this->connection, $sql);
        if ($result !== false) {
            return true;
        } else {
            $this->setErrorMessage();
            $this->triggerError();
            return false;
        }
    }
  
    /**
     * SQL実行し、結果データを取得
     *
     * ・第3パラメータが、'one'の場合
     *    最初の1カラムを文字列として取得
     * ・第3パラメータが、'record'の場合
     *    最初の1レコードを配列として取得
     * ・第3パラメータが、'all'の場合
     *    全レコードを配列として取得
     *
     * @access public
     * @param string SQL文
     * @param mixed  実行結果を取得
     * @param string 結果取得フラグ
     * @return boolean 実行OK: true, 実行NG: false
     */
    function queryRef($sql, &$data, $get='all')
    {
        // SQL実行
        $this->execRef($sql, $query);
  
        // 実行時エラー判定
        if (!is_resource($query)) {
            $data = $query;
            return ($query !== false);
        }

        switch ($get) {
        // 1フィールド
        case 'one':
            if (odbc_fetch_row($query)) {
                $data = odbc_result($query, 1);
            } else {
                $data = '';
            }
            break;
        // 1レコード
        case 'record':
            $data = array();
            if (odbc_fetch_row($query)) {
                $cols = odbc_num_fields($query);
                for ($i=1; $i<=$cols; $i++) {
                    $data[odbc_field_name($query, $i)] = odbc_result($query, $i);
                }
                $data = $this->caseFieldName($data);
            }
            break;
        // 全レコード
        case 'all':
            $data = array();
            $cols = odbc_num_fields($query);
            while (odbc_fetch_row($query)) {
                $tmp = array();
                for ($i=1; $i<=$cols; $i++) {
                    $tmp[odbc_field_name($query, $i)] = odbc_result($query, $i);
                }
                $data[] = $this->caseFieldName($tmp);
            }
            break;
        }
        odbc_free_result($query);

        $this->convertDecoding($data);
        return true;
    }

    /**
     * SQLを実行し、ページ毎にデータを取得
     * ※実行前に「initializePage」メソッド実行必須
     * ※SQL文の「FROM」句の直前にタブ文字必須。（それ以外のタブ文字不可）
     *
     * @access public
     * @param string SQL文
     * @param array 実行結果格納配列
     * @param object ページオブジェクト
     * @param int 1ページの表示件数
     * @param int 表示対象ページ数 
     * @return bool 実行結果
     */
    function queryPageRef($sql, &$result, &$pager, $limit=20, $page=1)
    {
        trigger_error("[SyL error] queryPageRef method not implemented this class (" . __CLASS__ . ")", E_USER_ERROR);
    }
}
