<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDriverPear.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * PEAR::DBクラス
 */
require_once 'DB.php';

/**
 *  DB操作クラス（PEAR）
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDriverPear.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_DBDriverPear extends SyL_DB
{
    /**
     * DB接続実行
     * 
     * @access public
     * @param string データベース名
     * @param string ユーザー名
     * @param string パスワード
     * @param string ホスト名
     * @param string ポート番号
     * @param bool 持続的接続を行うか
     * @return boolean 接続OK: true, 接続NG: false
     */
    function openConnection($dbname, $user, $passwd, $host, $port, $persistent)
    {
        $dsn = $this->dbtype . '://';
        if ($user) {
            $dsn .= ($passwd) ? "{$user}:{$passwd}" : $user;
        }
        if ($host) {
            $dsn .= ($port) ? "@{$host}:{$port}" : "@{$host}";
        }
        if ($dbname) {
            $dsn  .= '/' . $dbname;
        }

        $this->connection =& DB::connect($dsn, $persistent);
        if (DB::isError($this->connection)) {
            $this->setErrorMessagePear($this->connection);
            return false;
        } else {
            return true;
        }
    }

    /**
     * DB接続終了
     * 
     * @access public
     */
    function closeConnection()
    {
        if (is_object($this->connection)) {
            $this->connection->disconnect();
            $this->connection = null;
        }
    }

    /**
     * 最後に起こったエラーメッセージをセット
     *
     * @access public
     * @param object PEAR_Errorオブジェクト
     */
    function setErrorMessagePear(&$error)
    {
        $error_message  = $error->getMessage() . ' (' . $error->getUserInfo() . ')';
        if ($error->getUserInfo()) {
            $error_message .= ' (' . $error->getUserInfo() . ')';
        }
        $this->setErrorMessage($error->getCode(), $error_message);
    }

    /**
     * パラメータサニタイズ（無効化）処理
     * 
     * @access public
     * @param string サニタイズ対象文字列
     * @return string サニタイズ後文字列
     */
    function escape($parameter)
    {
        return $this->connection->escapeSimple($parameter);
    }

    /**
     * パラメータを値によりクォートした文字列を取得する
     * ※サニタイズも実行される
     *
     * @access public
     * @param string クォート前文字列
     * @return string クォート後文字列
     */
    function quote($value)
    {
        return $this->connection->quoteSmart($value);
    }

    /**
     * トランザクション開始
     *
     * @access public
     */
    function beginTransaction()
    {
        parent::beginTransaction();
        $result = $this->connection->autoCommit(false);
        if (DB::isError($result)) {
            $this->setErrorMessagePear($result);
            return false;
        } else {
            return true;
        }
    }

    /**
     * トランザクション破棄
     *
     * @access public
     */
    function rollBack()
    {
        $result = $this->connection->rollback();
        parent::rollBack();
        if (DB::isError($result)) {
            $this->setErrorMessagePear($result);
            return false;
        } else {
            return true;
        }
    }

    /**
     * トランザクション確定
     *
     * @access public
     */
    function commit()
    {
        $result = $this->connection->commit();
        parent::commit();
        if (DB::isError($result)) {
            $this->setErrorMessagePear($result);
            return false;
        } else {
            return true;
        }
    }

    /**
     * SQLを実行し、結果取得
     * 
     * ・SQL文が、select句の場合
     *   実行結果をリソースとして取得
     * ・SQL文が、insert, update, delete句の場合
     *   実行結果影響件数を取得
     * ・SQL文が、上記以外の場合
     *   実行後、true or falseを返却
     *
     * @access public
     * @param string SQL文
     * @param mixed 実行結果件数、またはDBリソース
     * @return boolean 実行OK: true, 実行NG: false
     */
    function execRef($sql, &$result)
    {
        $this->convertEncoding($sql);
        parent::execRef($sql, $result);
    }

    /**
     * SQLを実行し、DBリソースを取得
     *
     * @access public
     * @param string SQL文
     * @return mixed 実行OK: DBリソース, 実行NG: false
     */
    function execSelect($sql)
    {
        $result =& $this->connection->query($sql);
        if (!DB::isError($result)) {
            return $result;
        } else {
            $this->setErrorMessagePear($result);
            $this->triggerError();
            return false;
        }
    }

    /**
     * SQLを実行し、実行結果影響件数を取得
     *
     * @access public
     * @param string SQL文
     * @return mixed 実行OK: 実行結果影響件数, 実行NG: false
     */
    function execUpdate($sql)
    {
        $result =& $this->connection->query($sql);
        if (!DB::isError($result)) {
            return $this->connection->affectedRows();
        } else {
            $this->setErrorMessagePear($result);
            $this->triggerError();
            return false;
        }
    }

    /**
     * SQL実行のみ
     *
     * @access public
     * @param string SQL文
     * @return boolean 実行OK: true, 実行NG: false
     */
    function execNoReturn($sql)
    {
        $result =& $this->connection->query($sql);
        if (!DB::isError($result)) {
            return true;
        } else {
            $this->setErrorMessagePear($result);
            $this->triggerError();
            return false;
        }
    }

    /**
     * SQL実行し、結果データを取得
     *
     * ・第3パラメータが、'one'の場合
     *    最初の1カラムを文字列として取得
     * ・第3パラメータが、'record'の場合
     *    最初の1レコードを配列として取得
     * ・第3パラメータが、'all'の場合
     *    全レコードを配列として取得
     *
     * @access public
     * @param string SQL文
     * @param mixed  実行結果を取得
     * @param string 結果取得フラグ
     * @return boolean 実行OK: true, 実行NG: false
     */
    function queryRef($sql, &$data, $get='all')
    {
        // SQL実行
        $this->execRef($sql, $query);

        // 取得判定
        if (!is_object($query)) {
            $data = $query;
            return ($query !== false);
        }

        switch ($get) {
        case 'one':
            $result =& $query->fetchRow(DB_FETCHMODE_ORDERED);
            if (is_array($result) && (count($result) > 0)) {
                list($data) = $result;
            }
            break;
        case 'record':
            $result =& $query->fetchRow(DB_FETCHMODE_ASSOC);
            if ($result === null) {
                $data = array();
            } else {
                $data = $this->caseFieldName($result);
            }
            break;
        case 'all':
            $data = array();
            while ($result =& $query->fetchRow(DB_FETCHMODE_ASSOC)) {
                $data[] = $this->caseFieldName($result);
            }
            break;
        }
        $query->free();

        $this->convertDecoding($data);
        return true;
    }

    /**
     * SQLを実行し、ページ毎にデータを取得
     *
     * @access public
     * @param string SQL文
     * @param array 実行結果格納配列
     * @param object ページオブジェクト
     * @param int 1ページの表示件数
     * @param int 表示対象ページ数 
     * @return bool 実行結果
     */
    function queryPageRef($sql, &$result, &$pager, $limit=20, $page=1)
    {
        // ページングオブジェクト
        $pager = $this->getPager($limit, $page);
        // 総件数取得SQL作成
        $sql_count = "SELECT COUNT(*) FROM (" . $sql . ") AS SyL$";
        // 件数取得SQL実行
        if (!$this->queryRef($sql_count, $record, 'one')) return false;
        $pager->setCountAll($record);

        // データ取得SQL実行
        $query =& $this->connection->limitQuery($sql, $pager->getStartOffset(), $pager->getCount());
        if (DB::isError($query)) {
            $this->setErrorMessagePear($query);
            $this->triggerError();
            return false;
        }

        $result = array();
        while($row =& $query->fetchRow(DB_FETCHMODE_ASSOC)){
           $result[] = $row;
        }
    }
}
