<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDriverSqlite.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * DB操作クラス（SQLITE）
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDriverSqlite.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_DBDriverSqlite extends SyL_DB
{
    /**
     * DBサーバーエンコード
     * 
     * @access protected
     * @var string
     */
    public $server_encode = 'utf-8';

    /**
     * DB接続実行
     * 
     * @access public
     * @param string データベース名
     * @param string ユーザー名
     * @param string パスワード
     * @param string ホスト名
     * @param string ポート番号
     * @param bool 持続的接続を行うか
     * @return boolean 接続OK: true, 接続NG: false
     */
    public function openConnection($dbname, $user, $passwd, $host, $port, $persistent)
    {
        if ($this->field_case) {
            ini_set('sqlite.assoc_case', $this->field_case);
        }

        $this->connection = $persistent ? sqlite_popen($dbname, 0666, $error_msg)
                                        : sqlite_open ($dbname, 0666, $error_msg);
        if (!$error_msg && is_resource($this->connection)) {
            return true;
        } else {
            $this->setErrorMessage('9999', $error_msg);
            return false;
        }
    }

    /**
     * DB接続終了
     * 
     * @access public
     */
    public function closeConnection()
    {
        if (is_resource($this->connection)) {
            sqlite_close($this->connection);
            $this->connection = null;
        }
    }

    /**
     * 最後に起こったエラーメッセージをセット
     *
     * @access public
     * @param string エラーコード
     * @param string エラーメッセージ
     */
    public function setErrorMessage($code='', $message='')
    {
        if ($code) {
            parent::setErrorMessage($code, $message);
        } else {
            $code = sqlite_last_error($this->connection);
            if ($code != 0) {
                $this->last_error_code    = $code;
                $this->last_error_message = sqlite_error_string($code);
            }
        }
    }

    /**
     * パラメータサニタイズ（無効化）処理
     * 
     * @access public
     * @param string サニタイズ対象文字列
     * @return string サニタイズ後文字列
     */
    public function escape($parameter)
    {
        return sqlite_escape_string($parameter);
    }

    /**
     * トランザクション開始
     *
     * @access public
     */
    public function beginTransaction()
    {
        parent::beginTransaction();
        @sqlite_exec($this->connection, 'BEGIN');
    }

    /**
     * トランザクション破棄
     *
     * @access public
     */
    public function rollBack()
    {
        @sqlite_exec($this->connection, 'ROLLBACK');
        parent::rollBack();
    }

    /**
     * トランザクション確定
     *
     * @access public
     */
    public function commit()
    {
        @sqlite_exec($this->connection, 'COMMIT');
        parent::commit();
    }

    /**
     * SQLを実行し、結果取得
     * 
     * ・SQL文が、select句の場合
     *   実行結果をリソースとして取得
     * ・SQL文が、insert, update, delete句の場合
     *   実行結果影響件数を取得
     * ・SQL文が、上記以外の場合
     *   実行後、true or falseを返却
     *
     * @access public
     * @param string SQL文
     * @param mixed 実行結果件数、またはDBリソース
     * @return boolean 実行OK: true, 実行NG: false
     */
    public function execRef($sql, &$result)
    {
        $this->convertEncoding($sql);
        return parent::execRef($sql, $result);
    }

    /**
     * SQLを実行し、DBリソースを取得
     *
     * @access public
     * @param string SQL文
     * @return mixed 実行OK: DBリソース, 実行NG: false
     */
    public function execSelect($sql)
    {
        $error_code = null;
        $error_msg  = null;
        if (PHP_VERSION >= 5.1) {
            $result = @sqlite_query($this->connection, $sql, SQLITE_BOTH, $error_msg);
            if (!$result) {
                $error_code = '9999';
            }
        } else {
            $result = @sqlite_query($this->connection, $sql, SQLITE_BOTH);
        }

        if ($result !== false) {
            return $result;
        } else {
            $this->setErrorMessage($error_code, $error_msg);
            $this->triggerError();
            return false;
        }
    }

    /**
     * SQLを実行し、実行結果影響件数を取得
     *
     * @access public
     * @param string SQL文
     * @return mixed 実行OK: 実行結果影響件数, 実行NG: false
     */
    public function execUpdate($sql)
    {
        $error_code = null;
        $error_msg  = null;
        if (PHP_VERSION >= 5.1) {
            $result = @sqlite_exec($this->connection, $sql, $error_msg);
            if (!$result) {
                $error_code = '9999';
            }
        } else {
            $result = @sqlite_exec($this->connection, $sql);
        }

        if ($result !== false) {
            return sqlite_changes($this->connection);
        } else {
            $this->setErrorMessage($error_code, $error_msg);
            $this->triggerError();
            return false;
        }
    }

    /**
     * SQL実行のみ
     *
     * @access public
     * @param string SQL文
     * @return boolean 実行OK: true, 実行NG: false
     */
    public function execNoReturn($sql)
    {
        $error_code = null;
        $error_msg  = null;
        if (PHP_VERSION >= 5.1) {
            $result = @sqlite_exec($this->connection, $sql, $error_msg);
            if (!$result) {
                $error_code = '9999';
            }
        } else {
            $result = @sqlite_exec($this->connection, $sql);
        }

        if ($result !== false) {
            return true;
        } else {
            $this->setErrorMessage($error_code, $error_msg);
            $this->triggerError();
            return false;
        }
    }

    /**
     * SQL実行し、結果データを取得
     *
     * ・第3パラメータが、'one'の場合
     *    最初の1カラムを文字列として取得
     * ・第3パラメータが、'record'の場合
     *    最初の1レコードを配列として取得
     * ・第3パラメータが、'all'の場合
     *    全レコードを配列として取得
     *
     * @access public
     * @param string SQL文
     * @param mixed  実行結果を取得
     * @param string 結果取得フラグ
     * @return boolean 実行OK: true, 実行NG: false
     */
    public function queryRef($sql, &$data, $get='all')
    {
        // 取得判定
        if (!$this->execRef($sql, $query) || !is_resource($query)) {
            $data = $query;
            return ($query !== false);
        }

        switch ($get) {
        case 'one':
            if (sqlite_num_rows($query) > 0) {
                $data = sqlite_fetch_single($query);
            } else {
                $data = '';
            }
            break;
        case 'record':
            $data = sqlite_fetch_array($query, SQLITE_ASSOC);
            if ($data === false) {
                $data = array();
            }
            break;
        case 'all':
            $data = sqlite_fetch_all($query, SQLITE_ASSOC);
            break;
        }
        $query = null;

        $this->convertDecoding($data);
        return true;
    }

    /**
     * SQLを実行し、ページ毎にデータを取得
     *
     * @access public
     * @param string SQL文
     * @param array 実行結果格納配列
     * @param object ページオブジェクト
     * @param int 1ページの表示件数
     * @param int 表示対象ページ数 
     * @return bool 実行結果
     */
    public function queryPageRef($sql, &$result, &$pager, $limit=20, $page=1)
    {
        trigger_error("[SyL error] " . __METHOD__ . " method not implemented this class (" . __CLASS__ . ")", E_USER_ERROR);
    }

    /**
     * 最後に挿入された行の ID あるいはシーケンスの値を取得
     *
     * @access public
     * @return int 最後に挿入された行のID
     */
    public function lastInsertId($seq='')
    {
        return sqlite_last_insert_rowid($this->connection);
    }

    /**
     * 接続しているDBサーバーのバージョンを取得する
     * 
     * @access public
     * @return string DBのバージョン
     */
    public function getVersion()
    {
        return sqlite_libversion();
    }
}
