<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBSqlBuilder.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * SQLテーブルクラス
 */
require_once dirname(__FILE__) . '/SqlBuilder/SyL_DBSqlBuilderTable.php';
/**
 * SQLカラムデータクラス
 */
require_once dirname(__FILE__) . '/SqlBuilder/SyL_DBSqlBuilderColumnData.php';

/**
 * SQL構築クラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBSqlBuilder.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_DBSqlBuilder
{
    /**
     * DBコネクションオブジェクト
     *
     * @access protected
     * @var object
     */
    var $conn = null;
    /**
     * テーブル名
     *
     * @access protected
     * @var array
     */
    var $tables = array();
    /**
     * カラム名
     *
     * @access private
     * @var array
     */
    var $column_datas = array();
    /**
     * 条件
     *
     * @access protected
     * @var array
     */
    var $wheres = array();
    /**
     * ソート
     *
     * @access private
     * @var array
     */
    var $order_bys = array();
    /**
     * グループ化
     *
     * @access private
     * @var array
     */
    var $group_bys = array();

    /**
     * コンストラクタ
     *
     * @access public
     * @param object DBオブジェクト
     */
    function SyL_DBSqlBuilder(&$conn)
    {
        $this->conn =& $conn;
    }

    /**
     * テーブルオブジェクトを作成する
     *
     * @access public
     * @param string テーブル名
     * @param string テーブル別名
     * @return object テーブルオブジェクト
     */
    function &getTable($name, $alias='')
    {
        $table =& new SyL_DBSqlBuilderTable($this->conn, $name, $alias);
        return $table;
    }

    /**
     * SQLに使用するテーブルをセットする
     *
     * @access public
     * @param object テーブルオブジェクト
     */
    function addTable($table)
    {
        $this->tables[] = $table;
    }

    /**
     * DAOテーブルスキーマ定義オブジェクトをセットする
     *
     * @access public
     * @param object テーブルスキーマ定義オブジェクト
     */
    function addTableDao($dao_table, $alias='')
    {
        $table =& new SyL_DBSqlBuilderTable($this->conn, $dao_table->getTableName(), $alias);
        $this->addTable($table);
    }

    /**
     * SQLに使用するカラムをセットする
     *
     * @param object カラムオブジェクト
     * @param mixed カラム値
     */
    function addColumn($column, $value=null)
    {
        $this->column_datas[] = new SyL_DBSqlBuilderColumnData($column, $value, '=');
    }

    /**
     * where条件を作成する（等価比較）
     *
     * @access public
     * @param object カラムオブジェクト
     * @param string 値
     * @param bool true/=, false/<>
     * @param string AND or OR
     */
    function addWhereEqual($column, $value, $equal=true, $operator='AND')
    {
        $op = ($equal) ? '=' : '<>';
        $this->addWhere(new SyL_DBSqlBuilderColumnData($column, $value, $op), $operator);
    }

    /**
     * where条件を作成する（NULL比較）
     *
     * @access public
     * @param object カラムオブジェクト
     * @param bool true/=, false/<>
     * @param string AND or OR
     */
    function addWhereNull($column, $equal=true, $operator='AND')
    {
        $op = ($equal) ? 'IS NULL' : 'IS NOT NULL';
        $this->addWhere(new SyL_DBSqlBuilderColumnData($column, null, $op), $operator);
    }

    /**
     * where条件を作成する（LIKE比較）
     *
     * @access public
     * @param object カラムオブジェクト
     * @param string 値
     * @param bool true/=, false/<>
     * @param string AND or OR
     */
    function addWhereLike($column, $value, $equal=true, $operator='AND')
    {
        $op = ($equal) ? 'LIKE' : 'NOT LIKE';
        $this->addWhere(new SyL_DBSqlBuilderColumnData($column, $value, $op), $operator);
    }

    /**
     * where条件を作成する（IN比較）
     *
     * @access public
     * @param object カラムオブジェクト
     * @param mixed 値
     * @param bool true/=, false/<>
     * @param string AND or OR
     */
    function addWhereIn($column, $value, $equal=true, $operator='AND')
    {
        $op = ($equal) ? 'IN' : 'NOT IN';
        $this->addWhere(new SyL_DBSqlBuilderColumnData($column, $value, $op), $operator);
    }

    /**
     * where条件を作成する
     *
     * @access public
     * @param object カラムデータオブジェクト
     * @param string AND or OR
     */
    function addWhere($where, $operator='AND')
    {
        switch (strtoupper($operator)) {
        case 'AND';
        case 'OR';
            break;
        default:
            trigger_error("Invalid SQL Operator ({$operator})", E_USER_ERROR);
        }

        $this->wheres[] = array($where, $operator);
    }

    /**
     * グループ化を追加する
     *
     * @access public
     * @param object カラムオブジェクト
     */
    function addGroupBy($column)
    {
        $this->group_bys[] = $column;
    }

    /**
     * ソート順を追加する
     *
     * @access public
     * @param object カラムオブジェクト
     * @param string ASC or DESC
     */
    function addOrderBy($column, $sort='ASC')
    {
        switch (strtoupper($sort)) {
        case 'ASC';
        case 'DESC';
            break;
        default:
            trigger_error("Invalid SQL Sort Operator ({$sort})", E_USER_ERROR);
        }
        $this->order_bys[] = array($column, $sort);
    }

    /**
     * SQL(SELECT)文を構築し取得する
     *
     * @access public
     * @return string SQL(SELECT)文
     */
    function getStatementSelect()
    {
        $column_datas = array();
        foreach ($this->column_datas as $column_data) {
            $column_datas[] = $column_data->column->getNameSelect();
        }

        $tables = array();
        foreach ($this->tables as $table) {
            $tables[] = $table->getNameFrom();
        }

        $wheres = array();
        foreach ($this->wheres as $where) {
            $wheres[] = $where[1] . ' ' . $where[0]->getFormat();
        }

        $group_bys = array();
        foreach ($this->group_bys as $column) {
            $group_bys[] = $column->getNameGroupBy();
        }

        $order_bys = array();
        foreach ($this->order_bys as $order_by) {
            $order_bys[] = $order_by[0]->getNameOrderBy() . ' ' . $order_by[1];
        }

        $sql  = "";
        $sql .= "SELECT ";
        $sql .=   implode(",", $column_datas) . " ";
        $sql .= "FROM ";
        $sql .=   implode(",", $tables) . " ";
        if (count($wheres) > 0) {
            $sql .= "WHERE ";
            $sql .=   "1=1 ";
            $sql .=   implode(" ", $wheres) . " ";
        }
        if (count($group_bys) > 0) {
            $sql .= "GROUP BY ";
            $sql .=   implode(",", $group_bys) . " ";
        }
        if (count($order_bys) > 0) {
            $sql .= "ORDER BY ";
            $sql .=   implode(",", $order_bys) . " ";
        }
        return $sql;
    }

    /**
     * SQL(INSERT)文を構築し取得する
     *
     * @access public
     * @return string SQL(INSERT)文
     */
    function getStatementInsert()
    {
        $column_names  = array();
        $column_values = array();
        foreach ($this->column_datas as $column_data) {
            $column_names[]  = $column_data->column->getName();
            $column_values[] = $column_data->getQuoteData();
        }

        $table = $this->tables[0]->getName();

        $sql  = "";
        $sql .= "INSERT INTO {$table} (";
        $sql .=   implode(",", $column_names);
        $sql .= ") VALUES (";
        $sql .=   implode(",", $column_values);
        $sql .= ") ";

        return $sql;
    }

    /**
     * SQL(UPDATE)文を構築し取得する
     *
     * @access public
     * @return string SQL(INSERT)文
     */
    function getStatementUpdate()
    {
        $column_values = array();
        foreach ($this->column_datas as $column_data) {
            $column_values[] = $column_data->column->getName() . ' = ' . $column_data->getQuoteData();
        }

        $table = $this->tables[0]->getName();

        $wheres = array();
        foreach ($this->wheres as $where) {
            $wheres[] = $where[1] . ' ' . $where[0]->getFormat();
        }

        $sql  = "";
        $sql .= "UPDATE {$table} SET ";
        $sql .=   implode(",", $column_values) . " ";
        if (count($wheres) > 0) {
            $sql .= "WHERE ";
            $sql .=   "1=1 ";
            $sql .=   implode(" ", $wheres) . " ";
        }

        return $sql;
    }

    /**
     * SQL(DELETE)文を構築し取得する
     *
     * @access public
     * @return string SQL(DELETE)文
     */
    function getStatementDelete()
    {
        $table = $this->tables[0]->getName();

        $wheres = array();
        foreach ($this->wheres as $where) {
            $wheres[] = $where[1] . ' ' . $where[0]->getFormat();
        }

        $sql  = "";
        $sql .= "DELETE FROM {$table} ";
        if (count($wheres) > 0) {
            $sql .= "WHERE ";
            $sql .=   "1=1 ";
            $sql .=   implode(" ", $wheres) . " ";
        }

        return $sql;
    }
}
