<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBSqlTableConditions.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * SQL条件作成クラス
 * 
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBSqlTableConditions.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_DBSqlTableConditions
{
    /**
     * テーブル別名
     * 
     * @access private
     * @var string
     */
    var $alias = '';
    /**
     * 条件配列
     * 
     * @access private
     * @var array
     */
    var $wheres = array();

    /**
     * テーブル別名をセットする
     *
     * @access public
     * @param string テーブル別名
     */
    function setAlias($alias)
    {
        $this->alias = $alias;
    }

    /**
     * whereパラメータ配列を取得する
     *
     * @access public
     * @return array whereパラメータ配列
     */
    function gets()
    {
        return $this->wheres;
    }

    /**
     * SQL条件文を取得する
     *
     * @access public
     * @param object コネクションオブジェクト
     * @return string SQL条件文
     */
    function getWhere(&$conn)
    {
        $wheres = array();
        $cond   = '';

        foreach ($this->wheres as $where) {
            if ($this->alias) {
                // 別名セット
                $where[0] = "{$this->alias}.{$where[0]}";
            }
            switch ($where[2]) {
            case 'IS NULL':
            case 'IS NOT NULL':
                $wheres[] = "{$where[0]} {$where[2]} ";
                break;
            case 'IN':
            case 'NOT IN':
                $wheres[] = "{$where[0]} {$where[2]} (" . implode(',', array_map(array(&$conn, 'quote'), $where[1])) . ") ";
                break;
            case 'BETWEEN':
                $wheres[] = "{$where[0]} {$where[2]} " . $conn->quote($where[1][0]) . " AND " . $conn->quote($where[1][1]) . " ";
                break;
            default:
                $wheres[] = "{$where[0]} {$where[2]} " . $conn->quote($where[1]) . " ";
            }

            $max = count($wheres) - 1;
            switch ($cond) {
            case 'AND':
                if ($where[3] != 'AND') {
                    $wheres[$max-1] .= ') ';
                    $wheres[$max] = ' ' . $where[3] . ' (' . $wheres[$max];
                } else {
                    $wheres[$max] = ' ' . $where[3] . ' ' . $wheres[$max];
                }
                $cond = 'AND';
                break;
            case 'OR':
                if ($where[3] != 'OR') {
                    $max = count($wheres) - 1;
                    $wheres[$max-1] .= ') ';
                    $wheres[$max] = ' ' . $where[3] . ' (' . $wheres[$max];
                } else {
                    $wheres[$max] = ' ' . $where[3] . ' ' . $wheres[$max];
                }
                $cond = 'OR';
                break;
            default:
                // 初期はAND
                $cond = 'AND';
            }
        }

        if (count($wheres) > 0) {
            return ' (' . implode('', $wheres) . ') ';
        } else {
            return '';
        }
    }

/*
  (AAA AND BBB) OR (CCC AND DDD)
  (AAA OR BBB) AND (CCC OR DDD)

*/

    /**
     * where条件を作成する（等価比較）
     *
     * @access public
     * @param string カラム名
     * @param string 値
     * @param bool true: =, false: <>
     * @param bool true: AND, false: OR
     */
    function addEqual($name, $value='', $equal=true, $and=true)
    {
        $op = ($equal) ? '=' : '<>';
        $this->add($name, $value, $op, $and);
    }

    /**
     * where条件を作成する（greater than比較）
     *
     * @access public
     * @param string カラム名
     * @param string 値
     * @param bool true: =, false: <>
     * @param bool true: AND, false: OR
     */
    function addGt($name, $value='', $equal=true, $and=true)
    {
        $op = ($equal) ? '>=' : '>';
        $this->add($name, $value, $op, $and);
    }

    /**
     * where条件を作成する（less than比較）
     *
     * @access public
     * @param string カラム名
     * @param string 値
     * @param bool true: =, false: <>
     * @param bool true: AND, false: OR
     */
    function addLt($name, $value='', $equal=true, $and=true)
    {
        $op = ($equal) ? '<=' : '<';
        $this->add($name, $value, $op, $and);
    }

    /**
     * where条件を作成する（NULL比較）
     *
     * @access public
     * @param string カラム名
     * @param string 値（ダミー値）
     * @param bool true: IS NULL, false: IS NOT NULL
     * @param bool true: AND, false: OR
     */
    function addNull($name, $value='', $equal=true, $and=true)
    {
        $op = ($equal) ? 'IS NULL' : 'IS NOT NULL';
        $this->add($name, null, $op, $and);
    }

    /**
     * where条件を作成する（LIKE比較）
     *
     * @param string カラム名
     * @param string 値
     * @param bool true: LIKE, false: NOT LIKE
     * @param bool true: AND, false: OR
     */
    function addLike($name, $value='', $equal=true, $and=true)
    {
        $op = ($equal) ? 'LIKE' : 'NOT LIKE';
        $this->add($name, $value, $op, $and);
    }

    /**
     * where条件を作成する（IN比較）
     *
     * @param string カラム名
     * @param array 値
     * @param bool true: IN, false: NOT IN
     * @param bool true: AND, false: OR
     */
    function addIn($name, $value=array(), $equal=true, $and=true)
    {
        $op = ($equal) ? 'IN' : 'NOT IN';
        if (!is_array($value)) {
            trigger_error("[SyL error] `" . __FUNCTION__ . "' method second parameter Array only", E_USER_ERROR);
        }
        $this->add($name, $value, $op, $and);
    }

    /**
     * where条件を作成する（BETWEEN比較）
     *
     * @access public
     * @param string カラム名
     * @param array 値
     * @param bool （ダミー値）
     * @param bool true: AND, false: OR
     */
    function addBetween($name, $value=array(), $equal=true, $and=true)
    {
        if (!is_array($value)) {
            trigger_error("[SyL error] `" . __FUNCTION__ . "' method second parameter Array only", E_USER_ERROR);
        }
        $this->add($name, $value, 'BETWEEN', $and);
    }

    /**
     * where条件を作成する
     *
     * @access private
     * @param string カラム名
     * @param mixed 値
     * @param string 演算子
     * @param bool true: AND, false: OR
     */
    function add($name, $value, $operator, $and)
    {
        $operator = strtoupper($operator);
        $logical  = $and ? 'AND' : 'OR';
        switch ($operator) {
        case '=':
        case '!=':
        case '<>':
        case '>':
        case '>=':
        case '<':
        case '<=':
        case 'NULL':
        case 'IS NULL':
        case 'NOT NULL':
        case 'IS NOT NULL':
        case 'LIKE':
        case 'NOT LIKE':
        case 'IN':
        case 'NOT IN':
        case 'BETWEEN':
            break;
        default:
            trigger_error("[SyL error] Not supported operator ({$operator})", E_USER_ERROR);
        }

        $this->wheres[] = array($name, $value, $operator, $logical);
    }
}
