<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_FilesystemDirectory.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * ファイルシステムのディレクトリクラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_FilesystemDirectory.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_FilesystemDirectory extends SyL_Filesystem
{
    /**
     * 要素の配列
     *
     * @access private
     * @var array
     */
    var $elements = array();
    /**
     * カレントのソート項目名
     *
     * @access private
     * @var string
     */
    var $sort = '';
    /**
     * カレントのソート順
     *
     * @access private
     * @var bool
     */
    var $asc = true;

    /**
     * ディレクトリ判定
     *
     * @access public
     * @return bool true: ディレクトリ、false: ディレクトリ以外
     */
    function isDir()
    {
        return true;
    }

    /**
     * 要素リストを取得する
     *
     * @access public
     * @param string ソート名 name, size, mtime, owner
     * @param bool ソート順 true: 昇順、false: 降順
     * @return array 要素オブジェクト（の配列）
     */
    function getList($sort='name', $asc=true)
    {
        $dirs  = array();
        $files = array();
        foreach (array_keys($this->elements) as $i) {
            if ($this->elements[$i]->isDir()) {
                $dirs[] =& $this->elements[$i];
            } else {
                $files[] =& $this->elements[$i];
            }
        }

        // ソート項目名セット
        $this->sort = $sort;
        // ソート順セット
        $this->asc = $asc;
        // ディレクトリのソート
        usort($dirs, array(&$this, 'sortList'));
        // ファイルのソート
        usort($files, array(&$this, 'sortList'));

        return array_merge($dirs, $files);
    }

    /**
     * 要素リストを取得する
     *
     * @access private
     * @param string ソート名 name, size, mtime, owner
     * @param bool ソート順 true: 昇順、false: 降順
     * @return array 要素オブジェクト（の配列）
     */
    function sortList($a, $b)
    {
        $result = 0;
        switch ($this->sort) {
        case 'name':
            $result = strcmp($a->getName(), $b->getName());
            break;
        case 'size':
            if (($a->getType() == 'file') && ($b->getType() == 'file')) {
                if ($a->getSize() < $b->getSize()) {
                    $result = -1;
                } else if ($a->getSize() > $b->getSize()) {
                    $result = 1;
                } else {
                    $result = 0;
                }
            }
            break;
        case 'mtime':
            if ($a->getStat('mtime') < $b->getStat('mtime')) {
                $result = -1;
            } else if ($a->getStat('mtime') > $b->getStat('mtime')) {
                $result = 1;
            } else {
                $result = 0;
            }
            break;
        case 'owner':
            $result = strcmp($a->getOwner(), $b->getOwner());
            break;
        }

        if ($this->asc) {
            return $result;
        } else {
            if ($result < 0) {
                return 1;
            } else if ($result > 0) {
                return -1;
            } else {
                return 0;
            }
        }
    }

    /**
     * ディレクトリ階層を取得する
     *
     * @access public
     * @param int 検索を行う最大階層
     */
    function createTree($limit_depth=1)
    {
        $this->clear();
        if ($dh = opendir($this->file)) {
            while (($name = readdir($dh)) !== false) {
                if (($name != '.') && ($name != '..')) {
                    switch ($this->file) {
                    case '/':
                    case '\\':
                        $name = "/{$name}";
                        break;
                    default:
                        $name = "{$this->file}/{$name}";
                    }
                    $element =& SyL_Filesystem::factory($name);
                    if ($element->isDir()) {
                        if ($limit_depth === null) {
                            $element->createTree(null);
                        } else if ($limit_depth > 1) {
                            $element->createTree($limit_depth - 1);
                        }
                    }
                    $this->add($element);
                }
            }
            closedir($dh);
        }
    }

    /**
     * 要素の数を返す
     *
     * @access public
     * @return int 要素の数
     */
    function getLength()
    {
        return count($this->elements);
    }

    /**
     * 要素を追加する
     *
     * @access public
     * @param object 要素オブジェクト
     */
    function add($element)
    {
        $this->elements[] = $element;
    }

    /**
     * 要素情報を削除する
     *
     * @access public
     */
    function clear()
    {
        $this->elements = array();
    }

    /**
     * 要素が存在するかチェックする
     *
     * @static
     * @access public 
     * @param string ファイル名
     * @return bool true: 存在する、false: 存在しない
     */
    function exists($file)
    {
        return is_dir($file);
    }

    /**
     * ディレクトリを削除する
     *
     * @static
     * @access public 
     * @param string ファイル名
     * @return bool true: 削除OK、false: 削除エラー
     */
    function remove($file)
    {
        $dir =& SyL_Filesystem::factory($file);
        $dir->createTree(2); // 2階層めまで取得
        foreach ($dir->getList() as $element) {
            if ($element->isDir()) {
               SyL_FilesystemDirectory::remove($element->getName());
            } else {
                SyL_FilesystemFile::remove($element->getName());
            }
        }
        return rmdir($file);
    }

    /**
     * ディレクトリを作成する
     *
     * @static
     * @access public
     * @param string ディレクトリ名（相対パス）
     * @param int モード
     */
    function createDirectory($dir, $mode=0777)
    {
        $dirnames = array();
        while (!is_dir($dir)) {
            if (preg_match('/^[^\\\\\/]*[\\\\\/]$/', $dir)) {
                return false;
            }
            $dirnames[] = basename($dir);
            $dir = dirname($dir);
        }

        foreach (array_reverse($dirnames) as $dirname) {
            $dir .= "/{$dirname}";
            mkdir($dir);
            chmod($dir, $mode);

        }
    }

    /**
     * ディレクトリ内ファイルを取得する
     *
     * @static
     * @access public
     * @param string ディレクトリ
     * @param bool true: ディレクトリ取得、false: ディレクトリ取得しない
     * @param bool true: ファイル取得、false: ファイル取得しない
     * @param int 検索を行う最大階層
     * @return array ディレクトリ内ファイル
     */
    function getFiles($file, $is_get_dir=true, $is_get_file=true, $limit_depth=1)
    {
        $files = array();
        if (is_dir($file)) {
            if ($dh = opendir($file)) {
                while (($name = readdir($dh)) !== false) {
                    if (($name != '.') && ($name != '..')) {
                        switch ($file) {
                        case '/':
                        case '\\':
                            $name = "/{$name}";
                            break;
                        default:
                            $name = "{$file}/{$name}";
                        }
                        $element =& SyL_Filesystem::factory($name);
                        if ($element->isDir()) {
                            if ($limit_depth === null) {
                                $files = array_merge($files, SyL_FilesystemDirectory::getFiles($element->getName(), $is_get_dir, $is_get_file, null));
                            } else if ($limit_depth > 1) {
                                $files = array_merge($files, SyL_FilesystemDirectory::getFiles($element->getName(), $is_get_dir, $is_get_file, $limit_depth - 1));
                            }
                            if ($is_get_dir) {
                                $files[] = $element;
                            }
                        } else {
                            if ($is_get_file) {
                                $files[] = $element;
                            }
                        }
                    }
                }
                closedir($dh);
            }
        }
        return $files;
    }

    /**
     * 要素を検索する
     *
     * @static
     * @access public
     * @param string ディレクトリ
     * @param string 検索ファイル名
     * @param bool true: ディレクトリ取得、false: ディレクトリ取得しない
     * @param bool true: ファイル取得、false: ファイル取得しない
     * @param int 検索を行う最大階層
     * @return array ディレクトリ内ファイル
     */
    function searchFiles($file, $word, $is_get_dir=true, $is_get_file=true, $limit_depth=1)
    {
        $files = array();
        foreach (SyL_FilesystemDirectory::getFiles($file, $is_get_dir, $is_get_file, $limit_depth) as $file) {
            if (strpos($file->getName(true), $word) !== false) {
                $files[] = $file;
            }
        }
        return $files;
    }

    /**
     * サイズ（ディレクトリ内の全ファイルサイズ）を取得する
     *
     * @static
     * @access public
     * @param string ディレクトリ
     * @param bool true: ディレクトリのサイズ含む、fase: ディレクトリのサイズ含まない
     * @return int サイズ
     */
    function getTotalSize($file, $is_dir_size=false)
    {
        $size = 0;
        foreach (SyL_FilesystemDirectory::getFiles($file, $is_dir_size, true, null) as $file) {
            $size += $file->getSize();
        }
        return $size;
    }
}
