<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_HttpClientRequest.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * 汎用プロパティクラス
 */
require_once dirname(__FILE__) . '/../Util/SyL_UtilProperty.php';

/**
 * HTTPクライアントリクエストクラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_HttpClientRequest.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_HttpClientRequest
{
    /**
     * HTTPリクエスト対象URL
     *
     * @access private
     * @var string
     */
    var $url = '';
    /**
     * HTTPリクエストメソッド
     *
     * @access private
     * @var string
     */
    var $method = '';
    /**
     * リクエストヘッダ格納オブジェクト
     *
     * @access private
     * @var string
     */
    var $headers = null;
    /**
     * リクエストパラメータ格納オブジェクト
     *
     * @access private
     * @var string
     */
    var $parameters = null;
    /**
     * ユーザーエージェント
     *
     * @access private
     * var string
     */
    var $useragent = 'SyL Framework/HttpClient';
    /**
     * 改行コード
     *
     * @access private
     * @var string
     */
    var $eol = "\r\n";

    /**
     * コンストラクタ
     * 
     * @access public
     * @param string HTTPリクエスト対象URL
     * @param string HTTPリクエストメソッド
     */
    function SyL_HttpClientRequest($url='/', $method='GET')
    {
        $this->url    = $url;
        $this->method = strtoupper($method);

        $this->headers    = new SyL_UtilProperty();
        $this->parameters = new SyL_UtilProperty();

        if ($this->method == 'POST') {
            $this->setHeader('content-type', 'application/x-www-form-urlencoded');
        }

        $this->setHeader('connection', 'Close');
        $this->setUserAgent($this->useragent);
    }

    /**
     * ユーザーエージェントをセット
     * 
     * @access public
     * @param string ユーザーエージェント
     */
    function setUserAgent($value)
    {
        $this->headers->set('user-agent', $value);
    }

    /**
     * Basic認証のID, パスワードをセット
     *
     * @access public
     * @param string ユーザー名
     * @param string パスワード
     */
    function setAuthorization($username, $password)
    {
        $this->headers->set('authorization', 'Basic ' . base64_encode("{$username}:{$password}"));
    }

    /**
     * ヘッダをセット
     * 
     * @access public
     * @param string パラメータ名
     * @param string パラメータ値
     */
    function setHeader($name, $value)
    {
        $this->headers->set(strtolower($name), $value);
    }

    /**
     * ヘッダの確認
     * 
     * @access public
     * @param string ヘッダ名
     * @return bool true: ヘッダあり、false: ヘッダ無し
     */
    function isHeader($name)
    {
        return $this->headers->is(strtolower($name));
    }

    /**
     * 全てのヘッダを取得
     * 
     * @access public
     * @return array 全てのヘッダ
     */
    function getHeaders()
    {
        $headers = array();
        foreach ($this->headers->gets() as $name => $value) {
            $name = preg_replace('/(^.|\-.)/e', 'strtoupper("$1")', $name);
            $headers[$name] = $value;
        }
        return $headers;
    }

    /**
     * パラメータをセット
     * 
     * @access public
     * @param string パラメータ名
     * @param string パラメータ値
     */
    function set($name, $value)
    {
        $this->parameters->set($name, $value);
    }

    /**
     * 複数パラメータをセット
     * 
     * @access public
     * @param array パラメータ値
     */
    function sets($values)
    {
        $this->parameters->sets($values);
    }

    /**
     * パラメータをリクエスト形式に変換
     * 
     * @access protected
     * @return string パラメータ
     */
    function build()
    {
        $parameter = array();
        foreach ($this->parameters->gets() as $name => $value) {
            $parameter[] = urlencode($name) . '=' .  urlencode($value);
        }
        return implode('&', $parameter);
    }

    /**
     * リクエストヘッダを取得する
     * 
     * @access public
     * @return string リクエストヘッダ
     */
    function getSource()
    {
        $parameter = $this->build();
        $url = $this->url;
        switch ($this->method) {
        case 'GET':
            if ($parameter) {
                 $url .= '?' . $parameter;
            }
            break;
        case 'POST':
            $this->setHeader('content-length', strlen($parameter));
            break;
        }

        $source  = "";
        $source .= "{$this->method} {$url} HTTP/1.1" . $this->eol;
        foreach ($this->getHeaders() as $name => $value) {
            $source .= "{$name}: $value" . $this->eol;
        }
        $source .= $this->eol;
        if ($this->method == 'POST') {
            $source .= $parameter;
        }
        return $source;
    }
}
