/**
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    k.watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id:$
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

if (!SyL) var SyL = {};
if (!SyL.Validation) SyL.Validation = {};

/**
 * バリデーション実行クラス
 */
SyL.Validation.Validator = {
  /**
   * 未入力チェックを行う
   */
  isRequire: function(value, options)
  {
    if (options && options['trim']) {
      value = value.replace(/^\s+|\s+$/g, "");
    }
    return ((value != null) && (value != ""));
  },
  /**
   * 比較チェックを行う
   */
  isCompare: function(value, options)
  {
    var target  = options['target'];
    var compare = (options && options['compare']) ? options['compare'] : '==';
    switch (compare) {
    case '===':
    case '==':  return (value == target);
    case '!==':
    case '!=':  return (value != target);
    case '<=':  return (value <= target);
    case '>=':  return (value >= target);
    case '<':   return (value <  target);
    case '>':   return (value >  target);
    default:    return false;
    }
  },
  /**
   * 半角数値チェックを行う
   */
  isNumeric: function(value, options)
  {
    var regex;
    if (options && options['dot']) {
        regex = /^[\-\+]?[0-9]+(\.[0-9]+)?$/;
    } else {
        regex = /^[\-\+]?[0-9]+$/;
    }

    if (!value.match(regex)) {
      return false;
    }

    if (options) {
      if (!isNaN(options['min'])) {
        if (parseFloat(options['min'], 10) > parseFloat(value, 10)) {
          options['min_error'] = true;
          return false;
        }
      }
      if (!isNaN(options['max'])) {
        if (parseFloat(value, 10) > parseFloat(options['max'], 10)) {
          options['max_error'] = true;
          return false;
        }
      }
    }

    return true;
  },
  /**
   * 正規表現チェックを行う
   */
  isRegex: function(value, options)
  {
    var regex = options['format'];
    if (!(regex instanceof RegExp)) {
      if (typeof(regex) == 'string') {
        regex = new RegExp(regex);
      } else {
        return false;
      }
    }

    return value.match(regex);
  },
  /**
   * バイト数チェックを行う
   */
  isLength: function(value, options)
  {
    var bytes = SyL.Validation.Validator.getBytes(value);
    if (options['max']) {
      if (bytes > options['max']) {
        options['max_error'] = true;
        return false;
      }
    }
    if (options['min']) {
      if (options['min'] > bytes) {
        options['min_error'] = true;
        return false;
      }
    }
    return true;
  },
  /**
   * 日付の妥当性チェック
   */
  isDate: function (value, options)
  {
    if (value.match(/^([0-9]{4})[\-\/]?([0-1][0-9])[\-\/]?([0-9]{2})\s?([0-1][0-9]|2[0-3]):?([0-5][0-9]):?([0-5][0-9])$/)) {
      yyyy = parseInt(RegExp.$1, 10);
      mm   = parseInt(RegExp.$2, 10);
      dd   = parseInt(RegExp.$3, 10);
      hh   = parseInt(RegExp.$4, 10);
      mi   = parseInt(RegExp.$5, 10);
      ss   = parseInt(RegExp.$6, 10);
    } else if (value.match(/^([0-9]{4})[\-\/]?([0-1][0-9])[\-\/]?([0-9]{2})$/)) {
      yyyy = parseInt(RegExp.$1, 10);
      mm   = parseInt(RegExp.$2, 10);
      dd   = parseInt(RegExp.$3, 10);
    } else {
      return false;
    }

    var date = new Date(yyyy, mm-1, dd); 
    if (!date || (yyyy != date.getFullYear()) || (mm != (date.getMonth() + 1))) {
      return false;
    }

    if (options) {
      var dt_date;
      if (options['date'] && options['date'].match(/^([0-9]{4})[\-\/]?([0-1][0-9])[\-\/]?([0-9]{2})$/)) {
        dt_date = new Date(parseInt(RegExp.$1, 10), parseInt(RegExp.$2, 10)-1, parseInt(RegExp.$3, 10)); 
        if (options['past']) {
          if (dt_date.getTime() < date.getTime()) {
            return false;
          }
        }
        if (options['future']) {
          if (dt_date.getTime() > date.getTime()) {
            return false;
          }
        }
      }

      if (options['min'] && options['min'].match(/^([0-9]{4})[\-\/]?([0-1][0-9])[\-\/]?([0-9]{2})$/)) {
        dt_date = new Date(parseInt(RegExp.$1, 10), parseInt(RegExp.$2, 10)-1, parseInt(RegExp.$3, 10)); 
        if (dt_date.getTime() > date.getTime()) {
          options['min_error'] = true;
          return false;
        }
      }

      if (options['max'] && options['max'].match(/^([0-9]{4})[\-\/]?([0-1][0-9])[\-\/]?([0-9]{2})$/)) {
        dt_date = new Date(parseInt(RegExp.$1, 10), parseInt(RegExp.$2, 10)-1, parseInt(RegExp.$3, 10)); 
        if (dt_date.getTime() < date.getTime()) {
          options['max_error'] = true;
          return false;
        }
      }
    }

    return true;
  },
  /**
   * バイト数を取得する
   */
  getBytes: function(value)
  {
    var bytes = 0;

    for (var i=0; i<value.length; i++) {
      // c < 0x7E
      //   ASCIIの範囲
      bytes += (value.charCodeAt(i) <= 0x7E) ? 1 : 2;
    }
    return bytes;
  }
}


/**
 * フォームバリデーションクラス
 */
SyL.Validation.Validation = function (form)
{
  this.initialize(form);
}

SyL.Validation.Validation.prototype = {
  initialize: function(form)
  {
    /**
     * 未入力（未選択）チェックを行う
     */
    this.isRequire = function(name, errorMessage, options)
    {
      return validate(SyL.Validation.Validator.isRequire, name, options, errorMessage);
    };
    /**
     * 比較チェックを行う
     */
    this.isCompare = function(name, errorMessage, options)
    {
      if (options) {
        if (options['element']) {
          options['target'] = getValue(options['element']);
        } else {
          options['target'] = options['value'];
        }
      } else {
        return errorMessage;
      }
      return validate(SyL.Validation.Validator.isCompare, name, options, errorMessage);
    };
    /**
     * 半角数値チェックを行う
     */
    this.isNumeric = function(name, errorMessage, options)
    {
      if (this.isRequire(name, null, {'min_valids': '0', 'max_valids': '0'})) {
        return '';
      }
      return validate(SyL.Validation.Validator.isNumeric, name, options, errorMessage);
    };
    /**
     * 正規表現チェックを行う
     */
    this.isRegex = function(name, errorMessage, options)
    {
      if (this.isRequire(name, null, {'min_valids': '0', 'max_valids': '0'})) {
        return '';
      }
      if (!options || !options['format']) {
        return errorMessage;
      }
      return validate(SyL.Validation.Validator.isRegex, name, options, errorMessage);
    };
    /**
     * バイト数チェックを行う
     */
    this.isLength = function(name, errorMessage, options)
    {
      if (this.isRequire(name, null, {'min_valids': '0', 'max_valids': '0'})) {
        return '';
      }
      if (!options || !options['max']) {
        return errorMessage;
      }
      return validate(SyL.Validation.Validator.isLength, name, options, errorMessage);
    };
    /**
     * 日付の妥当性チェック
     */
    this.isDate = function (name, errorMessage, options)
    {
      if (this.isRequire(name, null, {'min_valids': '0', 'max_valids': '0'})) {
        return '';
      }
      return validate(SyL.Validation.Validator.isDate, name, options, errorMessage);
    };
    /**
     * 検証を実行する
     */
    function validate(func, name, options, errorMessage)
    {
      if (!options) {
        options = {};
      }
      var value = getValue(name);
      var valid = true;
      if (value instanceof Array) {
        var min_valids = (options['min_valids']) ? options['min_valids'] : 0;
        var max_valids = (options['max_valids']) ? options['max_valids'] : 0;

        var ok = 0;
        var len = value.length;
        for (var i=0; i<len; i++) {
          if (func(value, options)) {
            ok++;
          }
        }

        if (max_valids == 0) {
          max_valids = len;
          if (min_valids == 0) {
            min_valids = len;
          }
        }
        valid = ((max_valids >= ok) && (min_valids <= ok));
      } else {
        valid = func(value, options);
        if (!valid) {
          if (options['min_error'] && options['min_error_message']) {
            errorMessage = options['min_error_message'];
          } else if (options['max_error'] && options['max_error_message']) {
            errorMessage = options['max_error_message'];
          }
        }
      }

      return valid ? '' : errorMessage;
    };
    /**
     * 要素値を取得する
     */
    function getValue(name)
    {
      var n = form.elements[name];
      var value = null;
      var type = form.elements[name].type;
      if (form.elements[name][0] && form.elements[name][0].type) {
        type = form.elements[name][0].type;
      }

      switch (type) {
      case "select-one":
        var i = n.selectedIndex;
        if (i && i >= 0) {
          value = n.options[i].value;
        }
        break;
      case "select-multiple":
        value = [];
        for (var i=0; i<n.options.length; i++) {
          if (n.options[i].selected) {
            value.push(n.options[i].value);
          }
        }
        break;

      case "text":
      case "textarea":
      case "hidden":
      case "password":
        value = n.value;
        break;
      case "file":
        var len = n.length;
        if (len) {
          value = [];
          for (var i=0; i<len; i++) {
            value.push(n[i].value);
          }
        } else {
          value = n.value;
        }
        break;

      case "radio":
        var len = n.length;
        if (len) {
          for (var i=0; i<len; i++) {
            if (n[i].checked) {
              value = n[i].value;
              break;
            }
          }
        } else {
          if (n.checked) {
            value = n.value;
          }
        }
        break;
      case "checkbox":
        var len = n.length;
        if (len) {
          value = [];
          for (var i=0; i<len; i++) {
            if (n[i].checked) {
              value.push(n[i].value);
            }
          }
        } else {
          if (n.checked) {
            value = n.value;
          }
        }
        break;
      }

      return value;
    };
  }
}


/**
 * バリデーションエラー格納クラス
 */
SyL.Validation.Errors = function()
{
  // エラーメッセージヘッダ
  this.errorMessageHeader = '';
  // エラーメッセージフッタ
  this.errorMessageFooter = '';

  this.initialize();
}

SyL.Validation.Errors.prototype = {
  initialize: function()
  {
    var displayAllMessage = true;
    var customErrorCallback = null;

    var errorMessages = {};

    this.setDisplayAllMessage = function(flag)
    {
      displayAllMessage = !!flag;
    };
    this.setCustomErrorCallback = function(errorCallback)
    {
      customErrorCallback = errorCallback;
    };
    this.isError = function(name)
    {
      if (name) {
        return !!errorMessages[name];
      } else {
        return (this.getErrorFirstElementName() != null);
      }
    };
    this.setErrorMessage = function(name, message)
    {
      errorMessages[name] = message;
    };
    this.getErrorMessage = function(name)
    {
      return (errorMessages[name] ? errorMessages[name] : null);
    };
    this.getErrorFirstElementName = function()
    {
      for (var name in errorMessages) {
        return name;
      }
      return null;
    };
    this.getErrorMessages = function()
    {
      var messages = [];
      for (var name in errorMessages) {
        messages.push(errorMessages[name]);
      }
      return messages;
    };

    this.raiseErrorMessage = function ()
    {
      if (!this.isError()) {
        return;
      }

      if (customErrorCallback) {
        customErrorCallback(errorMessages);
      } else {
        if (displayAllMessage) {
          var displayMessage = this.getErrorMessages().join("\n");
          displayMessage = this.errorMessageHeader + displayMessage + this.errorMessageFooter;
        } else {
          name = this.getErrorFirstElementName();
          displayMessage = this.getErrorMessage(name);
        }
        alert(displayMessage);
      }
    };

    this.focusElement = function(form, name)
    {
      if (typeof(form) != "object") {
        throw "[SyL.Validation.Errors.focusElement] Invalid method parameter (" + typeof(form) + ")";
      }
      if (!name) {
        name = this.getErrorFirstElementName();
      }
      if (!form.elements[name]) {
        return;
      }

      if (form.elements[name][0]) {
        form.elements[name][0].focus();
      } else {
        form.elements[name].focus();
      }

      switch (form.elements[name].type) {
      case "text":
      case "textarea":
      case "password":
      case "file":
        form.elements[name].select();
        break;
      }
    };
  }
}
