<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_MailMessage.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * メールメッセージクラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_MailMessage.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_MailMessage
{
    /**
     * 送信元
     * 
     * @access private
     * @var string
     */
    var $from = array();
    /**
     * 返信先メールアドレス
     * 
     * @access private
     * @var string
     */
    var $reply_to = '';
    /**
     * 送信先
     * 
     * @access private
     * @var array
     */
    var $to = array();
    /**
     * CCメールアドレス
     * 
     * @access private
     * @var array
     */
    var $cc = array();
    /**
     * BCCメールアドレス
     * 
     * @access private
     * @var array
     */
    var $bcc = array();
    /**
     * 件名
     * 
     * @access private
     * @var string
     */
    var $subject = '';
    /**
     * MIMEバージョン
     * 
     * @access protected
     * @var string
     */
    var $mime_version = '1.0';
    /**
     * 本文
     *
     * [body]              - 本文
     * [name]              - 添付ファイル名
     * [file]              - 添付ファイル
     * [type]              - Content-Type
     * [transfer_encoding] - Content-Transfer-Encoding
     * 
     * @access protected
     * @var array
     */
    var $attachments = array();
    /**
     * エンコード方式
     *
     * ※現在Base64のみ対応
     * 
     * B: base64
     * Q: Quoted-Printable
     * 
     * @access protected
     * @var string
     */
    var $transfer_encoding = 'B';
    /**
     * X-Mailerヘッダー
     * 
     * @access protected
     * @var string
     */
    var $x_mailer = 'SyL Mail 1.0';
    /**
     * その他ヘッダ
     * 
     * @access protected
     * @var array
     */
    var $headers = array();
    /**
     * メール内容のエンコード
     * 
     * @access protected
     * @var string
     */
    var $mail_encode = '';
    /**
     * プログラム側のエンコード
     * 
     * @access protected
     * @var string
     */
    var $internal_encode = '';
    /**
     * 言語
     * 
     * @access protected
     * @var string
     */
    var $language = '';
    /**
     * 改行
     * 
     * @access protected
     * @var string
     */
    var $eol = "\r\n";

    /**
     * 送信元アドレスをセット
     *
     * @access public
     * @param string 送信元アドレス
     * @param string 送信元名
     */
    function setFrom($from, $name='')
    {
        $this->from = array($from, $name);
        if ($this->reply_to == '') {
            $this->reply_to = $from;
        }
    }

    /**
     * 送信元アドレスを取得
     *
     * @access public
     * @return array 送信元アドレス
     */
    function getFrom()
    {
        return $this->from;
    }

    /**
     * 送信先アドレスをセット
     * 
     * @access public
     * @param mixed 送信先アドレス
     * @param string 送信先名
     */
    function addTo($to, $name='')
    {
        $this->to[] = array($to, $name);
    }

    /**
     * 送信先アドレスを取得
     *
     * @access public
     * @return array 送信元アドレス
     */
    function getTo()
    {
        return $this->to;
    }

    /**
     * 送信先アドレス(CC)をセット
     * 
     * @access public
     * @param mixed 送信先アドレス(CC)
     * @param string 送信先名(CC)
     */
    function addCc($cc, $name='')
    {
        $this->cc[] = array($cc, $name);
    }

    /**
     * 送信先アドレス(CC)を取得
     *
     * @access public
     * @return array 送信元アドレス(CC)
     */
    function getCc()
    {
        return $this->cc;
    }

    /**
     * 送信先アドレス(BCC)をセット
     * 
     * @access public
     * @param mixed 送信先アドレス(BCC)
     * @param string 送信先名(BCC)
     */
    function addBcc($bcc, $name='')
    {
        $this->bcc[] = array($bcc, $name);
    }

    /**
     * 送信先アドレス(BCC)を取得
     *
     * @access public
     * @return array 送信元アドレス(BCC)
     */
    function getBcc()
    {
        return $this->bcc;
    }

    /**
     * 件名をセット
     * 
     * @access public
     * @param string 件名
     */
    function setSubject($subject)
    {
        $this->subject = $subject;
    }

    /**
     * 件名を取得
     * 
     * @access public
     * @return string 件名
     */
    function getSubject()
    {
        return $this->subject;
    }

    /**
     * 本文をセット
     * 
     * @access public
     * @param string 本文
     */
    function setBody($body)
    {
        $this->attachments[] = array(
          'body'              => $body,
          'type'              => 'text/plain; charset="' . $this->mail_encode . '"',
          'transfer_encoding' => '7bit',
          'name'              => null,
          'file'              => null
        );
    }

    /**
     * 本文を取得
     * 
     * @access public
     * @param string 本文
     */
    function getBody()
    {
        return $this->attachments;
    }

    /**
     * メールエンコードをセットする
     *
     * @access public
     * @param string メールエンコード
     */
    function setMailEncode($mail_encode)
    {
        $this->mail_encode = $mail_encode;
    }

    /**
     * 内部パラメータエンコードをセットする
     *
     * @access public
     * @param string 内部パラメータエンコード
     */
    function setInternalEncode($internal_encode)
    {
        $this->internal_encode = $internal_encode;
    }

    /**
     * 言語をセットする
     *
     * @access public
     * @param string 言語
     */
    function setLanguage($language)
    {
        $this->language = $language;
    }

    /**
     * その他のメールヘッダをセット
     * 
     * @access public
     * @param string ヘッダのキー
     * @param string ヘッダの値
     */
    function addHeader($key, $value)
    {
        $this->headers[$key] = $value;
    }

    /**
     * その他のメールヘッダを取得
     * 
     * @access public
     * @return array その他のメールヘッダ
     */
    function getHeader($key)
    {
        return isset($this->headers[$key]) ? $this->headers[$key] : null;
    }

    /**
     * 添付ファイルをセット
     * 
     * @access public
     * @param string ファイルパス（ファイル名まで含む）
     */
    function addFile($path, $contents_type='application/octet-stream')
    {
        // ファイル名取得
        $name = basename($path);
        // 添付ファイル取得
        $contents = file_get_contents($path);
        switch ($this->transfer_encoding) {
        case 'B':
            $contents = chunk_split(base64_encode($contents));
            $transfer_encoding = 'Base64';
            break;
        default:
            trigger_error("[SyL error] Content-Transfer-Encoding not supported ({$this->transfer_encoding})", E_USER_ERROR);
            break;
        }

        // 添付ファイル情報セット
        $this->attachments[] = array(
          'body'              => null,
          'type'              => $contents_type,
          'transfer_encoding' => $transfer_encoding,
          'name'              => $this->convertEncoding($name, true),
          'file'              => $contents
        );
    }

    /**
     * RCPT_TO用メールアドレスリストを取得する
     * 
     * @access public
     * @return array RCPT_TO用メールアドレスリスト
     */
    function getRcptTo()
    {
        $rcptto = array();
        foreach ($this->to as $to) {
            if (array_search($to[0], $rcptto) === false) {
                $rcptto[] = $to[0];
            }
        }
        foreach ($this->cc as $cc) {
            if (array_search($cc[0], $rcptto) === false) {
                $rcptto[] = $cc[0];
            }
        }
        foreach ($this->bcc as $bcc) {
            if (array_search($bcc[0], $rcptto) === false) {
                $rcptto[] = $bcc[0];
            }
        }
        return $rcptto;
    }

    /**
     * パラメータをメール送信用文字列にエンコードする
     *
     * @access public
     * @param string エンコード対象文字列
     * @param bool MIMEエンコードフラグ
     * @return エンコード後文字列
     */
    function convertEncoding($parameter, $mime_encode=false)
    {
        $parameter = mb_convert_encoding(mb_convert_kana($parameter, 'KV', $this->internal_encode), $this->mail_encode, $this->internal_encode);
        if ($mime_encode) {
            $language = mb_language();
            mb_language($this->language);
            $internal_encode = mb_internal_encoding();
            mb_internal_encoding($this->mail_encode);
            $parameter = mb_encode_mimeheader($parameter, $this->mail_encode, $this->transfer_encoding);
            mb_internal_encoding($internal_encode);
            mb_language($language);
        }
        return $parameter;
    }

    /**
     * メールアドレスをメール送信用文字列にエンコーディング
     * 
     * @access public
     * @param string メールアドレス
     * @param string メールアドレスの名前
     * @return string エンコーディングのメールアドレス
     */
    function convertEncodingAddress($address, $name='')
    {
        if ($name) {
            return '"' . $this->convertEncoding($name, true) . '" <' . $address . '>';
        } else {
            return $address;
        }
    }

    /**
     * メール送信用文字列をパラメータにデコードする
     *
     * @access public
     * @param string デコード対象文字列
     * @param bool MIMEエンコードフラグ
     * @return デコード後文字列
     */
    function convertDecoding($parameter, $mime_encode=false)
    {
        if ($mime_encode) {
            $language = mb_language();
            mb_language($this->language);
            $internal_encode = mb_internal_encoding();
            mb_internal_encoding($this->mail_encode);
            $parameter = mb_decode_mimeheader($parameter);
            mb_internal_encoding($internal_encode);
            mb_language($language);
        }
        return mb_convert_encoding($parameter, $this->internal_encode, $this->mail_encode);
    }

    /**
     * メール送信用文字列をメールアドレスにデコーディング
     * 
     * @access public
     * @param string メールアドレス
     * @return array メールアドレスとメールアドレスの名前の配列
     */
    function convertDecodingAddress($address)
    {
        if (preg_match('/^(.*)[ ]*<(.+@.+)>$/', $address, $matches)) {
            switch (count($matches)) {
            case 3:
                $matches[1] = trim($matches[1]);
                if (preg_match('/^"(.+)"$/', $matches[1], $matches1)) {
                    $matches[1] = $matches1[1];
                }
                return array($matches[2], $this->convertDecoding(trim($matches[1]), true));
            case 2: return array($matches[2], '');
            }
        } else {
            return array($address, '');
        }
    }

    /**
     * メールメッセージを取得
     *
     * @access public
     * @param bool SMTP用本文エスケープ
     * @return string メールヘッダを含む全文取得
     */
    function getMessage($body_escape=false)
    {
        // バウンダリ
        $boundary = '--' . md5(uniqid(rand()));

        // From必須
        if (count($this->from) != 2) {
            trigger_error("[SyL error] From Address not found", E_USER_ERROR);
        }
        // To必須
        if (count($this->to) == 0) {
            trigger_error("[SyL error] To Address not found", E_USER_ERROR);
        }

        // メールヘッダ
        $data  = '';
        $data .= "From: "     . $this->convertEncodingAddress($this->from[0], $this->from[1]) . $this->eol;
        $data .= "Subject: "  . $this->convertEncoding($this->subject, true) . $this->eol;
        for ($i=0; $i<count($this->to); $i++) {
            $data .= "To: " . $this->convertEncodingAddress($this->to[$i][0], $this->to[$i][1]) . $this->eol;
        }
        for ($i=0; $i<count($this->cc); $i++) {
            $data .= "Cc: " . $this->convertEncodingAddress($this->cc[$i][0], $this->cc[$i][1]) . $this->eol;
        }
        for ($i=0; $i<count($this->bcc); $i++) {
            $data .= "Bcc: " . $this->convertEncodingAddress($this->bcc[$i][0], $this->bcc[$i][1]) . $this->eol;
        }
        $data .= "Reply-To: " . $this->reply_to . $this->eol;
        $data .= "X-Mailer: " . $this->x_mailer . $this->eol;
        $data .= "MIME-Version: " . $this->mime_version . $this->eol;
        foreach ($this->headers as $name => $value) {
            if (is_array($value)) {
                $value = implode(', ', $value);
            }
            $data .= "{$name}: {$value}" . $this->eol;
        }

        switch (count($this->attachments)) {
        case 0:
            break;
        case 1:
            $data .= $this->getAttachmentData($this->attachments[0], $body_escape);
            break;
        default:
            $data .= "Content-Type: multipart/mixed; boundary=\"$boundary\"" . $this->eol;
            $data .= $this->eol;
            for ($i=0; $i<count($this->attachments); $i++) {
                $data .= '--' . $boundary . $this->eol;
                $data .= $this->getAttachmentData($this->attachments[$i], $body_escape);
            }
            $data .= '--' . $boundary . "--" . $this->eol;
        }
        return $data;
    }

    /**
     * メールメッセージをセット
     *
     * @access public
     * @param string メールメッセージ
     */
    function setMessage($data)
    {
        $datas = explode($this->eol . $this->eol, $data, 2);
        // ヘッダを配列に変換
        $headers = $this->getHeaders($datas[0]);

        // メールエンコーディング取得
        if (isset($headers['content-type'])) {
            if (preg_match('/charset="?([^"]+)"?/is', $headers['content-type'][0], $matches)) {
                $this->mail_encode = trim($matches[1]);
            }
        }

        // メールヘッダ取得
        foreach ($headers as $name => $header) {
            switch ($name) {
            case 'from':     $this->from     = $this->convertDecodingAddress($header[0]); break;
            case 'to':       $this->to[]     = $this->convertDecodingAddress($header[0]); break;
            case 'reply-to': $this->reply_to = $this->convertDecodingAddress($header[0]); break;
            case 'subject':  $this->subject  = $this->convertDecoding($header[0], true);  break;
            case 'date':       $this->headers['Date']       = $header[0]; break;
            case 'message-id': $this->headers['Message-ID'] = $header[0]; break;
            case 'mime-version': $this->mime_version = $header[0]; break;
            case 'x-mailer':     $this->x_mailer     = $header[0]; break;
            case 'content-transfer-encoding':
                switch ($header[0]) {
                case 'base64':           $this->transfer_encoding = 'B'; break;
                case 'quoted-printable': $this->transfer_encoding = 'Q'; break;
                }
                break;
            }
        }

        // メッセージ取得
        if (isset($datas[1])) {
            if (!isset($headers['content-type']) || (strpos($headers['content-type'][0], 'text/') !== false)) {
                // テキストパート
                $this->attachments[] = array(
                  'body'              => $datas[1],
                  'type'              => $headers['content-type'][0],
                  'transfer_encoding' => isset($headers['content-transfer-encoding'][0]) ? $headers['content-transfer-encoding'][0] : ''
                );
            } else if (strpos($headers['content-type'][0], 'multipart/') !== false) {
                // マルチパート
                if (preg_match('/boundary="?([^"]+)"? ?/', $headers['content-type'][0], $matches)) {
                    $boundary = "--{$matches[1]}";
                    // マルチパートの終了以下を削除
                    $pos = strpos($datas[1], "{$this->eol}{$boundary}--{$this->eol}");
                    if ($pos !== false) {
                        $datas[1] = substr($datas[1], 0, $pos);
                    }
                    // マルチパート分割
                    $messages = explode("{$this->eol}{$boundary}{$this->eol}", $datas[1]);

                    // 先頭部分削除
                    array_shift($messages);
                    foreach ($messages as $message) {
                        $messages2 = explode($this->eol . $this->eol, $message, 2);
                        switch (count($messages2)) {
                        case 2:
                            // ヘッダあり
                            $headers_sub = $this->getHeaders($messages2[0]);
                            if (isset($headers_sub['content-type'])) {
                                if (preg_match('/charset="?([^"]+)"?/is', $headers_sub['content-type'][0], $matches)) {
                                    $this->mail_encode = trim($matches[1]);
                                }
                            }
                            $this->transfer_encoding = '7bit';
                            if (isset($headers_sub['content-transfer-encoding'])) {
                                switch ($headers_sub['content-transfer-encoding'][0]) {
                                case 'base64':
                                    $this->transfer_encoding = 'B';
                                    $messages2[1] = base64_decode($messages2[1]);
                                    break;
                                }
                            }
                            $filename = null;
                            if (isset($headers_sub['content-disposition'])) {
                                if (preg_match('/filename="?([^"]+)"?/is', $headers_sub['content-disposition'][0], $matches)) {
                                    $filename = $this->convertDecoding(trim($matches[1]), true);
                                }
                            }

                            if ($this->transfer_encoding == '7bit') {
                                // テキスト系
                                $this->attachments[] = array(
                                  'body'              => $this->convertDecoding($messages2[1]),
                                  'type'              => isset($headers_sub['content-type'][0]) ? $headers_sub['content-type'][0] : '',
                                  'transfer_encoding' => isset($headers_sub['content-transfer-encoding'][0]) ? $headers_sub['content-transfer-encoding'][0] : '',
                                  'name'              => null,
                                  'file'              => null
                                );
                            } else {
                                // その他添付系
                                $this->attachments[] = array(
                                  'body'              => null,
                                  'type'              => isset($headers_sub['content-type'][0]) ? $headers_sub['content-type'][0] : '',
                                  'transfer_encoding' => isset($headers_sub['content-transfer-encoding'][0]) ? $headers_sub['content-transfer-encoding'][0] : '',
                                  'name'              => $filename,
                                  'file'              => $messages2[1]
                                );
                            }
                            break;
                        case 1:
                            // ヘッダなし。強制的に本文に
                            $this->attachments[] = array(
                              'body'              => $messages2[0],
                              'type'              => null,
                              'transfer_encoding' => null,
                              'name'              => null,
                              'file'              => null
                            );
                            break;
                        }
                    }
                }
            }
        }
    }

    /**
     * 本文、添付生データを取得する
     *
     * @access private
     * @param array 元データ配列
     * @param bool SMTP用本文エスケープ
     * @return string 本文、添付生データ
     */
    function getAttachmentData(&$attachment, $body_escape)
    {
        $data  = '';
        $data .= "Content-Type: {$attachment['type']}" . $this->eol;
        $data .= "Content-Transfer-Encoding: {$attachment['transfer_encoding']}" . $this->eol;
        if (isset($attachment['body'])) {
            // テキスト
            $body = $body_escape ? preg_replace('/(\r\n|\n|\r)\.(\r\n|\n|\r)/s', '$1..$2', $attachment['body']) : $attachment['body'];
            $data .= $this->eol;
            $data .= $this->convertEncoding($body) . $this->eol;
        } else {
            // 添付
            $data .= "Content-Disposition: attachment; filename=\"{$attachment['name']}\"" . $this->eol;
            $data .= $this->eol;
            $data .= $attachment['file'] . $this->eol;
        }
        return $data;
    }

    /**
     * メールヘッダを配列として取得する
     * ※ヘッダのキーはすべて小文字になる
     *
     * @access private
     * @param string メールヘッダ
     * @return array メールヘッダの配列
     */
    function getHeaders($header)
    {
        $tmp = '';
        $headers = array();
        foreach (explode($this->eol, $header) as $line) {
            if (preg_match('/^[ |\t]/', $line, $matches)) {
                $tmp .= $this->eol . $line;
            } else {
                if ($tmp) {
                    if (preg_match('/^([^\:]+)\:(.+)$/is', $tmp, $matches)) {
                        $headers[strtolower($matches[1])][] = trim($matches[2]);
                    }
                }
                $tmp = $line;
            }
        }
        if ($tmp) {
            if (preg_match('/^([^\:]+)\:(.+)$/is', $tmp, $matches)) {
                $headers[strtolower($matches[1])][] = trim($matches[2]);
            }
        }
        return $headers;
    }
}
