<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_ValidationValidator.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * 検証個別クラス
 * 
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_ValidationValidator.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_ValidationValidator extends SyL_Validation
{
    /**
     * 検証タイプ
     *
     * @access protected
     * @var string
     */
    var $type = '';
    /**
     * パラメータ
     *
     * @access protected
     * @var array
     */
    var $parameters = array();
    /**
     * パラメータが配列時の最低検証OK数
     * ※ 0 はパラメータ数
     *
     * @access protected
     * @var int
     */
    var $min_valids = 0;
    /**
     * パラメータが配列時の最大検証OK数
     * ※ 0 はパラメータ数
     *
     * @access protected
     * @var int
     */
    var $max_valids = 0;

    /**
     * コンストラクタ
     *
     * @access public
     * @param string バリデータタイプ
     * @param string エラーメッセージのフォーマット
     * @param array 検証パラメータ
     */
    function SyL_ValidationValidator($type, $error_message, $parameters=array())
    {
        $this->type          = $type;
        $this->error_message = $error_message;
        $this->parameters    = array_merge($this->parameters, $parameters);
    }

    /**
     * 個別検証オブジェクトの取得
     *
     * @static
     * @access public
     * @param string 検証名
     * @param string エラーメッセージ
     * @param array 検証パラメータ
     * @param string カスタムバリデーションディレクトリ
     * @return object 個別検証オブジェクト
     */
    function &create($type, $error_message, $parameters=array(), $custom_dir='')
    {
        $classname = 'SyL_ValidationValidator' . ucfirst($type);
        if ($custom_dir) {
            include_once $custom_dir . "/{$classname}.php";
        } else {
            include_once dirname(__FILE__) . "/Validator/{$classname}.php";
        }
        $obj =& new $classname($type, $error_message, $parameters);
        return $obj;
    }

    /**
     * 検証パラメータをセット
     *
     * @access public
     * @param array 検証パラメータ
     */
    function setParameters($parameters)
    {
        $this->parameters = array_merge($this->parameters, $parameters);
    }

    /**
     * 最低検証OK数をセット
     *
     * @access public
     * @param int 最低検証OK数
     */
    function setMinValids($min_valids)
    {
        $this->min_valids = $min_valids;
    }

    /**
     * 最大検証OK数をセット
     *
     * @access public
     * @param int 最大検証OK数
     */
    function setMaxValids($max_valids)
    {
        $this->max_valids = $max_valids;
    }

    /**
     * 即時検証処理を実行する
     *
     * @static
     * @access public
     * @param string 検証名
     * @param string 検証対象値
     * @param array 検証パラメータ
     * @return bool true: エラー無し, false: エラーあり
     */
    function executeImmediate($validate_name, $value, $parameters=array())
    {
        $validator =& SyL_ValidationValidator::create($validate_name, '', $parameters);
        return $validator->execute($value);
    }

    /**
     * 検証処理を実行する
     *
     * @access public
     * @param string 検証対象値
     * @param string 検証対象名
     * @return bool true: エラー無し, false: エラーあり
     */
    function execute($value, $name='')
    {
        if (is_array($value)) {
            if (count($value) == 0) {
                if ($this->isRequire()) {
                    $this->replaceErrorMessage($name);
                    return false;
                } else {
                    return true;
                }
            } else {
                $ok = 0;
                foreach ($value as $tmp) {
                    if ($this->execute($tmp, $name)) {
                        $ok++;
                    }
                }
                if ($this->max_valids == 0) {
                    $this->max_valids = count($value);
                    if ($this->min_valids == 0) {
                        $this->min_valids = count($value);
                    }
                }
                if (($this->max_valids < $ok) || ($this->min_valids > $ok)) {
                    $this->replaceErrorMessage($name);
                    return false;
                } else {
                    return true;
                }
            }
        } else {
            // 必須チェック or ファイルチェック以外で、検証値が空の場合はtrue
            if (!$this->isRequire()) {
                if (!call_user_func(array(__CLASS__, 'executeImmediate'), 'require', $value)) {
                    return true;
                }
            }

            if (!$this->validate($value)) {
                $this->replaceErrorMessage($name);
                return false;
            } else {
                return true;
            }
        }
    }

    /**
     * 検証処理を実行する
     *
     * @access public
     * @param mixed 検証対象値
     * @return bool true: エラー無し, false: エラーあり
     */
    function validate($value)
    {
    }

    /**
     * エラーメッセージを取得する
     *
     * @access public
     * @param string 要素名
     */
    function replaceErrorMessage($name)
    {
        foreach ($this->parameters as $key => $value) {
            if (is_scalar($value)) {
                $search = array('{' . $key . '}', '%' . $key . '%');
                $this->error_message = str_replace($search, $value, $this->error_message);
            }
        }
        $this->error_message = str_replace(array('{min_valids}', '%min_valids%'), $this->min_valids, $this->error_message);
        $this->error_message = str_replace(array('{max_valids}', '%max_valids%'), $this->max_valids, $this->error_message);

        parent::replaceErrorMessage($name);
    }

    /**
     * 検証処理のJavaScriptを取得する
     *
     * @access public
     * @param string フォーム要素表示名
     * @param array フォーム要素の部品配列（radio, select, checkboxの場合のみ）
     * @return string JavaScript処理ロジック
     */
    function getJs($display_name)
    {
        $this->replaceErrorMessage($display_name);
        return $this->getJsCode();
    }

    /**
     * 検証処理のJavaScriptを取得する
     *
     * @access public
     * @param array フォーム要素の部品配列（radio, select, checkboxの場合のみ）
     * @return string JavaScript処理ロジック
     */
    function getJsCode()
    {
        return '';
    }
}

/**
 * エイリアス
 */
class SyL_Validator extends SyL_ValidationValidator{}
