<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_AdmOperation.php,v 1.1 2009/01/11 05:34:31 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * DAOクラス
 */
require_once SYL_FRAMEWORK_DIR . '/lib/DB/SyL_DBDao.php';
/**
 * DAOテーブルクラス
 */
require_once SYL_FRAMEWORK_DIR . '/lib/DB/Dao/SyL_DBDaoTable.php';

/**
 * 上位レイヤーデータ管理クラス - 共通
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_AdmOperation.php,v 1.1 2009/01/11 05:34:31 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_AdmOperation
{
    /**
     * アクションフォームオブジェクト
     *
     * @access private
     * @var object
     */
    var $form = null;
    /**
     * DAOオブジェクト
     *
     * @access private
     * @var object
     */
    var $dao = null;
    /**
     * テーブルオブジェクトの配列
     *
     * @access private
     * @var array
     */
    var $tables = array();
    /**
     * テーブル関連オブジェクトの配列
     *
     * @access private
     * @var array
     */
    var $relation = null;
    /**
     * 主キー配列
     * 
     * @access protected
     * @var array
     */
    var $keys = array();

    /**
     * コンストラクタ
     *
     * @access public
     * @param object アクションフォームオブジェクト
     * @param object DBオブジェクト
     * @param array 主キー配列
     */
    function SyL_AdmOperation(&$form, &$conn, $keys=array())
    {
        $this->form     =& $form;
        $this->dao      =& new SyL_DBDao($conn);
        $this->relation = $this->dao->createRelation();
        foreach ($keys as $key) {
           $key = $this->decodeKey($key);
           if (!is_array($key)) {
               SyL_Error::trigger("[Adm error] Invalid primary parameter");
           }
           $this->keys[] = $key;
        }
    }

    /**
     * クラスのプロパティからテーブルと関連を取得する
     *
     * @access public
     */
    function loadConfig()
    {
        foreach ($this->form->getTableClasses() as $alias => $classname) {
            $classname = SyL_Loader::convertClass($classname, true);
            $table =& new $classname();
            $this->addTable($table, $alias);
        }
        $relations = $this->form->getTableRelations();
        if ($relations) {
            foreach ($relations as $columns) {
                $this->addRelation($columns);
            }
        }

        // 主キー存在チェック
        if (count($this->getPrimary()) == 0) {
            SyL_Error::trigger("[Adm error] Primary key is required in Adm table object");
        }
    }

    /**
     * テーブルオブジェクトをセットする
     *
     * @access public
     * @param object テーブルオブジェクト
     * @param string 別名
     */
    function addTable(&$table, $alias)
    {
        // 表示項目
        foreach ($this->form->getListColumns($alias) as $column) {
            // テーブルオブジェクトに取得カラムを追加
            $table->set($column);
            // テーブルバリデーションを追加
            $column_def = $table->getColumn($column);
            if (isset($column_def['validate'])) {
                $element_config = $this->form->getConfig($column);
                if (isset($element_config['validate'])) {
                    $element_config['validate'] = array_merge($column_def['validate'], $element_config['validate']);
                } else {
                    $element_config['validate'] = $column_def['validate'];
                }
                $this->form->addConfig($column, $element_config);
            }
        }
        if ($alias && is_string($alias)) {
            // デフォルト全カラム取得フラグ無効
            $table->setDefaultAll(false);
            // テーブル別名セット
            $table->setAliasName($alias);
            $this->tables[$alias] =& $table;
        } else {
            SyL_Error::trigger("[Adm error] Invalid table alias name ({$alias})");
        }
    }

    /**
     * 関連をセットする
     *
     * @access public
     * @param string 関連カラム定義
     */
    function addRelation($join_column)
    {
        $columns  = explode(',', $join_column);
        $operator = array_shift($columns);
        list($column1, $column2) = array_map('trim', explode('=', $columns[0]));
        list($alias1, $columnname1) = explode('.', $column1, 2);
        list($alias2, $columnname2) = explode('.', $column2, 2);
        if ($operator == '+') {
            $this->relation->addLeftJoin($this->tables[$alias1], $this->tables[$alias2], $columns);
        } else {
            $this->relation->addInnerJoin($this->tables[$alias1], $this->tables[$alias2], $columns);
        }
    }

    /**
     * 主キーカラムを取得する
     *
     * @access public
     * @param bool 別名含むフラグ
     * @return array 主キーカラム
     */
    function getPrimary($alias_flag=true)
    {
        $name = $this->form->getMaintenanceTable();
        foreach (array_keys($this->tables) as $alias) {
            if ($name == $alias) {
                return $this->tables[$alias]->getPrimary($alias_flag);
            }
        }
        return array();
    }

    /**
     * 一覧表示が可能か判定する
     *
     * @access public
     * @return bool true: 一覧表示可能、false: 一覧表示不可
     */
    function isLst()
    {
        $enable = $this->form->getStructConfig('enable_lst');
        return ($enable === null) ? true : (bool)$enable;
    }

    /**
     * 新規登録が可能か判定する
     *
     * @access public
     * @return bool true: 新規登録可能、false: 新規登録不可
     */
    function isNew()
    {
        $enable = $this->form->getStructConfig('enable_new');
        return ($enable === null) ? true : (bool)$enable;
    }

    /**
     * 更新が可能か判定する
     *
     * @access public
     * @return bool true: 更新可能、false: 更新不可
     */
    function isUpd()
    {
        $enable = $this->form->getStructConfig('enable_upd');
        return ($enable === null) ? true : (bool)$enable;
    }

    /**
     * 削除が可能か判定する
     *
     * @access public
     * @return bool true: 削除可能、false: 削除不可
     */
    function isDel()
    {
        $enable = $this->form->getStructConfig('enable_del');
        return ($enable === null) ? true : (bool)$enable;
    }
     
    /**
     * 詳細表示が可能か判定する
     *
     * @access public
     * @return bool true: 詳細表示可能、false: 詳細表示不可
     */
    function isVew()
    {
        $enable = $this->form->getStructConfig('enable_vew');
        return ($enable === null) ? true : (bool)$enable;
    }

    /**
     * 検索ボックス表示が可能か判定する
     *
     * @access public
     * @return bool true: 表示可能、false: 表示不可
     */
    function isSch()
    {
        $enable = $this->form->getStructConfig('enable_sch');
        return ($enable === null) ? true : (bool)$enable;
    }

    /**
     * 主キーパラメータをエンコードする
     *
     * @access protected
     * @param array 主キーパラメータ
     * @return string エンコード後主キーパラメータ
     */
    function encodeKey($key)
    {
        $tmp = array();
        foreach ($key as $name => $value) {
            $tmp[] = "{$name}={$value}";
        }
        return implode("\t", $tmp);
    }

    /**
     * 主キーパラメータをデコードする
     *
     * @access protected
     * @param string 主キーパラメータ
     * @return array デコード後主キーパラメータ
     */
    function decodeKey($key)
    {
        $tmp = array();
        foreach (explode("\t", $key) as $tmp1) {
            $tmp2 = explode('=', $tmp1);
            if (count($tmp2) != 2) {
                return array();
            }
            $tmp[$tmp2[0]] = $tmp2[1];
        }
        return $tmp;
    }

    /**
     * 主キー条件を追加する
     *
     * @access protected
     * @param object 条件オブジェクト
     * @param string テーブル別名
     * @param bool 全てのキー
     */
    function setConditionKey(&$conditions, $alias, $all=false)
    {
        if (count($this->keys) == 0) {
            return;
        }

        $primaries = $this->getPrimary(true);
        if ($all) {
            foreach ($this->keys as $key) {
                $and = false;
                foreach ($key as $name => $value) {
                    if (array_search("{$alias}.{$name}", $primaries) === false) {
                        break;
                    }
                    $conditions->addEqual($name, $value, true, $and);
                    $and = true;
                }
            }
        } else {
            $and = false;
            foreach ($this->keys[0] as $name => $value) {
                if (array_search("{$alias}.{$name}", $primaries) === false) {
                    break;
                }
                $conditions->addEqual($name, $value, true, $and);
                $and = true;
            }
        }
    }

    /**
     * 詳細表示データを取得する
     *
     * @access public
     * @return array 詳細表示データ配列
     */
    function getRecord()
    {
        $tables = array();
        $primaries = $this->getPrimary(true);
        foreach (array_keys($this->tables) as $alias) {
            // テーブルオブジェクト取得
            $table =& $this->tables[$alias];
            if ($alias == $this->form->getMaintenanceTable()) {
                // 条件オブジェクト作成
                $conditions = $this->dao->createCondition();
                // 主キー条件セット
                $this->setConditionKey($conditions, $alias);
                // 条件オブジェクトセット
                $table->setConditions($conditions);
            }
            // テーブルオブジェクトセット
            $tables[$alias] =& $table;
        }

        // 表示データ取得
        $result = $this->dao->select($tables, $this->relation);
        return (count($result) == 0) ? null : $result[0];
    }

    /**
     * 主キーパラメータを取得する（先頭のみ）
     *
     * @access public
     * @return array 主キーパラメータ
     */
    function getKey()
    {
        return (count($this->keys) > 0) ? $this->keys[0] : array();
    }

    /**
     * 主キークエリパラメータを取得する（先頭のみ）
     *
     * @access public
     * @return string 主キークエリパラメータ
     */
    function getKeyQuery()
    {
        return (count($this->keys) > 0) ? $this->encodeKey($this->keys[0]) : '';
    }

    /**
     * バリデーションを実行
     *
     * @access public
     * @return bool バリデーション結果
     */
    function validate()
    {
        return $this->form->validate();
    }
}
