<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_ContextJs.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * リクエストクラス
 */
require_once SYL_FRAMEWORK_DIR . '/core/SyL_Request.php';

/**
 * JavaScript出力フレームワーク情報管理クラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_ContextJs.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_ContextJs extends SyL_Context
{
    /**
     * ビュータイプ
     * 
     * @access protected
     * @var string
     */
    var $view_type = 'js.default';

    /**
     * コンストラクタ
     *
     * @access public
     * @param object データ管理オブジェクト
     */
    function SyL_ContextJs(&$data)
    {
        // JSインポート判定
        if ($data->getImport()) {
            $this->is_execute = false;
        }
        parent::SyL_Context($data);
    }

    /**
     * リクエストオブジェクトを取得
     *
     * @access public
     * @return object リクエストオブジェクト
     */
    function &getRequest()
    {
        return SyL_Request::singleton();
    }

    /**
     * 環境変数からGETリクエストか判定する
     *
     * @access public
     * @return bool true: GETリクエスト、false: GETリクエスト以外
     */
    function isGet()
    {
        return (isset($_SERVER['REQUEST_METHOD']) && ($_SERVER['REQUEST_METHOD'] == 'GET'));
    }

    /**
     * 環境変数からPOSTリクエストか判定する
     *
     * @access public
     * @return bool true: POSTリクエスト、false: POSTリクエスト以外
     */
    function isPost()
    {
        return (isset($_SERVER['REQUEST_METHOD']) && ($_SERVER['REQUEST_METHOD'] == 'POST'));
    }

    /**
     * 環境変数からSSLか判定する
     *
     * @access public
     * @return bool true: POSTリクエスト、false: POSTリクエスト以外
     */
    function isSsl()
    {
        return (isset($_SERVER['HTTPS']) && ($_SERVER['HTTPS'] == 'on'));
    }

    /**
     * 環境変数からサーバー名を取得する
     *
     * @access public
     * @return string サーバー名
     */
    function getServerName()
    {
        if (isset($_SERVER['HTTP_HOST'])) {
            return $_SERVER['HTTP_HOST'];
        } else if (isset($_SERVER['SERVER_NAME'])) {
            if (isset($_SERVER['SERVER_PORT']) && ($_SERVER['SERVER_PORT'] != '80')) {
                return $_SERVER['SERVER_NAME'] . ':' . $_SERVER['SERVER_PORT'];
            } else {
                return $_SERVER['SERVER_NAME'];
            }
        } else {
            return null;
        }
    }

    /**
     * 環境変数からリクエストURL（SCRIPT_NAME）を取得する
     *
     * @access public
     * @return string SCRIPT_NAME
     */
    function getScriptName()
    {
        $request_uri = isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : null;
        $script_name = isset($_SERVER['SCRIPT_NAME']) ? $_SERVER['SCRIPT_NAME'] : null;
        if ($request_uri && $script_name) {
            if (preg_match ('/^' . preg_quote($script_name, '/') . '/', $request_uri)) {
                return $script_name;
            } else {
                // 拡張子が無い場合の対応
                $path_parts = pathinfo($script_name);
                if (isset($path_parts['extension'])) {
                    $path_parts['basename'] = basename($script_name, '.' . $path_parts['extension']);
                }
                if ($path_parts['dirname'] == '/') {
                    $path_parts['dirname'] = '';
                }
                return $path_parts['dirname'] . '/' . $path_parts['basename'];
            }
        } else {
            return $script_name;
        }
    }

    /**
     * 環境変数からリクエストURL（PHP_SELF）を取得する
     *
     * @access public
     * @return string PHP_SELF
     */
    function getPhpSelf()
    {
        return isset($_SERVER['PHP_SELF']) ? $_SERVER['PHP_SELF'] : null;
    }

    /**
     * 環境変数からリクエストURL（REQUEST_URI）を取得する
     *
     * @access public
     * @return string PHP_SELF
     */
    function getRequestUri()
    {
        return isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : null;
    }

    /**
     * 環境変数からPATHINFOを取得する
     *
     * @access public
     * @return string PATHINFO
     */
    function getPathInfo()
    {
        return isset($_SERVER['PATH_INFO']) ? $_SERVER['PATH_INFO'] : null;
    }

    /**
     * PATH_INFO内の各URLを配列として取得する
     *
     * @access protected
     * @param object コンテキストオブジェクト
     * @return array PATH_INFO内の各URL
     */
    function getUrlNames()
    {
        $pathinfo = $this->getPathInfo();
        if (substr($pathinfo, -1) == '/') {
            $pathinfo = substr($pathinfo, 0, -1);
        } else {
            $pathinfo = dirname($pathinfo);
        }
        // アクションディレクトリ取得
        $urls = substr($pathinfo, 1); // 先頭「/」削除
        return (($urls !== false) && ($urls !== '')) ? explode('/', $urls) : array();
    }
}
