<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_RequestFile.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * アップロードファイル保持クラス
 *
 * ※ファイルインデックスは、「getFileNext」「uploadAll」メソッドで
 *   インクリメントされる。
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_RequestFile.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_RequestFile extends SyL_UtilProperty
{
    /**
     * アップロードされたファイルのパーミッション
     * 
     * @access private
     * @var int
     */
    var $mode = 0666;
    /**
     * アップロードエラー配列
     * 
     * @access private
     * @var array
     */
    var $errors = array();
    /**
     * アップロードディレクトリ
     * 
     * @access private
     * @var string
     */
    var $upload_dir = SYL_UPLOAD_DIR;

    /**
     * コンストラクタ
     *
     * @access public
     */
    function SyL_RequestFile()
    {
        if (isset($_FILES)) {
            // ファイルキャッシュクリア
            clearstatcache();

            // 取得パラメータが配列か判定
            foreach ($_FILES as $key => $value) {
                if (is_array($value['error'])) {
                    for ($i=0; $i<count($value['error']); $i++) {
                        if ($value['error'][$i] > 0) {
                            $this->errors[$key][$i] = $this->getErrorMessagePHP($value['error'][$i], $i);
                        } else if ($value['size'][$i] == 0) {
                            $this->errors[$key][$i] = "[SyL error] Upload file is size of 0 (" . ($i+1) . ")";
                        } else if (!is_uploaded_file($value['tmp_name'][$i])) {
                            $this->errors[$key][$i] = "[SyL error] 'is_uploaded_file' function error (" . ($i+1) . ")";
                        }
                    }
                } else {
                    if ($value['error'] > 0) {
                        $this->errors[$key]['0'] = $this->getErrorMessagePHP($value['error']);
                    } else if ($value['size'] == 0) {
                        $this->errors[$key]['0'] = "[SyL error] Upload file is size of 0";
                    } else if (!is_uploaded_file($value['tmp_name'])) {
                        $this->errors[$key]['0'] = "[SyL error] Upload file 'is_uploaded_file' function error ";
                    }
                }
            }

            // パラメータ取得
            $this->parameters =& $_FILES;
        }
    }

    /**
     * エラーメッセージを取得
     * 
     * @access private
     * @param string エラーNo
     * @param int ファイル番号
     * @return string エラーメッセージ
     */
    function getErrorMessagePHP($errorno, $i='')
    {
        if (is_numeric($i)) {
            $i = '(' . ($i+1) . ')';
        }
        switch ($errorno) {
        case UPLOAD_ERR_INI_SIZE:   return "[SyL error] Upload size is more than php.ini file {$i}"; break;
        case UPLOAD_ERR_FORM_SIZE:  return "[SyL error] Upload size is more than html form init size {$i}"; break;
        case UPLOAD_ERR_PARTIAL:    return "[SyL error] Upload file can not upload a part of upload file {$i}"; break;
        case UPLOAD_ERR_NO_FILE:    return "[SyL error] Upload file is not exist {$i}"; break;
        case UPLOAD_ERR_NO_TMP_DIR: return "[SyL error] TMP_DIR undefined {$i}"; break;
        default:                    return "[SyL error] Undefined upload error (no {$errorno}) {$i}"; break;
        }
    }

    /**
     * エラー判定
     * 
     * @access public
     * @param bool true: エラーあり、false: エラー無し
     */
    function isError()
    {
        return (count($this->errors) > 0);
    }

    /**
     * エラーメッセージ取得
     * 
     * @access public
     * @param string フォーム名
     * @return array エラーメッセージ
     */
    function getErrorMessage($name)
    {
        return (isset($this->errors[$name]) ? $this->errors[$name] : array());
    }

    /**
     * アップロードディレクトリをセット
     * 
     * @access public
     * @param string アップロードディレクトリ
     */
    function setUploadDir($upload_dir, $absolute=false)
    {
        $upload_dir = str_replace('..', '.', $upload_dir);
        if (!preg_match('/^(.*)(\/|\\\\)$/', $upload_dir)) {
            $upload_dir .= '/';
        }

        if ($absolute) {
            $this->upload_dir = $upload_dir;
        } else {
            $this->upload_dir = SYL_UPLOAD_DIR . $upload_dir;
        }

        if (is_dir($this->upload_dir)) {
            if (!is_writable($this->upload_dir)) {
                trigger_error("[SyL error] Upload Directory permission denied ({$this->upload_dir})", E_USER_ERROR);
            }
        } else {
            if (!@mkdir($this->upload_dir, $this->mode)) {
                trigger_error("[SyL error] Upload Directory not found and unable to create directory ({$this->upload_dir})", E_USER_ERROR);
            }
        }
    }

    /**
     * ファイルアップロード実行
     * 
     * @access public
     * @param string フォーム要素名
     * @return boolean true: アップロード成功, false: アップロードエラー
     */
    function upload($name='')
    {
        $files = ($name) ? array($name) : array_keys($this->parameters);

        $upload_flag = true;
        foreach ($files as $name) {
            $file = $this->get($name);
            if (is_array($file['tmp_name'])) {
                for ($i=0; $i<count($file['tmp_name']); $i++) {
                    if ($file['error'][$i] == 0) {
                        if (!$this->_upload($name, $file['tmp_name'][$i], $file['name'][$i], $i)) {
                            $upload_flag = false;
                        }
                    }
                }
            } else {
                if ($file['error'] == 0) {
                    if (!$this->_upload($name, $file['tmp_name'], $file['name'])) {
                        $upload_flag = false;
                    }
                }
            }
        }
        return $upload_flag;
    }

    /**
     * ファイルアップロード実行
     * 
     * @access public
     * @param string フォーム要素名
     * @param string アップロードファイル
     * @return boolean true: アップロード成功, false: アップロードエラー
     */
    function _upload($name, $tmp_name, $upload_file, $i='')
    {
        if (is_numeric($i)) {
            $num = '(' . ($i+1) . ')';
        } else {
            $num = '';
            $i   = '0';
        }
        if (!is_writable($this->upload_dir)) {
            $this->errors[$name][$i] = "[SyL error] Upload Directory permission denied {$num} ({$this->upload_dir}{$upload_file})";
            return false;
        } else if (!@move_uploaded_file($tmp_name, $this->upload_dir . $upload_file)) {
            $this->errors[$name][$i] = "[SyL error] Upload file 'move_uploaded_file' function error {$num} ({$this->upload_dir}{$upload_file})";
            return false;
        } else {
            if ($this->mode) {
                @chmod($this->upload_dir . $upload_file, $this->mode);
            }
            return true;
        }
    }

}
