<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_RequestSession.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * セッション保持クラス
 * 
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_RequestSession.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_RequestSession extends SyL_UtilProperty
{
    /**
     * セッションURLパス
     *
     * @access private
     * @var string
     */
    var $path = '';
    /**
     * セッション開始フラグ
     *
     * @access private
     * @var bool
     */
    var $start = false;

    /**
     * コンストラクタ
     *
     * @access public
     */
    function SyL_RequestSession()
    {
        if (!$this->start) {
            // セッション名をセット
            if (SYL_SESSION_NAME) {
                session_name(SYL_SESSION_NAME);
            }
            // キャッシュ制御を変更
            if (SYL_SESSION_CACHE) {
                session_cache_limiter(SYL_SESSION_CACHE);
            }
            // セッション各種パラメータ
            list($this->path, $session_domain, $session_secure) = explode(':', SYL_SESSION_COOKIE_PARAMS, 3);
            if (!$this->path) $this->path = '/';
            if (!$session_domain) $session_domain = null;
            $session_secure = (bool)$session_secure;
            session_set_cookie_params(0, $this->path, $session_domain, $session_secure);
            // セッション保存パス
            $this->setSavePath();

            // セッション開始
            $this->startSession();
            // セッションID変更
            if (SYL_SESSION_REGENERATE) {
                $this->regenerateSession();
            }
        }
    }

    /**
     * セッション保存パスを設定
     *
     * @access protected
     */
    function setSavePath()
    {
        if (SYL_SESSION_SAVE_PATH) {
            session_save_path(SYL_SESSION_SAVE_PATH);
        }
    }

    /**
     * セッションを開始する
     *
     * @access public
     * @param array 旧セッションのデータ
     */
    function startSession($old_session_data=null)
    {
        if (!$this->start) {
            session_start();
            $this->start = true;
            if (is_array($old_session_data)) {
                $_SESSION = $old_session_data;
            }
            $this->parameters =& $_SESSION;
        }
    }

    /**
     * セッションが開始されているか判定
     *
     * @access public
     * @return true: 開始されている、false: 開始されていない
     */
    function isSession()
    {
        return $this->start;
    }

    /**
     * セッションIDを変更する
     * ※セッション開始後に実行する
     *
     * @access public
     */
    function regenerateSession()
    {
        if (version_compare(PHP_VERSION, '5.1.0', '>=')) {
            session_regenerate_id(true);
        } else {
            $old_session_id   = $this->getSessionId();
            $old_session_data = $this->parameters;
            $_SESSION = array();
            session_destroy();
            $this->start = false;

            $session_dir = session_save_path();
            $new_session_id = '';
            do {
              $new_session_id = md5(uniqid(rand(), true));
            } while (file_exists("{$session_dir}/sess_{$new_session_id}"));
            session_id($new_session_id);

            $this->startSession($old_session_data);
        }
    }

    /**
     * 全パラメータ削除
     * 
     * @access public
     */
    function deletes()
    {
        parent::deletes();
        if (isset($_COOKIE[session_name()])) {
           setcookie(session_name(), '', time()-42000, $this->path);
        }
        session_destroy();
        $this->start = false;
    }

    /**
     * セッション値を保存し終了する
     * 
     * @access public
     * @param array パラメータ配列
     */
    function writeCloseSession($values)
    {
        $this->startSession();
        parent::sets($values);
        $this->closeSession();
    }

    /**
     * セッションを終了する
     * 
     * @access public
     */
    function closeSession()
    {
        if ($this->start) {
            session_write_close();
            $this->start = false;
            parent::deletes();

            // 2回目以降のsession_start()でクッキーを発行しない対応
            if (ini_get('session.use_cookies')) {
              ini_set('session.use_cookies', '0');
            }
        }
    }

    /**
     * セッションIDを取得
     * 
     * @access public
     * @return mixed セッションID
     */
    function getSessionId()
    {
        return session_id();
    }

    /**
     * パラメータ名付セッションIDを取得
     * 
     * @access public
     * @return string パラメータ名付セッションID
     */
    function getSidQuery()
    {
        return session_name() . '=' . $this->getSessionId();
    }
}
