<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_Auth.php,v 1.1 2009/01/11 05:34:30 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * ユーザークラス
 */
require_once SYL_FRAMEWORK_DIR . '/core/SyL_User.php';

/**
 * 認証クラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_Auth.php,v 1.1 2009/01/11 05:34:30 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_Auth
{
    /**
     * セッションオブジェクト
     * 
     * @access private
     * @var object
     */
    var $session = null;
    /**
     * セッション名
     * 
     * @access private
     * @var string
     */
    var $session_name = '';
    /**
     * チャンレンジ値をセットするセッション変数名
     *
     * @access private
     * @var string
     */
    var $challenge_response_session_name = 'syl_auth_challenge';

    /**
     * コンストラクタ
     *
     * @access public
     */
    function SyL_Auth()
    {
    }

    /**
     * SyL_Authクラスのインスタンス取得
     *
     * @access public
     * @param string 認証タイプ
     * @return object 認証オブジェクト
     */
    function &singleton($type='')
    {
        static $singleton = array();
        if (!$type) {
            list($type) = explode(':', SYL_AUTH_TYPE, 2);
        }

        if (!isset($singleton[$type])) {
            SyL_Loggers::info("Initialize Auth Type: {$type}");

            $classname = 'SyL_Auth' . ucfirst($type);
            include_once SYL_FRAMEWORK_DIR . "/core/Auth/{$classname}.php";
            // 認証オブジェクト作成
            $singleton[$type] = new $classname();
            // ユーザークラス取得
            $classname = SYL_AUTH_USER_CLASS ? SyL_Loader::convertClass(SYL_AUTH_USER_CLASS, true) : 'SyL_User';
            // セッションオブジェクトセット
            $request =& SyL_Request::singleton();
            $singleton[$type]->setSessionObject($request->getSessionObject());
            // セッション名セット
            $singleton[$type]->setSessionName($classname);
            $user =& $singleton[$type]->getUser();
            // ユーザーオブジェクトが作成されていない、またはログインしていなければ作成
            if (!$user) {
                $obj =& new $classname();
                $singleton[$type]->setUser($obj);
            }
        }
        return $singleton[$type];
    }

    /**
     * セッションオブジェクトをセットする
     * 
     * @access private
     * @param object セッションタイプ
     */
    function setSessionObject(&$session)
    {
        $this->session =& $session;
    }

    /**
     * セッション名をセット
     * 
     * @access private
     * @param string セッション名
     */
    function setSessionName($session_name)
    {
        $this->session_name = $session_name;
    }

    /**
     * ユーザーオブジェクト取得
     * 
     * @access public
     * @return object ユーザーオブジェクト
     */
    function &getUser()
    {
        return $this->session->getRef($this->session_name);
    }

    /**
     * ユーザーオブジェクトをセット
     * 
     * @access public
     * @param object ユーザーオブジェクト
     */
    function setUser(&$user)
    {
        return $this->session->set($this->session_name, $user);
    }

    /**
     * ログインパラメータ名を取得
     * 
     * @access public
     * @return array ログインパラメータ
     */
    function getLoginParameterName()
    {
        $user =& $this->getUser();
        return array($user->getUserIdParamName(), $user->getPasswordParamName());
    }

    /**
     * 認証エリアの判定
     * 
     * @access public
     * @return bool true: 認証エリア、false: 認証エリア外
     */
    function isAuthArea()
    {
        $session_error_url = call_user_func(array(__CLASS__, 'getAuthSessionErrorUrl'));
        if ($_SERVER['PHP_SELF'] == $session_error_url) {
            SyL_Loggers::info("Auth Area: false. (because SYL_AUTH_SESSION_ERROR_URL)");
            return false;
        }

        $request_uri = '';
        if (isset($_SERVER['REQUEST_URI'])) {
            $request_uri = $_SERVER['REQUEST_URI'];
        } else {
            trigger_error('[SyL error] $_SERVER["REQUEST_URI"] not exist', E_USER_ERROR);
        }

        $auth_url = call_user_func(array(__CLASS__, 'getAuthUrl'));

        SyL_Loggers::debug("REQUEST_URI: $request_uri");
        SyL_Loggers::debug("AUTH_URL: $auth_url");

        if (preg_match('/^' . preg_quote($auth_url, '/') . '/', $request_uri)) {
            SyL_Loggers::info("Auth Area: true");
            return true;
        } else {
            SyL_Loggers::info("Auth Area: false");
            return false;
        }
    }

    /**
     * ログインアクション判定
     * 
     * @access public
     * @return bool true: ログインアクションあり、false: ログインアクションなし
     */
    function isLoginAction()
    {
        list($id, $passwd) = $this->getLoginParameterName();
        SyL_Loggers::debug("[Auth] login parameter ID: $id PASSWD: $passwd");
        if (($_SERVER['REQUEST_METHOD'] == 'POST') && isset($_POST[$id]) && isset($_POST[$passwd])) {
            SyL_Loggers::info("Login Action: true");
            return true;
        } else {
            SyL_Loggers::info("Login Action: false");
            return false;
        }
    }

    /**
     * ログイン認証実行
     * 
     * @access public
     * @return bool OK: 認証OK, NG: 認証NG
     */
    function doLogin()
    {
        return false;
    }

    /**
     * ログイン認証パスワード判定
     * 
     * @access protected
     * @param string クライアント送信ユーザー名
     * @param string クライアント送信パスワード
     * @param string サーバー側パスワード
     * @param string サーバー側ハッシュ化方法
     * @return bool OK: 認証OK, NG: 認証NG
     */
    function isLoginPassword($client_username, $client_password, $server_password)
    {
        if (!$this->isLoginAction()) {
            SyL_Loggers::notice("Login Failed. No Login Action. (class: " . get_class($this) . " user: {$client_username})");
        }
        if (($client_username === null) || ($client_username === '')) {
            SyL_Loggers::warn("Login Failed. Empty UserName. (class: " . get_class($this) . " user: {$client_username})");
            return false;
        }

        // ユーザーオブジェクト取得
        $user =& $this->getUser();

        // パスワードハッシュ化
        $hashes = $user->getPasswordHash();
        if ($hashes[0] || $hashes[1]) {
            $challenge = $this->getChallengeCode();
            if ($hashes[0] && $hashes[1]) {
                // ○ クライアント（ハッシュ） -> サーバー（ハッシュ）
                //    => そのまま比較
                if ($challenge) {
                    // - チャレンジあり
                    //  CL: md5(チャレンジ ＋ md5(パスワード))
                    //  SV: パスワード == md5(チャレンジ ＋ パスワード)
                    SyL_Loggers::info("Auth hash CL: {$hashes[0]} SV: {$hashes[1]} challenge: on");
                    $server_password = call_user_func($hashes[0], $challenge . $server_password);
                } else {
                    // - チャレンジなし
                    //  CL: md5(パスワード)
                    //  SV: パスワード == パスワード
                    SyL_Loggers::info("Auth hash CL: {$hashes[0]} SV: {$hashes[1]} challenge: off");
                }
            } else if ($hashes[0]) {
                // ○ クライアント（ハッシュ） -> サーバー（平）
                //    => サーバー側パスワードをハッシュ化して比較
                if ($challenge) {
                    // - チャレンジあり
                    //  CL: md5(チャレンジ ＋ md5(パスワード))
                    //  SV: パスワード == md5(チャレンジ ＋ md5(パスワード))
                    SyL_Loggers::info("Auth hash CL: {$hashes[0]} SV: (none) challenge: on");
                    $server_password = call_user_func($hashes[0], $server_password);
                    $server_password = call_user_func($hashes[0], $challenge . $server_password);
                } else {
                    // - チャレンジなし
                    //  CL: md5(パスワード)
                    //  SV: パスワード == md5(パスワード)
                    SyL_Loggers::info("Auth hash CL: {$hashes[0]} SV: (none) challenge: off");
                    $server_password = call_user_func($hashes[0], $server_password);
                }
            } else if ($hashes[1]) {
                // ○ クライアント（平） -> サーバー（ハッシュ）
                //    => 取得パスワードをハッシュ化して比較
                //  CL: なし
                //  SV: md5(パスワード)  == パスワード
                SyL_Loggers::info("Auth hash CL: (none) SV: {$hashes[1]} challenge: off");
                $client_password = call_user_func($hashes[1], $client_password);
            } else {
                trigger_error("[SyL error] Unknown auth error", E_USER_ERROR);
            }
        } else {
            // ○ クライアント（平） -> サーバー（平）
            //    => そのまま比較
            //  CL: なし
            //  SV: パスワード == パスワード
            SyL_Loggers::info("Auth hash CL: (none) SV: (none) challenge: off");
        }

        if (($server_password !== '')   &&
            ($server_password !== null) &&
            ($server_password === $client_password)) {
            SyL_Loggers::info("Login Success. (class: " . get_class($this) . " user: {$client_username})");
            // ユーザーIDセット
            $user->setUserId($client_username);
            // ログインOKイベント
            $user->triggerLoginSuccess($client_username);
            $this->setUser($user);
            return true;
        } else {
            SyL_Loggers::warn("Login Failed. Invalid Password. (class: " . get_class($this) . " user: {$client_username})");
            // ログインエラーイベント
            $user->triggerLoginFailed($client_username);
            $this->setUser($user);
            return false;
        }
    }

    /**
     * ログイン認証チェック
     * 
     * @access public
     * @return bool OK: 認証OK, NG: 認証NG
     */
    function isLogin()
    {
        $user =& $this->getUser();
        return $user->isLogin();
    }

    /**
     * ログイン認証削除
     * 
     * @access public
     */
    function deleteLogin()
    {
        $this->setUser($user=null);
    }

    /**
     * ログアウト処理を行う
     * 
     * @access public
     * @return boolean OK: true, NG: false
     */
    function doLogout()
    {
        $user =& $this->getUser();
        $client_username = $user->getUserId();
        SyL_Loggers::info("Logout Success. (class: " . get_class($this) . " user: {$client_username})");
        // ログアウトイベント
        $user->triggerLogout($client_username);
        $this->deleteLogin();
        session_destroy();
    }

    /**
     * チャレンジコードを生成し、セッションにセットして、取得する
     * 
     * @access public
     * @param string チャレンジコードの桁数
     * @return string チャレンジコード
     */
    function createChallengeCode($len=8)
    {
        $challenge = '';
        for ($i=0; $i<$len; $i++) {
            $challenge .= chr(mt_rand(48, 126));
        }
        $challenge = str_replace('\\', '_', $challenge); // \ は置き換え
        $this->session->set($this->challenge_response_session_name, $challenge);
        return $challenge;
    }

    /**
     * チャレンジコードを取得する
     * 
     * @access public
     * @return string チャレンジコード
     */
    function getChallengeCode()
    {
        $challenge = $this->session->get($this->challenge_response_session_name);
        return $challenge;
    }

    /**
     * ログイン後TOP画面URLを取得
     * 
     * @static
     * @access public
     */
    function getAuthUrl()
    {
        $request_uri = isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : '';
        $script_name = isset($_SERVER['SCRIPT_NAME']) ? $_SERVER['SCRIPT_NAME'] : '';
        if ($request_uri && $script_name) {
            if (preg_match ('/^' . preg_quote($script_name, '/') . '/', $request_uri)) {
                return $script_name . SYL_AUTH_PATHINFO;
            } else {
                // 拡張子が無い場合の対応
                $path_parts = pathinfo($script_name);
                if (isset($path_parts['extension'])) {
                    $path_parts['basename'] = basename($script_name, '.' . $path_parts['extension']);
                }
                if ($path_parts['dirname'] == '/') {
                    $path_parts['dirname'] = '';
                }
                return $path_parts['dirname'] . '/' . $path_parts['basename'] . SYL_AUTH_PATHINFO;
            }
        } else {
            trigger_error("[SyL error] Unable to get env(_SERVER['REQUEST_URI'] or _SERVER['SCRIPT_NAME'])", E_USER_ERROR);
        }
    }

    /**
     * セッションエラー画面へリダイレクト
     * 
     * @static
     * @access public
     * @param string 追加クエリストリング
     */
    function redirectTop($query_string='')
    {
        if ($query_string) {
            $query_string = '?' . $query_string;
        }
        SyL_Response::redirect(call_user_func(array(__CLASS__, 'getAuthUrl')) . $query_string);
    }

    /**
     * ログインエラーURLを取得
     * 
     * @static
     * @access public
     */
    function getAuthLoginErrorUrl()
    {
        if (SYL_AUTH_LOGIN_ERROR_URL) {
            return SYL_AUTH_LOGIN_ERROR_URL;
        } else {
            trigger_error("[SyL error] Define `SYL_AUTH_LOGIN_ERROR_URL' not value");
        }
    }

    /**
     * ログインエラー画面へリダイレクト
     * 
     * @static
     * @access public
     */
    function redirectLoginError()
    {
        SyL_Response::redirect(call_user_func(array(__CLASS__, 'getAuthLoginErrorUrl')));
    }

    /**
     * セッションエラーURLを取得
     * 
     * @static
     * @access public
     */
    function getAuthSessionErrorUrl()
    {
        if (SYL_AUTH_SESSION_ERROR_URL) {
            return SYL_AUTH_SESSION_ERROR_URL;
        } else {
            trigger_error("[SyL error] Define `SYL_AUTH_SESSION_ERROR_URL' not value");
        }
    }

    /**
     * セッションエラー画面へリダイレクト
     * 
     * @static
     * @access public
     */
    function redirectSessionError()
    {
        SyL_Response::redirect(call_user_func(array(__CLASS__, 'getAuthSessionErrorUrl')));
    }
}
