<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_Data.php,v 1.1 2009/01/11 05:34:30 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * 汎用プロパティクラス
 */
require_once SYL_FRAMEWORK_DIR . '/lib/Util/SyL_UtilProperty.php';

/**
 * フレームワークデータ保持クラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_Data.php,v 1.1 2009/01/11 05:34:30 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_Data extends SyL_UtilProperty
{
    /**
     * 入力データ
     *
     * @access protected
     * @var array
     */
    var $input_parameters = array();
    /**
     * 配列パラメータをgetメソッドで取得できるフラグ
     *
     * @access protected
     * @var bool
     */
    var $input_get_array = false;

    /**
     * データ保持クラスのインスタンス取得
     *
     * @access public
     * @return object データ保持オブジェクト
     */
    function &getInstance()
    {
        static $singleton;
        if (!is_object($singleton)) {
            $classname = 'SyL_Data' . ucfirst(SYL_ENV_TYPE);
            include_once SYL_FRAMEWORK_DIR . "/core/Data/{$classname}.php";
            // 子クラスのインスタンス生成
            $singleton = new $classname();
            // 初期パラメータ取得
            $singleton->initInputs();
        }
        return $singleton;
    }

    /**
     * プロパティをセット
     * 
     * @access public
     * @param string プロパティ名
     * @param string プロパティ値
     */
    function set($name, $value)
    {
        $this->checkName($name);
        parent::set($name, $value);
    }

    /**
     * プロパティをセット（参照）
     * 
     * @access public
     * @param string プロパティ名
     * @param string プロパティ値
     */
    function setRef($name, &$value)
    {
        $this->checkName($name);
        parent::setRef($name, $value);
    }

    /**
     * 複数プロパティをセット
     * 
     * @access public
     * @param array プロパティ配列
     */
    function sets($values)
    {
        foreach (array_keys($values) as $name) {
            $this->set($name, $values[$name]);
        }
    }

    /**
     * 複数プロパティをセット（参照）
     * 
     * @access public
     * @param array プロパティ配列
     */
    function setsRef(&$values)
    {
        foreach (array_keys($values) as $name) {
            $this->setRef($name, $values[$name]);
        }
    }

    /**
     * プロパティを取得
     * 
     * @access public
     * @param string プロパティ名
     * @param mixed デフォルト値
     * @param string 有効値正規表現
     * @return string プロパティ値
     */
    function get($name, $default=null, $valid=null)
    {
        $value = parent::get($name);
        if (($value === null) && isset($this->input_parameters[$name])) {
            $value = $this->input_parameters[$name];
            // 入力パラメータの配列は、getaメソッドで明示的に取得
            if (!$this->input_get_array && is_array($value)) {
                $value = null;
            }
        }
        if ($value === null) {
            return $default;
        } else {
            if ($valid !== null) {
                $valid = "/{$valid}/";
                return preg_match($valid, $value) ? $value : $default;
            } else {
                return $value;
            }
        }
    }

    /**
     * 入力パラメータ配列を明示的に取得する
     * 
     * @access public
     * @param string プロパティ名
     * @return array プロパティ値
     */
    function geta($name, $index=null)
    {
        if (isset($this->input_parameters[$name]) && is_array($this->input_parameters[$name])) {
            if ($index !== null) {
                return isset($this->input_parameters[$name][$index]) ? $this->input_parameters[$name][$index] : null;
            } else {
                return $this->input_parameters[$name];
            }
        } else {
            return null;
        }
    }

    /**
     * プロパティを取得（参照）
     * 
     * @access public
     * @param string プロパティ名
     * @param mixed デフォルト値
     * @param string 有効値正規表現
     * @return string プロパティ値
     */
    function &getRef($name, $default=null, $valid=null)
    {
        $value =& parent::getRef($name);
        if (($value === null) && isset($this->input_parameters[$name])) {
            $value =& $this->input_parameters[$name];
            // 入力パラメータの配列は、getaメソッドで明示的に取得
            if (!$this->input_get_array && is_array($value)) {
                $value = null;
            }
        }
        if ($value === null) {
            return $default;
        } else {
            if ($valid !== null) {
                $valid = "/{$valid}/";
                return preg_match($valid, $value) ? $value : $default;
            } else {
                return $value;
            }
        }
    }

    /**
     * 全プロパティを取得
     * 
     * @access public
     * @param bool 外部取得パラメータも取得するフラグ
     * @return array 全プロパティ値
     */
    function gets($init_param=true)
    {
        return ($init_param) ? parent::gets() + $this->input_parameters : parent::gets();
    }

    /**
     * 全プロパティを取得（参照）
     * 
     * @access public
     * @param bool 外部取得パラメータも取得するフラグ
     * @return array 全プロパティ値
     */
    function &getsRef($init_param=true)
    {
        $values =& parent::getsRef();
        if ($init_param) {
            foreach (array_keys($this->input_parameters) as $name) {
                if (!isset($values[$name])) {
                    $values[$name] =& $this->input_parameters[$name];
                }
            }
        }
        return $values;
    }

    /**
     * getメソッド配列パラメータ取得フラグをセット
     * 
     * @access public
     * @param bool 配列パラメータ取得フラグ
     */
    function isGetArray($input_get_array=false)
    {
        $this->input_get_array = (bool)$input_get_array;
    }

    /**
     * 初期入力パラメータ数を取得
     * 
     * @access public
     * @return int 初期入力パラメータ数
     */
    function getInitCount()
    {
        return count($this->input_parameters);
    }

    /**
     * 全プロパティを削除
     * 
     * @access public
     */
    function deletes()
    {
        $this->input_parameters = array();
        parent::deletes();
    }

    /**
     * プロパティの確認
     * 
     * @access public
     * @param string プロパティ名
     * @return bool true: プロパティあり、false: プロパティ無し
     */
    function is($name)
    {
        if (parent::is($name)) {
            return true;
        } else {
            return isset($this->input_parameters[$name]);
        }
    }

    /**
     * 初期入力パラメータをセット
     * 
     * @access protected
     */
    function initInputs()
    {
    }

    /**
     * 最後的にパラメータを取得するメソッド
     * 
     * @access protected
     * @return array パラメータ
     */
    function finalOutputs()
    {
    }

    /**
     * 入力パラメータをセット
     * 
     * @access public
     * @param array 入力パラメータ
     */
    function setInputs($input_parameters)
    {
        $this->input_parameters = $input_parameters;
    }

    /**
     * 入力パラメータを出力パラメータに透過的に渡す
     * 
     * @access public
     * @param string 入力パラメータ名
     */
    function throwInput($name)
    {
        if (isset($this->input_parameters[$name])) {
            $this->set($name, $this->input_parameters[$name]);
        } else {
            $this->set($name, null);
        }
    }

    /**
     * 全入力パラメータを出力パラメータに透過的に渡す
     * 
     * @access public
     */
    function throwInputs()
    {
        foreach ($this->input_parameters as $name => $value) {
            if (!parent::is($name)) {
                $this->set($name, $value);
            }
        }
    }

    /**
     * 全パラメータに指定関数を適用する
     * 
     * @access private
     * @param array パラメータ配列
     * @param string 関数名
     */
    function apply($func)
    {
        $func_parameters = func_get_args();
        $func = array_shift($func_parameters);
        // 既存のプロパティに適用
        parent::applyRecursive($func, $this->parameters, $func_parameters);
        // 入力パラメータに適用
        parent::applyRecursive($func, $this->input_parameters, $func_parameters);
    }

    /**
     * プライベートプロパティ名チェック
     * 
     * @access public
     * @param string プロパティ名
     */
    function checkName($name)
    {
        if ($name[0] == '_') {
            trigger_error("[SyL error] Data property name invalid first char ({$name})", E_USER_ERROR);
        }

        // セットする変数と同名の入力値は削除
        if (isset($this->parameters[$name])) {
            unset($this->parameters[$name]);
        }
    }

    /**
     * 全パラメータに文字コードチェック関数を適用する
     * 
     * @access protected
     * @param array パラメータ
     * @param string 文字コード
     */
    function applyCheckEncodingRecursive(&$parameters, $encoding)
    {
        if (is_array($parameters)) {
            foreach($parameters as $key => $value) {
                $this->applyCheckEncodingRecursive($parameters[$key], $encoding);
            }
        } else {
            if (is_scalar($parameters)) {
                if (!mb_check_encoding($parameters, $encoding)) {
                    trigger_error("[SyL error] Invalid parameter encoding ({$parameters})", E_USER_ERROR);
                }
            }
        }
    }
}
