<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDriverPdofirebird.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * DB操作クラス（PDO）
 */
require_once dirname(__FILE__) . '/../SyL_DBPdo.php';

/**
 *  DB操作クラス（PDO::Firebird）
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDriverPdofirebird.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_DBDriverPdofirebird extends SyL_DBPdo
{
    /**
     * エンコード変換テーブル
     *
     *  Firebird <=> PHP
     *  (client)     (client)
     * 
     * ※全て小文字で記述
     *
     * @access protected
     * @var string
     */
    var $encode_table = array(
      // PHP, DB
      array('eucjp-win', 'eucj_0208'),
      array('euc-jp',    'eucj_0208'),
      array('euc',       'eucj_0208'),
      array('sjis-win',  'sjis_0208' ),
      array('sjis',      'sjis_0208'),
      array('utf-8',     'unicode_fss')
    );

    /**
     * DB接続実行
     * 
     * @access public
     * @param string データベース名
     * @param string ユーザー名
     * @param string パスワード
     * @param string ホスト名
     * @param string ポート番号
     * @param bool 持続的接続を行うか
     * @return boolean 接続OK: true, 接続NG: false
     */
    public function openConnection($dbname, $user, $passwd, $host, $port, $persistent)
    {
        $conn_dbtype  = substr($this->dbtype, 3);
        $conn_strings = array();
        $conn_user    = null;
        $conn_passwd  = null;
        if ($host) {
            $conn_strings[] = "dbname={$host}:{$dbname}";
        } else {
            $conn_strings[] = "dbname={$dbname}";
        }
        if ($user && $passwd) {
            $conn_user   = $user;
            $conn_passwd = $passwd;
        } else if ($user) {
            $conn_user = $user;
        }

        $conn_string = $conn_dbtype . ':' . implode(';', $conn_strings);
        return $this->pdoConnection($conn_string, $conn_user, $conn_passwd);
    }

    /**
     * SQLを実行し、結果取得
     * 
     * ・SQL文が、select句の場合
     *   実行結果をリソースとして取得
     * ・SQL文が、insert, update, delete句の場合
     *   実行結果影響件数を取得
     * ・SQL文が、上記以外の場合
     *   実行後、true or falseを返却
     *
     * @access public
     * @param string SQL文
     * @param mixed 実行結果件数、またはDBリソース
     * @return boolean 実行OK: true, 実行NG: false
     */
    public function execRef($sql, &$result)
    {
        $this->convertEncoding($sql);
        return parent::execRef($sql, $result);
    }

    /**
     * SQL実行し、結果データを取得
     *
     * ・第3パラメータが、'one'の場合
     *    最初の1カラムを文字列として取得
     * ・第3パラメータが、'record'の場合
     *    最初の1レコードを配列として取得
     * ・第3パラメータが、'all'の場合
     *    全レコードを配列として取得
     *
     * @access public
     * @param string SQL文
     * @param mixed  実行結果を取得
     * @param string 結果取得フラグ
     * @return boolean 実行OK: true, 実行NG: false
     */
    public function queryRef($sql, &$data, $get='all')
    {
        if (parent::queryRef($sql, $data, $get)) {
            $this->convertDecoding($data);
            return true;
        } else {
            return false;
        }
    }

    /**
     * SQLを実行し、ページ毎にデータを取得
     *
     * @access public
     * @param string SQL文
     * @param array 実行結果格納配列
     * @param object ページオブジェクト
     * @param int 1ページの表示件数
     * @param int 表示対象ページ数 
     * @return bool 実行結果
     */
    public function queryPageRef($sql, &$result, &$pager, $limit=20, $page=1)
    {
        // ページングオブジェクト
        $pager = $this->getPager($limit, $page);
        // 総件数取得SQL作成
        $sql_count = "SELECT COUNT(*) FROM (" . $sql . ") AS SyL$";
        // 件数取得SQL実行
        if (!$this->queryRef($sql_count, $record, 'one')) return false;
        $pager->setCountAll($record);

        // ページ単位レコード取得
        $sql_page = $sql;
        if (preg_match('/^(select)\s+(.+)$/i', trim($sql_page), $matches)) {
            $sql_page = $matches[1] . ' first ' . $pager->getCount() . ' skip ' . $pager->getStartOffset() . ' ' . $matches[2];
        } else {
            trigger_error("[SyL error] SQL invalid ({$sql_page})", E_USER_ERROR);
        }
        // データ取得SQL実行
        return $this->queryRef($sql_page, $result, 'all');
    }

    /**
     * 接続しているDBサーバーのバージョンを取得する
     * 
     * @access public
     * @return string DBのバージョン
     */
    public function getVersion()
    {
/*
        $sql = "SHOW VERSION";
        $this->queryRef($sql, $version, 'one');
        list($version) = explode('-', $version);
        
        return $version;
*/
        return false;
    }
}
