<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDriverSybase.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * DB操作クラス（MSSQL, Sybase）
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDriverSybase.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_DBDriverSybase extends SyL_DB
{
    /**
     * エンコードフラグ
     *
     * @access private
     * @var bool
     */
    var $client_encode_flag = false;
    /**
     * DBサーバーエンコード
     * 
     * @access protected
     * @var string
     */
    var $server_encode = 'sjis-win';

    /**
     * DB接続実行
     * 
     * @access public
     * @param string データベース名
     * @param string ユーザー名
     * @param string パスワード
     * @param string ホスト名
     * @param string ポート番号
     * @param bool 持続的接続を行うか
     * @return boolean 接続OK: true, 接続NG: false
     */
    function openConnection($dbname, $user, $passwd, $host, $port, $persistent)
    {
        $host_tmp = $host;
        if (!$host_tmp) $host_tmp  = 'localhost';
        if ($port)      $host_tmp .= ':' . $port;

        if ($this->client_encode) {
            $this->connection = $persistent ? sybase_pconnect($host_tmp, $user, $passwd, $this->client_encode) : sybase_connect($host_tmp, $user, $passwd, $this->client_encode);
            $this->client_encode_flag = true;
        } else {
            $this->connection = $persistent ? sybase_pconnect($host_tmp, $user, $passwd) : sybase_connect($host_tmp, $user, $passwd);
        }
        return (is_resource($this->connection) && sybase_select_db($dbname, $this->connection));
    }

    /**
     * DB接続終了
     * 
     * @access public
     */
    function closeConnection()
    {
        if (is_resource($this->connection)) {
            sybase_close($this->connection);
            $this->connection = null;
        }
    }

    /**
     * クライアント側文字エンコーティング設定
     * 
     * @access public
     * @param string DB文字コード
     * @param string PHP側エンコード
     */
    function setClientEncoding($client_encode, $server_encode='')
    {
        $this->client_encode_flag = false;
        parent::setClientEncoding($client_encode, $server_encode);
    }

    /**
     * 最後に起こったエラーメッセージをセット
     *
     * @access public
     * @param string エラーコード
     * @param string エラーメッセージ
     */
    function setErrorMessage($code='', $message='')
    {
        if ($code) {
            parent::setErrorMessage($code, $message);
        } else {
            $message = sybase_get_last_message();
            if ($message) {
                $this->last_error_code    = '9999';
                $this->last_error_message = $message;
            }
        }
    }

    /**
     * パラメータサニタイズ（無効化）処理
     * 
     * @access public
     * @param string サニタイズ対象文字列
     * @return string サニタイズ後文字列
     */
    function escape($parameter)
    {
        return str_replace("'", "''", $parameter);
    }

    /**
     * トランザクション開始
     *
     * @access public
     */
    function beginTransaction()
    {
        parent::beginTransaction();
        @sybase_query('BEGIN TRAN', $this->connection);
    }

    /**
     * トランザクション破棄
     *
     * @access public
     */
    function rollBack()
    {
        @sybase_query('ROLLBACK TRAN', $this->connection);
        parent::rollBack ();
    }

    /**
     * トランザクション確定
     *
     * @access public
     */
    function commit()
    {
        @sybase_query('COMMIT TRAN', $this->connection);
        parent::commit();
    }

    /**
     * SQLを実行し、結果取得
     * 
     * ・SQL文が、select句の場合
     *   実行結果をリソースとして取得
     * ・SQL文が、insert, update, delete句の場合
     *   実行結果影響件数を取得
     * ・SQL文が、上記以外の場合
     *   実行後、true or falseを返却
     *
     * @access public
     * @param string SQL文
     * @param mixed 実行結果件数、またはDBリソース
     * @return boolean 実行OK: true, 実行NG: false
     */
    function execRef($sql, &$result)
    {
        if (!$this->client_encode_flag) {
            $this->convertEncoding($sql);
        }
        parent::execRef($sql, $result);
    }

    /**
     * SQLを実行し、DBリソースを取得
     *
     * @access public
     * @param string SQL文
     * @return mixed 実行OK: DBリソース, 実行NG: false
     */
    function execSelect($sql)
    {
        $result = @sybase_query($sql, $this->connection);
        if ($result !== false) {
            return $result;
        } else {
            $this->setErrorMessage();
            $this->triggerError();
            return false;
        }
    }

    /**
     * SQLを実行し、実行結果影響件数を取得
     *
     * @access public
     * @param string SQL文
     * @return mixed 実行OK: 実行結果影響件数, 実行NG: false
     */
    function execUpdate($sql)
    {
        $result = @sybase_query($sql, $this->connection);
        if ($result !== false) {
            return sybase_affected_rows($this->connection);
        } else {
            $this->setErrorMessage();
            $this->triggerError();
            return false;
        }
    }

    /**
     * SQL実行のみ
     *
     * @access public
     * @param string SQL文
     * @return boolean 実行OK: true, 実行NG: false
     */
    function execNoReturn($sql)
    {
        $result = @sybase_query($sql, $this->connection);
        if ($result !== false) {
            return true;
        } else {
            $this->setErrorMessage();
            $this->triggerError();
            return false;
        }
    }

    /**
     * SQL実行し、結果データを取得
     *
     * ・第3パラメータが、'one'の場合
     *    最初の1カラムを文字列として取得
     * ・第3パラメータが、'record'の場合
     *    最初の1レコードを配列として取得
     * ・第3パラメータが、'all'の場合
     *    全レコードを配列として取得
     *
     * @access public
     * @param string SQL文
     * @param mixed  実行結果を取得
     * @param string 結果取得フラグ
     * @return boolean 実行OK: true, 実行NG: false
     */
    function queryRef($sql, &$data, $get='all')
    {
        // SQL実行
        $this->execRef($sql, $query);

        // 取得判定
        if (!is_resource($query)) {
            $data = $query;
            return ($query !== false);
        }

        switch ($get) {
        case 'one':
            if (sybase_num_rows($query) > 0) {
                list($data) = sybase_fetch_row($query);
            } else {
                $data = '';
            }
            break;
        case 'record':
            $data = sybase_fetch_assoc($query);
            if ($data === false) {
                $data = array();
            } else {
                $data = $this->caseFieldName($data);
            }
            break;
        case 'all':
            $data = array();
            while ($result = sybase_fetch_assoc($query)) {
                $data[] = $this->caseFieldName($result);
            }
            break;
        }
        sybase_free_result($query);

        if (!$this->client_encode_flag) {
            $this->convertDecoding($data);
        }
        return true;
    }

    /**
     * SQLを実行し、ページ毎にデータを取得
     *
     * @access public
     * @param string SQL文
     * @param array 実行結果格納配列
     * @param object ページオブジェクト
     * @param int 1ページの表示件数
     * @param int 表示対象ページ数 
     * @return bool 実行結果
     */
    function queryPageRef($sql, &$result, &$pager, $limit=20, $page=1)
    {
        if (preg_match('/^select\s+(.+)order\s+by(.+)$/i', $sql, $matches)) {
            // ページングオブジェクト
            $pager = $this->getPager($limit, $page);
            // 総件数取得SQL作成
            $sql_count = "SELECT COUNT(*) FROM (SELECT " . $matches[1] . ") AS SyL$";
            // 件数取得SQL実行
            if (!$this->queryRef($sql_count, $record, 'one')) return false;
            $pager->setCountAll($record);

            // ページ単位レコード取得
            $sql_page = trim($sql);

            $sort_asc  = $matches[2];
            $sort_desc = '';
            foreach (array_map('trim', explode(',', $sort_asc)) as $column) {
                $column = explode(' ', $column);
                $sort = 'DESC';
                $i = 1;
                while (isset($column[$i])) {
                    switch (strtoupper($column[$i])) {
                    case 'ASC':  $sort = 'DESC'; break 2;
                    case 'DESC': $sort = 'ASC';  break 2;
                    }
                    $i++;
                }
                if ($sort_desc != '') {
                    $sort_desc .= ', ';
                }
                $sort_desc .= $column[0] . ' ' . $sort;
            }

            $num = 0;
            if ($pager->getCurrentPage() >= $pager->getTotalPage()) {
                $num = $pager->getCountAll() % $pager->getCount();
            } else {
                $num = $pager->getCount();
            }

            $sql_page  = "";
            $sql_page .= "SELECT * FROM (";
            $sql_page .=   "SELECT TOP " . $num . " * FROM (";
            $sql_page .=     "SELECT TOP " . ($pager->getCount() + $pager->getStartOffset()) . " ";
            $sql_page .=     $matches[1] . ' ORDER BY ' . $sort_asc;
            $sql_page .=   ") AS SyL1$ ";
            $sql_page .=   "ORDER BY ";
            $sql_page .=     $sort_desc;
            $sql_page .= ") AS SyL2$ ";
            $sql_page .= "ORDER BY ";
            $sql_page .=   $sort_asc;
        } else {
            trigger_error("[SyL error] Mssql SQL invalid [require `order by'] ({$sql_page})", E_USER_ERROR);
        }

        // データ取得SQL実行
        return $this->queryRef($sql_page, $result, 'all');
    }

    /**
     * 接続しているDBサーバーのバージョンを取得する
     * 
     * @access public
     * @return string DBのバージョン
     */
    function getVersion()
    {
        $sql = "SELECT @@VERSION";
        $this->queryRef($sql, $version, 'one');
        return $version;
    }
}
