<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DB.php,v 1.1 2009/01/11 05:34:29 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * DBスキーマ取得クラス
 */
require_once 'DB/SyL_DBSchema.php';

/**
 *  DB操作クラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DB.php,v 1.1 2009/01/11 05:34:29 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_DB
{
    /**
     * インスタンスDB名
     * 
     * @access protected
     * @var string
     */
    var $dbtype = '';
    /**
     * 接続情報
     * 
     * @access protected
     * @var array
     */
    var $dsn = array();
    /**
     * コネクションリソース or オブジェクト
     * 
     * @access protected
     * @var mixed
     */
    var $connection = null;
    /**
     * トランザクションフラグ
     * 
     * @access protected
     * @var boolean
     */
    var $transaction = false;
    /**
     * DBクライアントエンコード
     * ※PHPとしての文字コード名称
     * 
     * @access protected
     * @var string
     */
    var $client_encode = '';
    /**
     * PHPエンコード
     * 
     * @access protected
     * @var string
     */
    var $server_encode = '';
    /**
     * エンコード変換テーブル
     * 
     * @access protected
     * @var string
     */
    var $encode_table = array();

    /**
     * フィールド配列を参照する場合の
     * フィールド名の大文字、小文字統一参照フラグ
     *
     * @access protected
     * @var string 大文字参照: CASE_UPPER
     *             小文字参照: CASE_LOWER
     *             そのまま  : （空文字を設定）
     */
    var $field_case = CASE_UPPER;

    /**
     * tigger_error発生フラグ
     * 
     * @access protected
     * @var bool
     */
    var $trigger_error_flag = true;
    /**
     * 最終エラーコード
     * 
     * @access protected
     * @var string
     */
    var $last_error_code = '0';
    /**
     * 最終エラーメッセージ
     * 
     * @access protected
     * @var string
     */
    var $last_error_message = '';
    /**
     * 最終SQL
     * 
     * @access protected
     * @var string
     */
    var $last_sql = '';
    /**
     * SQLログ取得コールバックメソッド
     * 
     * @access protected
     * @var mixed
     */
    var $callback_sql_log = null;
    /**
     * エラーログ取得コールバックメソッド
     * 
     * @access protected
     * @var mixed
     */
    var $callback_error_log = null;

    /**
     * コンストラクタ
     *
     * @access public
     * @param string Dbタイプ
     * @param string クライアントエンコード
     * @param string サーバーエンコード
     */
    function SyL_DB($dbtype)
    {
        set_magic_quotes_runtime(0);
        $this->dbtype = $dbtype;

        $this->callback_sql_log   = array(&$this, 'logSql');
        $this->callback_error_log = array(&$this, 'logError');
    }

    /**
     * SyL_DBクラスのインスタンス取得
     *
     * ※取得方法
     *   $conn =& SyL_DB::getConnection();
     *
     * $dsn = '[dbtype]:dbname=[dbname];host=[hostname];port=[port];user=[username];passwd=[password];client_encode=[client_encode];persistent=[persistent]';
     *
     * PDOの場合のdbtype
     *   pdo.mysql
     *
     * @access public
     * @param string 接続文字列
     */
    function &getConnection($dsn='', $server_encode='')
    {
        if (!$dsn) {
            if (defined('SYL_DB_DSN') && SYL_DB_DSN) {
                $dsn = SYL_DB_DSN;
            } else {
                trigger_error("[SyL error] DSN(connection string) not found", E_USER_ERROR);
            }
        }

        static $singleton = array();
        if (!isset($singleton[$dsn]) || !is_object($singleton[$dsn])) {
            $tmp  = explode(':', $dsn, 2);
            $type = ucfirst($tmp[0]);
            $dbtype = $dbname = $user = $passwd = $host = $port = $persistent = $client_encode = null;
            if (isset($tmp[1])) {
                foreach (explode(';', $tmp[1]) as $tmp2) {
                    if (strpos($tmp2, '=') !== false) {
                        list($name, $value) = explode('=', $tmp2, 2);
                        $$name = $value;
                    }
                }
            }

            if (!$dbtype) {
                $dbtype = $type;
            }
            // ドライバクラス名
            $classname = 'SyL_DBDriver' . $type;

            // 実インスタンス作成
            include_once dirname(__FILE__) . "/DB/Driver/{$classname}.php";
            $singleton[$dsn] = new $classname($dbtype);
            if ($server_encode) $singleton[$dsn]->server_encode = $server_encode;
            if ($client_encode) $singleton[$dsn]->client_encode = $client_encode;
            if (!$singleton[$dsn]->openConnection($dbname, $user, $passwd, $host, $port, $persistent)) {
                $singleton[$dsn] = null;
            } else {
                $singleton[$dsn]->setDsnInfo($dbname, $user, $passwd, $host, $port, $persistent);
                if ($client_encode) {
                    $singleton[$dsn]->setClientEncoding($client_encode);
                }
            }
        }
        return $singleton[$dsn];
    }

    /**
     * DB接続実行
     * 
     * @access protected
     * @param string データベース名
     * @param string ユーザー名
     * @param string パスワード
     * @param string ホスト名
     * @param string ポート番号
     * @param string 持続的接続を行うか
     @ @return boolean 接続OK: true, 接続NG: false
     */
    function openConnection($dbname, $user, $passwd, $host, $port, $persistent)
    {
        trigger_error("[SyL error] openConnection method not found ({$dbname})", E_USER_ERROR);
    }

    /**
     * DB接続終了
     * 
     * @access public
     */
    function closeConnection() {}

    /**
     * DB接続ドライバ名を取得する
     * 
     * @access public
     * @return string DB接続ドライバ名
     */
    function getType()
    {
        return $this->dbtype;
    }

    /**
     * DB接続情報をセットする
     * 
     * @access public
     * @param string データベース名
     * @param string ユーザー名
     * @param string パスワード
     * @param string ホスト名
     * @param string ポート番号
     * @param string 持続的接続を行うか
     */
    function setDsnInfo($dbname, $user, $passwd, $host, $port, $persistent)
    {
        $this->dsn = array(
            'db'         => $dbname,
            'user'       => $user,
            //'password'   => $passwd,
            'host'       => $host,
            'port'       => $port,
            'persistent' => $persistent
        );
    }

    /**
     * DB接続情報を取得する
     * 
     * @access public
     * @return array DB接続情報
     */
    function getDsnInfo()
    {
        return $this->dsn;
    }

    /**
     * DBクライアント側文字エンコーティング設定
     * 
     * @access public
     * @param string DB文字コード
     * @param string PHP側エンコード
     */
    function setClientEncoding($client_encode, $server_encode='')
    {
        $client_encode = strtolower($client_encode);
        foreach ($this->encode_table as $encode_table) {
            if ($encode_table[1] == $client_encode) {
                $client_encode = $encode_table[0];
                break;
            }
        }
        $this->client_encode = $client_encode;

        if ($server_encode) {
            $this->server_encode = $server_encode;
        }
    }
  
    /**
     * パラメータサニタイズ（無効化）処理
     * 
     * @access public
     * @param string サニタイズ対象文字列
     * @return string サニタイズ後文字列
     */
    function escape($parameter)
    {
        return !is_numeric($parameter) ? addslashes($parameter) : $parameter;
    }

    /**
     * パラメータサニタイズ（無効化）処理のLike文用
     * 
     * @access public
     * @param string サニタイズ対象文字列
     * @return string サニタイズ後文字列
     */
    function escapeLike($parameter)
    {
        return preg_replace('/(\_|\%)/', '\\\\$1', $this->escape($parameter));
    }

    /**
     * パラメータを値によりクォートした文字列を取得する
     * ※サニタイズも実行される
     *
     * @access public
     * @param string クォート前文字列
     * @return string クォート後文字列
     */
    function quote($value)
    {
        if (($value === '') || ($value === null)) {
            return 'NULL';
        } else if (is_int($value) || is_float($value)) {
            return (string)$value;
        } else {
            return "'" . $this->escape($value) . "'";
        }
    }

    /**
     * トランザクション開始
     *
     * @access public
     */
    function beginTransaction()
    {
        // SQLログ
        call_user_func($this->callback_sql_log, 'begin');
        $this->transaction = true;
        return true;
    }

    /**
     * トランザクション破棄
     *
     * @access public
     */
    function rollBack()
    {
        // SQLログ
        call_user_func($this->callback_sql_log, 'rollback');
        $this->transaction = false;
        return true;
    }

    /**
     * トランザクション確定
     *
     * @access public
     */
    function commit()
    {
        // SQLログ
        call_user_func($this->callback_sql_log, 'commit');
        $this->transaction = false;
        return true;
    }

    /**
     * SQLを実行し、結果取得
     * 
     * ・SQL文が、select句の場合
     *   実行結果をリソースとして取得
     * ・SQL文が、insert, update, delete句の場合
     *   実行結果影響件数を取得
     * ・SQL文が、上記以外の場合
     *   実行後、true or falseを返却
     *
     * @access public
     * @param string SQL文
     * @param mixed 実行結果件数、またはDBリソース
     * @return boolean 実行OK: true, 実行NG: false
     */
    function execRef($sql, &$result)
    {
        // SQLログ
        call_user_func($this->callback_sql_log, $sql);
        // SQL保持
        $this->last_sql = $sql;
        // メソッド名抽出
        list($method) = explode(' ', $sql, 2);

        // SQL実行
        $result = false;
        switch (strtolower($method)) {
        case 'select': 
        case 'show':  $result = $this->execSelect($sql); break;
        case 'insert':
        case 'update':
        case 'delete': $result = $this->execUpdate($sql); break;
        default:       $result = $this->execNoReturn($sql); break;
        }
        return ($result !== false);
    }

    /**
     * SQLを実行し、結果取得
     * 
     * ・SQL文が、select句の場合
     *   実行結果をリソースとして取得
     * ・SQL文が、insert, update, delete句の場合
     *   実行結果影響件数を取得
     * ・SQL文が、上記以外の場合
     *   実行後、true or falseを返却
     *
     * @access public
     * @param string SQL文
     * @return mixed 実行結果件数、またはDBリソース
     */
    function exec($sql)
    {
        $this->execRef($sql, $result);
        return $result;
    }

    /**
     * SQLを実行し、DBリソースを取得
     *
     * @access protected
     * @param string SQL文
     * @return mixed 実行OK: DBリソース, 実行NG: false
     */
    function execSelect($sql)
    {
        trigger_error("[SyL error] execSelect method not found ({$dbname})", E_USER_ERROR);
    }

    /**
     * SQLを実行し、実行結果影響件数を取得
     *
     * @access protected
     * @param string SQL文
     * @return mixed 実行OK: 実行結果影響件数, 実行NG: false
     */
    function execUpdate($sql)
    {
        trigger_error("[SyL error] execUpdate method not found ({$dbname})", E_USER_ERROR);
    }

    /**
     * SQL実行のみ
     *
     * @access protected
     * @param string SQL文
     * @return boolean 実行OK: true, 実行NG: false
     */
    function execNoReturn($sql)
    {
        trigger_error("[SyL error] execNoReturn method is not found ({$dbname})", E_USER_ERROR);
    }

    /**
     * SQL実行し、結果データを取得
     *
     * @access public
     * @param string SQL文
     * @return mixed 実行結果
     */
    function query($sql)
    {
        $this->queryRef($sql, $data, 'all');
        return $data;
    }

    /**
     * SQL実行し、結果データを取得
     *
     * ・第3パラメータが、'one'の場合
     *    最初の1カラムを文字列として取得
     * ・第3パラメータが、'record'の場合
     *    最初の1レコードを配列として取得
     * ・第3パラメータが、'all'の場合
     *    全レコードを配列として取得
     *
     * @access public
     *
     * @param string SQL文
     * @param mixed  実行結果を取得
     * @param string 結果取得フラグ
     * @return boolean 実行OK: true, 実行NG: false
     */
    function queryRef($sql, &$data, $get='all') {}

    /**
     * SQLを実行し、ページ毎にデータを取得
     *
     * @access public
     * @param string SQL文
     * @param array 実行結果格納配列
     * @param object ページオブジェクト
     * @param int 1ページの表示件数
     * @param int 表示対象ページ数 
     * @return bool 実行結果
     */
    function queryPageRef($sql, &$result, &$pager, $limit, $page=1) {}

    /**
     * ページングオブジェクトを取得する
     *
     * @access protected
     * @param int 1ページの表示件数
     * @param int 表示対象ページ数 
     * @return object ページングオブジェクト
     */
    function &getPager($limit=20, $page=1)
    {
        include_once dirname(__FILE__) . '/Util/SyL_UtilPager.php';
        $pager = new SyL_UtilPager();
        $pager->setCount($limit);
        $pager->setPage($page);
        return $pager;
    }

    /**
     * SQL構築クラスを取得する
     *
     * @access public
     * @return object SQL構築オブジェクト
     */
    function &createBuilder()
    {
        include_once dirname(__FILE__) . '/Sql/SyL_SqlBuilder.php';
        $builder =& new SyL_SQLBuilder($this);
        return $builder;
    }

    /**
     * SQL構築クラスから取得したSQLを実行する
     *
     * @access public
     * @param object SQL構築オブジェクト
     * @param string select or insert or update or delete
     * @return mixed 実行結果
     */
    function execBuilder(&$builder, $action='select')
    {
        switch ($action) {
        case 'select':
        case 'insert':
        case 'update':
        case 'delete':
            break;
        default:
            trigger_error("[SyL error] Invalid action ({$action})", E_USER_ERROR);
        }
        $func = 'getStatement' . ucfirst($action);
        $sql  = call_user_func(array(&$builder, $func));
        return $this->exec($sql);
    }

    /**
     * SQL構築クラスから取得したSQLを実行する
     *
     * @access public
     * @param object SQL構築オブジェクト
     * @return mixed 実行結果
     */
    function queryBuilder(&$builder)
    {
        return $this->execBuilder($builder, 'select');
    }

    /**
     * SQLを実行し、ページ毎にデータを取得
     *
     * @access public
     * @param object SQL構築オブジェクト
     * @param array 実行結果格納配列
     * @param object ページオブジェクト
     * @param int 1ページの表示件数
     * @param int 表示対象ページ数 
     * @return bool 実行結果
     */
    function queryBuilderPageRef(&$builder, &$result, &$pager, $limit, $page=1)
    {
        $sql = $builder->getStatementSelect();
        return $this->queryPageRef($sql, $result, $pager, $limit, $page=1);
    }

    /**
     * 指定形式のパラメータを引数に、SQL（DML）を組み立て実行する
     * ※サニタイズも自動で行われる
     * ※変数は配列で指定する以外は、型で判定される。
     *
     * array(
     *  'name'     => $name,
     *  'address'  => $adress,
     *  'datetime' => array('function' => 'current_timestamp'),
     *  'id'       => array('number' => 1),
     *  'IS_NULL'  => array('null' => ''),
     *  ...
     * );
     *
     * @access public
     * @param string テーブル名
     * @param array カラムとそのデータ配列
     * @param string insert or update or delete
     * @param string 条件パラメータ（update, delete時のみ）
     * @return bool 実行結果 true: OK、false: エラー
     */
    function execPerform($table, $columns, $action='insert', $where='')
    {
        switch (strtolower($action)) {
        case 'insert':
            $field = array();
            $data  = array();
            foreach ($columns as $column => $value) {
                $field[] = $column;
                if (is_array($value)) {
                    list($attribute, $value) = each($value);
                    if ((strtolower($attribute) === 'null') || ($value === null) || ($value === '')) {
                        $data[] = "NULL";
                    } else {
                        switch (strtolower($attribute)) {
                        case 'function':
                        case 'number':
                            $data[] = $value;
                            break;
                        default:
                            $data[] = $this->quote($value);
                            break;
                        }
                    }
                } else {
                    $data[] = $this->quote($value);
                }
            }
            // INSERT SQL
            $sql = "INSERT INTO {$table} (" . implode(',', $field) . ") VALUES (" . implode(',', $data) . ")";
            break;

        case 'update':
            $update = array();
            foreach ($columns as $column => $value) {
                if (is_array($value)) {
                    list($attribute, $value) = each($value);
                    if ((strtolower($attribute) === 'null') || ($value === null) || ($value === '')) {
                        $update[] = $column . " = NULL";
                    } else {
                        switch (strtolower($attribute)) {
                        case 'function':
                        case 'number':
                            $update[] = $column . " = " . $value;
                            break;
                        default:
                            $update[] = $column . " = " . $this->quote($value);
                            break;
                        }
                    }
                } else {
                    $update[] = $column . " = " . $this->quote($value);
                }
            }

            if ($where) {
                $where = "WHERE " . $where;
            }
            // UPDATE SQL
            $sql = "UPDATE {$table} SET " . implode(',', $update) . " " . $where;
            break;

        case 'delete':
            if ($where) {
                $where = "WHERE " . $where;
            }
            // DELETE SQL
            $sql = "DELETE FROM {$table} {$where}";
            break;

        default:
          trigger_error("[SyL error] queryPerform method is insert or update or delete only", E_USER_ERROR);
        }

        return $this->exec($sql);
    }

    /**
     * 最後に挿入された行の ID あるいはシーケンスの値を取得
     *
     * ※使用できないDBはfalseを返す
     *
     * @access public
     * @param string シーケンス名
     * @return int 最後に挿入された行のID
     */
    function lastInsertId($seq='')
    {
        return false;
    }

    /**
     * 接続しているDBサーバーのバージョンを取得する
     * 
     * @access public
     * @return string DBのバージョン
     */
    function getVersion()
    {
        return false;
    }

    /**
     * トリガエラーを自動で起動しないように設定
     *
     * @access public
     * @param boolean true: トリガエラーON、false: トリガエラーOFF
     */
    function isTriggerError($trigger_error_flag)
    {
        $this->trigger_error_flag = ($trigger_error_flag === true);
    }

    /**
     * trigger_error発動
     * ただし、トランザクション中は発動しない
     *
     * @access public
     */
    function triggerError()
    {
        // SQLログ
        call_user_func($this->callback_error_log, $this->getErrorMessage());
        if (!$this->transaction && $this->trigger_error_flag) {
            trigger_error($this->getErrorMessage(), E_USER_ERROR);
        }
    }

    /**
     * エラー判定
     *
     * @access public
     * @return bool true: エラーあり、false: エラー無し
     */
    function isError()
    {
        return ($this->last_error_code != '0');
    }

    /**
     * 最後に起こったエラーコードを取得
     *
     * @access public
     * @return string 最後に起こったエラーコード
     */
    function errorCode()
    {
        return $this->last_error_code;
    }

    /**
     * 最後に起こったエラーメッセージを取得
     *
     * @access public
     * @return string 最後に起こったエラーメッセージ
     */
    function errorInfo()
    {
        return $this->last_error_message;
    }

    /**
     * 最後に起こったエラーメッセージをセット
     *
     * @access public
     * @param string エラーコード
     * @param string エラーメッセージ
     */
    function setErrorMessage($code='', $message='')
    {
        $this->last_error_code    = $code;
        $this->last_error_message = $message;
    }

    /**
     * 最後に起こったエラーメッセージを取得
     *
     * エラーが起きていない場合は、''
     *
     * @access public
     * @return string 最後に起こったエラーメッセージ
     */
    function getErrorMessage()
    {
        if ($this->errorInfo()) {
            $eol = (PHP_SAPI == 'cli') ? "\n" : '<br>';
            return $error_message = "[SyL DB Error " . $this->errorCode() . "] " . $this->errorInfo() . "{$eol}----------{$eol}{$this->last_sql}{$eol}----------{$eol}";
        } else {
            return '';
        }
    }

    /**
     * 大文字または小文字参照を取得する
     *
     * @access public
     * @return int 大文字または小文字参照
     */
    function getFieldCase()
    {
        return $this->field_case;
    }

    /**
     * 結果セット配列のフィールド名を大文字または小文字参照にする
     *
     * @access public
     * @param array 変換前配列
     * @return array 変換後配列
     */
    function caseFieldName($record)
    {
        if (is_array($record) && $this->field_case) {
            return array_change_key_case($record, $this->field_case);
        } else {
            return $record;
        }
    }

    /**
     * エンコード変換（PHP => DB）を行う
     *
     * (1) set names sjis
     *    db     ?
     *    client sjis
     *    view   sjis
     * (2) convertEncoding
     *    ・encode
     *    db     ?
     *    client sjis  mb_convert_encoding($value, ?, 'sjis');
     *    view   sjis
     *    ・decode
     *    db     ?
     *    client sjis  mb_convert_encoding($value, 'sjis', ?);
     *    view   sjis
     * @access public
     * @param mixed 変換前値
     * @return mixed 変換後値
     */
    function convertEncoding(&$value)
    {
        if ($this->client_encode && $this->server_encode) {
            if (is_array($value)) {
                foreach ($value as $key => $value2) {
                    $this->convertEncoding($value[$key]);
                }
            } else {
                $value = mb_convert_encoding($value, $this->server_encode, $this->client_encode);
            }
        }
    }

    /**
     * デコード変換（DB => PHP）を行う
     *
     * @access public
     * @param mixed 変換前値
     * @return mixed 変換後値
     */
    function convertDecoding(&$value)
    {
        if ($this->client_encode && $this->server_encode) {
            if (is_array($value)) {
                foreach ($value as $key => $value2) {
                    $this->convertDecoding($value[$key]);
                }
            } else {
                $value = mb_convert_encoding($value, $this->client_encode, $this->server_encode);
            }
        }
    }

    /**
     * ログ取得コールバックメソッドをセット
     *
     * @access public
     * @param mixed ログ取得コールバックメソッド
     */
    function setCallbackSqlLog($callback_sql_log)
    {
        if (is_callable($callback_sql_log)) {
            $this->callback_sql_log = $callback_sql_log;
        } else {
            trigger_error("[SyL error] " . __FUNCTION__ . " argument invalid function (" . print_r($callback_sql_log, true) . ")");
        }
    }

    /**
     * エラーメッセージ取得コールバックメソッドをセット
     *
     * @access public
     * @param mixed エラーメッセージ取得コールバックメソッド
     */
    function setCallbackErrorLog($callback_error_log)
    {
        if (is_callable($callback_error_log)) {
            $this->callback_error_log = $callback_error_log;
        } else {
            trigger_error("[SyL error] " . __FUNCTION__ . " argument invalid function (" . print_r($callback_error_log, true) . ")");
        }
    }

    /**
     * オリジナルリソース、またはオブジェクトを取得
     *
     * @access public
     * @return mixed オリジナルリソース、またはオブジェクト
     */
    function &getResource()
    {
        return $this->connection;
    }

    /**
     * 実行SQL文を取得
     *
     * @access public
     * @param string SQL文
     */
    function logSql($sql)
    {
    }

    /**
     * エラーメッセージを取得
     *
     * @access public
     * @param string エラーメッセージ
     */
    function logError($error_message)
    {
    }

    /**
     * スキーマ取得オブジェクトを取得する
     *
     * @access public
     * @return object スキーマ取得オブジェクト
     */
    function &getSchema()
    {
        return SyL_DBSchema::factory($this);
    }

}
