<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_UtilUserAgent.php,v 1.1 2009/01/11 05:34:29 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * ユーザーエージェント判定クラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_UtilUserAgent.php,v 1.1 2009/01/11 05:34:29 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_UtilUserAgent
{
    /**
     * ユーザーエージェント
     *
     * @access private
     * @var string
     */
    var $user_agent = '';

    /**
     * プラットフォーム環境（OS）配列（マスタ）
     *
     * @access private
     * @var array
     */
    var $platforms = array(
      'Windows NT 6\.0' => 'Windows Vista',
      'Windows NT 5\.2' => 'Windows 2003 or XP x64',
      'Windows NT 5\.1' => 'Windows XP',
      'Windows NT 5\.0' => 'Windows 2000',
      'Windows 2000'    => 'Windows 2000',
      'Windows NT 4\.0' => 'Windows NT 4.0',
      'winnt[ ]?4\.0'   => 'Windows NT 4.0',
      'winnt'           => 'Windows NT',
      'windows (9[58])' => 'Windows {$1}',
      'win(9[58])'      => 'Windows {$1}',
      'Win 9x 4\.9'     => 'Windows Me',
      'windows ce'      => 'Windows CE',
      'windows'         => 'Unknown Windows OS',
      'Mac OS X ([0-9]+)_([0-9]+)' => 'Mac OS X {$1}.{$2}',
      'Mac OS X ([0-9\.]+)' => 'Mac OS X {$1}',
      'Mac OS X'  => 'Mac OS X',
      'Mac OS'    => 'Mac OS',
      'ppc mac'   => 'Mac OS',
      'Macintosh' => 'Mac OS',
      'Debian package ([0-9\.]+)' => 'Debian GNU/Linux {$1}',
      'Debian.+etch'  => 'Debian GNU/Linux 4.0',
      'Debian.+sarge' => 'Debian GNU/Linux 3.1',
      'Debian.+woody' => 'Debian GNU/Linux 3.0',
      'Debian'        => 'Debian GNU/Linux',
      'Fedora\/[0-9\-\.]+fc([0-9]+)' => 'Fedora Core {$1}',
      'Remi\/[0-9\-\.]+fc([0-9]+)' => 'Fedora Core {$1}',
      'Fedora'          => 'Fedora Core',
      'CentOS\/[0-9\-\.]+el([0-9]+)' => 'CentOS {$1}',
      'CentOS'          => 'CentOS',
      'Red Hat\/[0-9\-\.]+el([0-9]+)' => 'Red Hat Linux {$1}',
      'Red Hat'         => 'Red Hat Linux',
      'Ubuntu\/([0-9\.]+)' => 'Ubuntu Linux {$1}',
      'Ubuntu'          => 'Ubuntu Linux',
      'Vine\/[0-9\-\.]+vl([0-9]+)' => 'Vine Linux {$1}',
      'VineLinux\/[0-9\-\.]+vl([0-9]+)' => 'Vine Linux {$1}',
      'Vine'            => 'Vine Linux',
      'Asianux\/[0-9\-\.]+el([0-9]+)' => 'Asianux {$1}',
      'Asianux'         => 'Asianux',
      'Mandriva\/[0-9\-\.]+mdv([0-9\.]+)' => 'Mandriva Linux {$1}',
      'Mandriva' => 'Mandriva Linux',
      'SUSE'     => 'SUSE Linux',
      'SunOS'    => 'Sun Solaris',
      'gnu'      => 'GNU/Linux',
      'linux'    => 'Linux',
      'unix'     => 'Unknown Unix OS',

      'Nintendo Wii'     => 'Wii',
      'PS2; PlayStation' => 'PlayStation 2',
      'PLAYSTATION 3'    => 'PlayStation 3',
      'Dreampassport'    => 'DreamCast',
    );

    /**
     * ブラウザ配列
     *
     * @access private
     * @var array
     */
    var $browsers = array(
      'Lunascape ([0-9\.]+)'  => 'Lunascape {$1}',
      'Sleipnir\/([0-9\.]+)'  => 'Sleipnir {$1}',
      'Opera[ \/]([0-9\.]+)'  => 'Opera {$1}',
      'MSIE ([0-9\.]+)'       => 'Internet Explorer {$1}',
      'Iceweasel\/([0-9\.]+)' => 'Iceweasel {$1}',
      'Epiphany\/([0-9\.]+)'  => 'Epiphany {$1}',
      'Firefox\/([0-9\.]+)'   => 'Firefox {$1}',
      'Chrome\/([0-9\.]+)'   => 'Chrome {$1}',
      'Safari\/85\.' => 'Safari 1.0',
      'Safari\/100'  => 'Safari 1.1',
      'Safari\/125'  => 'Safari 1.2',
      'Safari\/312'  => 'Safari 1.3',
      'Safari\/41\d' => 'Safari 2.0',
      'Safari\/523'  => 'Safari 3.0',
      'Safari\/525'  => 'Safari 3.1',
      'Sylera\/([0-9\.]+)'    => 'Sylera {$1}',
      'SeaMonkey\/([0-9\.]+)' => 'SeaMonkey {$1}',
      'Camino\/([0-9\.]+)'    => 'Camino {$1}',
      'Netscape\d?\/([0-9\.]+)' => 'Netscape {$1}',
      'Galeon\/([0-9\.]+)'    => 'Galeon {$1}',
      'Konqueror\/([0-9\.]+)' => 'Konqueror {$1}',
      'Lynx\/([0-9\.]+)'      => 'Lynx {$1}',
      'jig browser' => 'jig browser',
      'hotjava'     => 'HotJava',
      'icab'        => 'iCab',

      'NetFront'    => 'NetFront',
      'UP\.Browser' => 'UP.Browser',
      'L\-mode'     => 'L-mode',
      'AVE\-Front'  => 'AVE-Front',

      'Dreampassport' => 'Dreampassport',

      'OmniWeb' => 'OmniWeb',
      'Mozilla' => 'Mozilla'
    );

    /**
     * モバイルキャリア配列（マスタ）
     *
     * @access private
     * @var array
     */
    var $carriers = array(
      'DoCoMo'     => 'DoCoMo',
      'J-PHONE'    => 'SoftBank',
      'Vodafone'   => 'SoftBank',
      'SoftBank'   => 'SoftBank',
      'MOT-'       => 'SoftBank',
      'KDDI-'      => 'KDDI(au)',
      'UP.Browser' => 'KDDI(au)',
      'WILLCOM'    => 'WILLCOM',
      'DDIPOCKET'  => 'WILLCOM',
      'PDXGW'      => 'WILLCOM'
    );

    /**
     * ROBOT配列（マスタ）
     *
     * @access private
     * @var array
     */
    var $robots = array(
      'Googlebot\-Image'  => 'Googlebot-Image',
      'Googlebot\-Mobile\/([0-9\.]+)' => 'Googlebot-Mobile {$1}',
      'Googlebot\/([0-9\.]+)' => 'Googlebot {$1}',
      'Yahoo\-MMCrawler'  => 'Yahoo-MMCrawler',
      'Yahoo! Slurp'      => 'Yahoo! Slurp',
      'Y!J\-'             => 'Y!J',
      'msnbot\-NewsBlogs' => 'msnbot-NewsBlogs',
      'msnbot\-media\/([0-9\.]+)' => 'msnbot-media {$1}',
      'msnbot\/([0-9\.]+)' => 'msnbot {$1}',
      'Yeti\/([0-9\.]+)'   => 'YetiBot {$1}',
      'Baiduspider'        => 'Baiduspider',
      'ichiro\/([0-9\.]+)' => 'ichiro {$1}',
      'blogoonbot'      => 'blogoonbot',
      'Comaneci_bot'    => 'Comaneci_bot',
      'e\-SocietyRobot' => 'e-SocietyRobot',
      'Gaisbot'         => 'Gaisbot',
      'GurujiBot'       => 'GurujiBot',
      'ICC\-Crawler'    => 'ICC-Crawler',
      'InfoSeek Sidewinder' => 'InfoSeek Sidewinder',
      'IDBot'         => 'IDBot',
      'IRLbot'        => 'IRLbot',
      'Jyxobot'       => 'Jyxobot',
      'Keywalkerbot'  => 'Keywalkerbot',
      'Modiphibot'    => 'Modiphibot',
      'MaplogCrawler' => 'MaplogCrawler',
      'MaSagool'      => 'MaSagool',
      'MJ12bot'       => 'MJ12bot',
      'multicrawler'  => 'multicrawler',
      'NaverBot'      => 'NaverBot',
      'Nutch'         => 'Nutch',
      'mootercrawler' => 'mootercrawler',
      'psbot'         => 'psbot',
      'RedBot'        => 'RedBot',
      'Robo Crawler'  => 'Robo Crawler',
      'Snapbot'       => 'Snapbot',
      'SnapPreviewBot' => 'SnapPreviewBot',
      'Steeler'       => 'Steeler',
      'SurveyBot'     => 'SurveyBot',
      'YodaoBot'      => 'YodaoBot',
      'TurnitinBot'   => 'TurnitinBot',
      'Vajradhara'    => 'Vajradhara',
      'WebGenBot'     => 'WebGenBot'
    );

    /**
     * コンストラクタ
     *
     * @access public
     * @param string ユーザーエージェント
     */
    function SyL_UtilUserAgent($user_agent='')
    {
        if ($user_agent) {
            $this->user_agent = $user_agent;
        } else {
            if (isset($_SERVER['HTTP_USER_AGENT'])) {
                $this->user_agent = $_SERVER['HTTP_USER_AGENT'];
            } else {
                trigger_error("[SyL error] `HTTP_USER_AGENT' not found", E_USER_ERROR);
            }
        }
    }

    /**
     * モバイル判定を行う
     *
     * @access public
     * @return bool true: モバイル、false: モバイル以外
     */
    function isMobile()
    {
        return ($this->getCarrier() !== null);
    }

    /**
     * ロボット判定を行う
     *
     * @access public
     * @return bool true: ロボット、false: ロボット以外
     */
    function isRobot()
    {
        return ($this->getRobot() !== null);
    }

    /**
     * プラットフォーム環境（OS）を取得する
     *
     * @access public
     * @return string プラットフォーム環境（OS）
     */
    function getPlatform()
    {
        return $this->getUserAgent($this->platforms);
    }

    /**
     * ブラウザ名を取得する
     *
     * @access public
     * @return string ブラウザ名
     */
    function getBrowser()
    {
        return $this->getUserAgent($this->browsers);
    }

    /**
     * キャリア名を取得する
     *
     * @access public
     * @return string キャリア名
     */
    function getCarrier()
    {
        return $this->getUserAgent($this->carriers);
    }

    /**
     * モバイルの機種名を取得する
     *
     * @access public
     * @return string モバイルの機種名
     */
    function getMobileModel()
    {
        switch ($this->getCarrier()) {
        case 'DoCoMo':
            // 例）DoCoMo/1.0/D505i/c20/TB/W20H10/serNMAIA000001
            if (preg_match('/^DoCoMo\/\d\.0\/([^\/]+)\/(.*)$/i', $this->user_agent, $matches)) {
                return $matches[1];
            // 例）DoCoMo/2.0 F900i(c100;TC;W22H12)
            } else if (preg_match('/^DoCoMo\/\d\.0[ ]*([^\(]+)(.*)$/i', $this->user_agent, $matches)) {
                return $matches[1];
            }
            break;
        case 'KDDI(au)':
            // 例）KDDI-HI32 UP.Browser/6.2.0.6.2 (GUI) MMP/2.0
            //     UP.Browser/3.04-ST13 UP.Link/3.4.5.9
            if (preg_match('/^([^\-]+)\-([^ ]+) (.+)$/i', $this->user_agent, $matches)) {
                return $matches[2];
            }
            break;
        case 'SoftBank':
            if (isset($_SERVER['HTTP_X_JPHONE_MSNAME'])) {
                return $_SERVER['HTTP_X_JPHONE_MSNAME'];
            } else if (preg_match('/^MOT\-(C|V)980.+$/i', $this->user_agent, $matches)) {
                switch ($matches[1]) {
                case 'C': return '702sMO';
                case 'V': return '702MO';
                }
            // 例）SoftBank/1.0/910T/TJ001/SN*** Browser/NetFront/3.3 Profile/MIDP-2.0 Configuration/CLDC-1.1
            } else if (preg_match('/^([^\/]+)\/([^\/]+)\/([^\/]+)\/?(.*)$/i', $this->user_agent, $matches)) {
                return $matches[3];
            }
            break;
        case 'WILLCOM':
            // 例）Mozilla/4.0 (compatible; MSIE 4.01; Windows CE; SHARP/WS003SH; PPC; 480x640)
            // 例）Mozilla/3.0(WILLCOM;SANYO/WX310SA/2;1/1/C128) NetFront/3.3
            if (strpos($this->user_agent, 'PDXGW') !== false) {
                return 'EDGE LINK';
            } else if (preg_match('/^Mozilla\/\d\.0([^\/]+)\/([^;\/]+);(.*)$/i', $this->user_agent, $matches)) {
                return $matches[2];
            } else if (preg_match('/^Mozilla\/\d\.0([^\/]+)\/([^\/]+)\/(.*)$/i', $this->user_agent, $matches)) {
                return $matches[2];
            }
            break;
        }
        return null;
    }

    /**
     * ロボット名を取得する
     *
     * @access public
     * @return string ロボット名
     */
    function getRobot()
    {
        return $this->getUserAgent($this->robots);
    }

    /**
     * 個別ユーザーエージェント取得ロジック
     *
     * @access private
     * @param array 個別ユーザーエージェント判定配列
     * @return string 個別ユーザーエージェント
     */
    function getUserAgent($env)
    {
        foreach ($env as $key => $value) {
            if (preg_match("/{$key}/i", $this->user_agent, $matches)) {
                for ($i=1; $i<count($matches); $i++) {
                    $value = str_replace('{$'.$i.'}', $matches[$i], $value);
                }
                return $value;
            }
        }
        return null;
    }
}
