<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_XmlReader.php,v 1.1 2009/01/11 05:34:31 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * XMLパーサークラス
 */
require_once 'SyL_XmlParser.php';

/**
 * XMLリーダクラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_XmlReader.php,v 1.1 2009/01/11 05:34:31 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_XmlReader extends SyL_XmlParser
{
    /**
     * XMLのタグの階層
     *
     * @access private
     * @var int
     */
    var $depth = 0;
    /**
     * タグ有効カレント階層
     *
     * @access private
     * @var int
     */
    var $depth_row = 0;
    /**
     * 現在XML解析中のパス
     *
     * @access private
     * @var array
     */
    var $current_path = array();
    /**
     * XMLのタグの属性値
     *
     * @access private
     * @var array
     */
    var $attribute = array();
    /**
     * XMLのタグの値
     *
     * @access private
     * @var string
     */
    var $text = null;
    /**
     * 表示文字コード
     *
     * @access private
     * @var string
     */
    var $client_encode = '';

    /**
     * コンストラクタ
     *
     * @access public
     */
    function SyL_XmlReader()
    {
        $this->client_encode = mb_internal_encoding();
    }

    /**
     * スクリプトのエンコードをセット
     *
     * @access public
     * @param string エンコード名
     */
    function setClientEncode($client_encode)
    {
        $this->client_encode = $client_encode;
    }

    /**
     * エレメント開始の実行処理
     *
     * @access public
     * @param resource XMLパーサー
     * @param string エレメント名
     * @param array エレメントの属性配列
     */
    function startElement(&$parser, $name, $attr)
    {
        // 文字コード変換
        $name = $this->convertEencoding($name);
        // 属性値取得
        $attribute = array();
        foreach ($attr as $key => $value) {
            $key   = $this->convertEencoding($key);
            $value = $this->convertEencoding($value);
            $attribute[$key] = $value;
        }

//echo "PATH - " . $name . " - " . $this->depth . " - " .$this->depth_row . "<br>";

        // 保存パスに1階層追加
        $this->current_path[$this->depth] = $name;
        // 保存属性に1階層追加
        $this->attribute[$this->depth] = $attribute;
        // 1階層追加
        $this->depth++;
    }

    /**
     * エレメント終了の実行処理
     *
     * @access public
     * @param resource XMLパーサー
     * @param string エレメント名
     */
    function endElement(&$parser, $name)
    {
//echo "EE - " . $this->depth . " - " . $this->depth_row . " - " . $this->getCurrentPath() . "\n";
        if ($this->text !== null) {
            $this->getElement($this->getCurrentPath(), $this->attribute[count($this->attribute)-1], $this->text);
        }
        // 今までの階層を全要素取得
        $cnt = count($this->attribute);
        $this->getDepthElements($cnt);

        // 1階層削除
        $this->depth--;
        $this->depth_row--;
        // 保存パスに1階層削除
        unset($this->current_path[$this->depth]);
        // 保存属性に1階層削除
        unset($this->attribute[$this->depth]);
        // Textを初期化する
        $this->text = null;
    }

    /**
     * キャラクタエレメントの実行処理
     *
     * @access public
     * @param resource XMLパーサー
     * @param string 文字列
     */
    function characters(&$parser, $text)
    {
        if (trim($text) !== '') {
            if ($this->text === null) {
                // 今までの階層を最下層以外全要素取得
                $cnt = count($this->attribute) - 1;
//echo $this->depth . " - " . $this->depth_row . " - " . $this->getCurrentPath() . " '$text' - $cnt \n";
                $this->getDepthElements($cnt);
                $this->depth_row++;
            }
            // 文字コード変換
            $this->text .= $this->convertEencoding($text);
        }
    }

    /**
     * 要素取得
     *
     * @access abstract
     * @param string 現在XML解析中のパス
     * @param string XMLのタグの属性値
     * @param string XMLのタグの値
     */
    function getElement($current_path, $attribute, $text) {}

    /**
     * 今までの階層を全要素取得
     *
     * @access private
     * @param int 階層数
     */
    function getDepthElements($cnt)
    {
        $tmp_path = '';
        for ($i=0; $i<$cnt; $i++) {
            $tmp_path .= '/' . $this->current_path[$i];
            if ($this->depth_row <= $i) {
                $this->getElement($tmp_path, $this->attribute[$i], null);
                $this->depth_row++;
            }
        }
    }

    /**
     * 現在XML解析中のパスを取得
     *
     * @access public
     * @return string 現在XML解析中のパス
     */
    function getCurrentPath()
    {
        return '/' . implode('/', $this->current_path);
    }

    /**
     * エンコーディング変換
     *
     * @access public
     * @param string エンコーディング変換変換前文字列
     * @return string エンコーディング変換変換後文字列
     */
    function convertEencoding($value)
    {
        if ($this->client_encode != $this->xml_encode) {
            return mb_convert_encoding($value, $this->client_encode, $this->xml_encode);
        } else {
            return $value;
        }
    }
}
