/********************************************************************/
/* Copyright (c) 2019 System fugen G.K. and Yuzi Mizuno          */
/* All rights reserved.                                             */
/********************************************************************/
#ifndef _MGCCisect_list_HH_
#define _MGCCisect_list_HH_
/** @file */
/** @addtogroup IsectContainer
 *  @{
 */

#include <list>
#include "mg/CCisect.h"

//Forward class declaration.
class MGCurve;

/// Defines a list of MGCCisect(curve to curve intersection).

///Used to represent Intersection points of two curves.
class MG_DLL_DECLR MGCCisect_list{
//We cannot use inheritance of std::list<MYELM> to make DLL.

public:

	using MYELM = MGCCisect;
	using MYLIST = std::list<MYELM>;

	typedef MYLIST::reference              reference;
	typedef MYLIST::const_reference        const_reference;
	typedef MYLIST::iterator               iterator;
	typedef MYLIST::const_iterator         const_iterator;
	typedef MYLIST::size_type              size_type;
	typedef MYLIST::difference_type        difference_type;
	typedef MYLIST::value_type             value_type;
	typedef MYLIST::allocator_type         allocator_type;
	typedef allocator_type::pointer       pointer;
	typedef allocator_type::const_pointer const_pointer;
	typedef MYLIST::reverse_iterator       reverse_iterator;
	typedef MYLIST::const_reverse_iterator const_reverse_iterator;

	MYLIST m_list;

	/////MYLIST's member function/////

	iterator begin() { return m_list.begin(); };
	const_iterator begin()const { return m_list.begin(); };
	iterator end() { return m_list.end(); };
	const_iterator end()const { return m_list.end(); };
	reverse_iterator rbegin() noexcept { return m_list.rbegin(); };
	reverse_iterator rend() noexcept { return m_list.rend(); };
	const_reverse_iterator rbegin() const noexcept { return m_list.rbegin(); };
	const_reverse_iterator rend() const noexcept { return m_list.rend(); };
	void clear() { m_list.clear(); };
	template <class... Args>
	void emplace_back(Args&&... args) { m_list.emplace_back(std::forward<Args>(args)...); };
	bool empty() const { return m_list.empty(); };
	iterator erase(iterator x) { return m_list.erase(x); };
	iterator erase(iterator first, iterator last) { return m_list.erase(first, last); };
	const MYELM& front() const { return m_list.front(); };
	MYELM& front() { return m_list.front(); };
	const MYELM& back() const { return m_list.back(); };
	MYELM& back() { return m_list.back(); };
	iterator insert(const_iterator it, const MYELM& x) { return m_list.insert(it, x); };
	iterator insert(iterator it, MYELM&& x) { return m_list.insert(it, std::move(x)); };
	template <class InputIterator>
	iterator insert(const_iterator it, InputIterator first, InputIterator last) {
		return m_list.insert(it, first, last);
	};
	void pop_back() { m_list.pop_back(); };
	void pop_front() { m_list.pop_front(); };
	void push_back(const MYELM& x) { m_list.push_back(x); };
	void push_front(const MYELM& x) { m_list.push_front(x); };
	void push_back(MYELM&& x) { m_list.push_back(std::move(x)); };
	void push_front(MYELM&& x) { m_list.push_front(std::move(x)); };
	size_t size() const { return m_list.size(); };
	void sort() { m_list.sort(); };

///String stream Function
MG_DLL_DECLR friend std::ostream& operator<< (std::ostream&, const MGCCisect_list& );

/// Constructor
explicit MGCCisect_list(const MGCurve *c1=NULL, const MGCurve *c2=NULL);

////////////Member Function.////////////

/// Adds the MGCCisect to the end of the list.
void append(
	const MGCCisect& isect///<isect to append.
);

/// _̑SẴR|[lgw肵āC_Xgɒǉ
///Add one intersection point to the list.
void append(
	const MGPosition& point,	///<Intesection point(x,y,z)
	double t1,					///<parameter value of curve 1.
	double t2,					///<parameter value of curve 2.
	const MGCCRELATION r1=MGCCREL_UNKNOWN///<Input relation
);

/// Adds the MGCCisect_list to the end of the list.
void append(const MGCCisect_list& lst);

///Return the pointer to curve1.
const MGCurve* curve1() const { return m_curve1; };

///Return the pointer to curve2.
const MGCurve* curve2() const { return m_curve2; };

/// Return the number of items that are in the list.
int entries() const{return int(size());};

/// Return(but does not remove) first element in the list.
/// If list is empty, behavior is undefined.
const MGCCisect& first() const{return front();};

///Return true (1) if there are no items in the list,
/// false(0) otherwise.
bool isEmpty() const{return empty();};

/// Adds the MGCCisect to the beginning of the list.
void prepend(const MGCCisect& isect){push_front(isect);};
//void push_front(const MGCCisect& isect){push_front(isect);};

///Remove the first MGCCisect int the list and return the MGCCisect.
///If i is not valid, behavior is undefined.
MGCCisect removeFirst();

///Remove the last MGCCisect in the list and return the MGCCisect.
///If i is not valid, behavior is undefined.
MGCCisect removeLast();

///Replace first and second order of curve 1 and 2.
MGCCisect_list& replace12() ;

private:
	const MGCurve* m_curve1;	///< Curve 1.
	const MGCurve* m_curve2;	///< Curve 2.
	double m_error;			///< Square of Tolerance in parameter space.

};

/** @} */ // end of IsectContainer group
#endif
