/********************************************************************/
/* Copyright (c) 2019 System fugen G.K. and Yuzi Mizuno          */
/* All rights reserved.                                             */
/********************************************************************/
#ifndef _MGCSisect_list_HH_
#define _MGCSisect_list_HH_

/** @addtogroup IsectContainer
 *  @{
 */

#include <list>
#include "mg/CSisect.h"

//Forward class declaration.
class MGCurve;
class MGFSurface;

/// MGCSisect_list defines linked list of MGCSisect.

/// Used to represent Intersection points of a curve and a surface.
class MG_DLL_DECLR MGCSisect_list{
//We cannot use inheritance of std::list<MYELM> to make DLL.

public:
	using MYELM = MGCSisect;
	using MYLIST = std::list<MYELM>;

	typedef MYLIST::reference              reference;
	typedef MYLIST::const_reference        const_reference;
	typedef MYLIST::iterator               iterator;
	typedef MYLIST::const_iterator         const_iterator;
	typedef MYLIST::size_type              size_type;
	typedef MYLIST::difference_type        difference_type;
	typedef MYLIST::value_type             value_type;
	typedef MYLIST::allocator_type         allocator_type;
	typedef allocator_type::pointer       pointer;
	typedef allocator_type::const_pointer const_pointer;
	typedef MYLIST::reverse_iterator       reverse_iterator;
	typedef MYLIST::const_reverse_iterator const_reverse_iterator;

	MYLIST m_list;

	/////MYLIST's member function/////

	iterator begin() { return m_list.begin(); };
	const_iterator begin()const { return m_list.begin(); };
	iterator end() { return m_list.end(); };
	const_iterator end()const { return m_list.end(); };
	reverse_iterator rbegin() noexcept { return m_list.rbegin(); };
	reverse_iterator rend() noexcept { return m_list.rend(); };
	const_reverse_iterator rbegin() const noexcept { return m_list.rbegin(); };
	const_reverse_iterator rend() const noexcept { return m_list.rend(); };
	void clear() { m_list.clear(); };
	bool empty() const { return m_list.empty(); };
	template <class... Args>
	void emplace_back(Args&&... args) { m_list.emplace_back(std::forward<Args>(args)...); };
	iterator erase(iterator x) { return m_list.erase(x); };
	iterator erase(iterator first, iterator last) { return m_list.erase(first, last); };
	const MYELM& front() const { return m_list.front(); };
	MYELM& front() { return m_list.front(); };
	const MYELM& back() const { return m_list.back(); };
	MYELM& back() { return m_list.back(); };
	iterator insert(const_iterator it, const MYELM& x) { return m_list.insert(it, x); };
	iterator insert(iterator it, MYELM&& x) { return m_list.insert(it, std::move(x)); };
	template <class InputIterator>
	iterator insert(const_iterator it, InputIterator first, InputIterator last) {
		return m_list.insert(it, first, last);
	};
	void pop_back() { m_list.pop_back(); };
	void pop_front() { m_list.pop_front(); };
	void push_back(const MYELM& x) { m_list.push_back(x); };
	void push_front(const MYELM& x) { m_list.push_front(x); };
	void push_back(MYELM&& x) { m_list.push_back(std::move(x)); };
	void push_front(MYELM&& x) { m_list.push_front(std::move(x)); };
	size_t size() const { return m_list.size(); };
	void sort() { m_list.sort(); };

///String stream Function
MG_DLL_DECLR friend std::ostream& operator<< (std::ostream&, const MGCSisect_list& );

////////////// Constructor////////////
explicit MGCSisect_list(const MGCurve *crv=nullptr, const MGFSurface *srf=nullptr);

////////////// Member Function. ////////////

void append(const MGCSisect& isect);

/// SẴR|[lgw肵Č_𐶐
void append(
		const MGPosition& point,		///<intersection point.
		double t,				///<Curve's parameter value.
        const MGPosition& uv,	///<Surface's parameter values.
		const MGCSRELATION rl=MGCSREL_UNKNOWN
								///<Curve and Surface relation
	);

void append(const MGCSisect_list& list);

///Return the pointer to the curve.
const MGCurve* curve() const {return m_curve;};

/// Return the number of items that are in the list.
int entries()const{	return (int)size();};

/// Return(but does not remove) first element in the list.
/// If list is empty, behavior is undefined.
const MGCSisect& first() const{return front();};

///Return true (1) if there are no items in the list,
/// false(0) otherwise.
int isEmpty() const{return empty();};

/// Return(but does not remove) last element in the list.
/// If list is empty, behavior is undefined.
const MGCSisect& last() const{return back();};

///Remove the parameter and return the parameter. If i is not valid, 
/// behavior is undefined.
MGCSisect removeAt(iterator i);

///Remove the first MGCSisect int the list and return the MGCSisect.
///If i is not valid, behavior is undefined.
MGCSisect removeFirst();

///Remove the last MGCSisect in the list and return the MGCSisect.
///If i is not valid, behavior is undefined.
MGCSisect removeLast();

///Return the pointer to the surface.
const MGFSurface* surface() const {return m_surface;};

private:
	const MGCurve *m_curve;		///< Curve.
	const MGFSurface *m_surface;	///< Surface.
	double m_errort;///<error to regard same curve point in parameter space.
	double m_erroru;///<error to regard same surface point in u-parameter space.
	double m_errorv;///<error to regard same surface point in v-parameter space.

};

/** @} */ // end of IsectContainer group
#endif
