/********************************************************************/
/* Copyright (c) 2019 System fugen G.K. and Yuzi Mizuno          */
/* All rights reserved.                                             */
/********************************************************************/
#ifndef _MGInterval_HH_
#define _MGInterval_HH_
#include "mg/EReal.h"
/** @file */
/** @addtogroup BASE
 *  @{
 */

//  MGInterval.h
//  header for class MGInterval
// Forward Declaration
class MGIfstream;
class MGOfstream;

/// Interval of 1 dimension, i.e. MGInterval is a real line.

///  ̋Ԃ\BQ MGEReal ŕ\B
class MG_DLL_DECLR MGInterval{

public:

///+operator.
MG_DLL_DECLR friend MGInterval operator+ (double, const MGInterval& );

///Scalar multiplication to the interval.
///The mid_point of the interval does not change. The width of the interval
///is widened(narrowed) by the multiplication of the value a.
MG_DLL_DECLR friend MGInterval operator* (double a, const MGInterval& );

///Comparison operator.
MG_DLL_DECLR friend bool operator>(const MGEReal&, const MGInterval&);
MG_DLL_DECLR friend bool operator<(const MGEReal&, const MGInterval&);
MG_DLL_DECLR friend bool operator>=(const MGEReal&, const MGInterval&);
MG_DLL_DECLR friend bool operator<=(const MGEReal&, const MGInterval&);
MG_DLL_DECLR friend bool operator>(double, const MGInterval& i);
MG_DLL_DECLR friend bool operator<(double, const MGInterval& i);

///String stream function
MG_DLL_DECLR friend std::ostream& operator<< (std::ostream&, const MGInterval&);

//////////////Constructor////////////

/// void Constructor
MGInterval();

///Construct interval from interval type and a point.
///When the type is MGINTERVAL_FINITE, start and end is the point.
///MGINTERVAL_TYPE:MGINTERVAL_EMPTY, MGINTERVAL_FINITE,
///MGINTERVAL_FINITE_ABOVE, MGINTERVAL_FINITE_BELOW, or MGINTERVAL_INFINITE.
///MGINTERVAL_FINITE_ABOVE is infinite below and finite above.
///MGINTERVAL_FINITE_BELOW is infinite above and finite below.
/// C^[o^Cvƈ_w肵ăC^[o𐶐B
///     MGINTERVAL_FINITE w莞 w肳ꂽ_̓_͈͂̃C^[o
///     MGINTERVAL_FINITE_ABOVE, MGINTERVAL_FINITE_BELOW w莞 
///     L̈_w
///     MGINTERVAL_INFINITE w莞 double ̒l̎w͕KvȂ
explicit MGInterval(MGINTERVAL_TYPE, double=0.0);

///Construct interval from two MGEReal points.
///t1 is lower limit, and t2 is upper limit.
///When t1> t2, the interval is empry.
/// QMGEReal̒lC^[o𐶐B
///***** This is the fundamental constructor.
MGInterval(const MGEReal& t1, const MGEReal& t2); 

/// Construct an interval which contains both input interval and a value.
MGInterval(const MGInterval&, const MGEReal&); 

///Construct a interval of one point.
explicit MGInterval(double t);

////////////Operator overload////////////

///Return low or high point.
///0<=i<=1, and for i=0, return low point, for i=1, return high point.
const MGEReal& operator() (int i)const;
MGEReal& operator() (int i);
const MGEReal& operator[] (int i)const;
MGEReal& operator[] (int i);

///Addition of two intervals. Generated interval is the minimum one
///that includes both intervals. The same as operator| .
///  g̃C^[oƗ^ꂽC^[ỏZsIuWFNg
///  𐶐B
MGInterval operator+ (const MGInterval& ) const;

///Translation of the interval.
///  g̃C^[o double ̉ZsuWFNgB
MGInterval operator+ (double) const;

///Addition of two intervals. Updated interval is the minimum one
///that includes both original intervals. The same as operator|=. 
///  g̃C^[oɉZg̃C^[oƂB
MGInterval& operator+= (const MGInterval& );

///Translation of the interval.
MGInterval& operator+= (double);

///Unary minus.
///  OuP}CiXBIuWFNg𐶐B
MGInterval operator- () const;

///Subtraction. Subtraction is defined as:
/// "this" minus the common of the two intervals. 
///  g̃C^[oƗ^ꂽC^[ǒZsIuWFNg
///  𐶐B
MGInterval operator- (const MGInterval& ) const ;

///Translation of the interval.
///  g̃C^[o double ̌ZsuWFNgB
MGInterval operator- (double) const;

///Subtraction. Subtraction is defined as:
/// "this" minus the common of the two intervals. 
///  g̃C^[oɌZg̃C^[oƂB
MGInterval& operator-= (const MGInterval& );

///Translation of the interval.
MGInterval& operator-= (double);

///Scalar multiplication to the interval.
///The mid_point of the interval does not change. The width of the interval
///is widened(narrowed) by the multiplication of the value a.
///  XJ[̏ZsIuWFNg𐶐B
MGInterval operator* (double a) const;

///Scalar multiplication to the interval.
///  XJ[̏Zsg̃C^[oƂB
///The mid_point of the interval does not change. The width of the interval
///is widened(narrowed) by the multiplication of the value a.
MGInterval& operator*= (double a);

///Scalar division to the interval.
///  XJ[̏ZsIuWFNg𐶐B
///The mid_point of the interval does not change. The width of the interval
///is widened(narrowed) by the division of the value a.
MGInterval operator/ (double a) const; 

///Scalar division to the interval.
///  XJ[̏Zsg̃C^[oƂB
///The mid_point of the interval does not change. The width of the interval
///is widened(narrowed) by the division of the value a.
MGInterval& operator/= (double a);

///Addition of two intervals. Generated interval is the minimum one
///that includes both intervals. The same as operator+ .
///  g̃C^[oƗ^ꂽC^[oORC^[o
///  B
MGInterval operator| (const MGInterval&) const ;

///Addition of two intervals. Updated interval is the minimum one
///that includes both original intervals. The same as operator+=. 
///g̃C^[oƗ^ꂽC^[oORĎg̃C^[o
///ƂB
MGInterval& operator|= (const MGInterval&);

///And operation. Generate the interval common to the both intervals.
///g̃C^[oƗ^ꂽC^[őʕ̃C^[o
///B
MGInterval operator& (const MGInterval&) const ;

///And operation. Update to the interval common to the both intervals.
///  g̃C^[oƗ^ꂽC^[oʕg̃C^[o
///  ƂB
MGInterval& operator&= (const MGInterval&);

///  Boolean Z

///Test if the two intervals have common part.
///  g̃C^[oƗ^ꂽC^[oʕĂ邩
///  ǂԋpB
bool operator&& (const MGInterval&)const;

///Test if this includes intrval2. 
///  g̃C^[o empty ̏ꍇ false(0) ԋpA^ꂽC^[o
///   empty ̏ꍇ true(1) ԋpB^ꂽC^[ỏ[g
///  ̃C^[ỏ[傫A^ꂽC^[ȍ[g
///  C^[ȍ[ true(1) ԋpB
///  ܂ÃC^[o empty ̎ true(1) ԋpB
bool operator>> (const MGInterval& intrval2) const;

///Test if this includes the value t.
///  ^ꂽlg̃C^[oɂ邩ǂԋpB܂܂
///   true(1) ԋpBg̃C^[o empty ̏ꍇ false(0) 
///  ԋpB
//bool operator>> (const MGEReal& t) const;

///Test if this is included in intrval2.
///  g̃C^[o empty ̏ꍇ true(1) ԋpA^ꂽC^[o
///   empty ̏ꍇ false(0) ԋpB^ꂽC^[ỏ[g
///  ̃C^[ỏ[菬A^ꂽC^[ȍ[g
///  C^[ȍ[傫 true(1) ԋpB
///  ܂ÃC^[o empty ̎ false(0) ԋpB
bool operator<< (const MGInterval& intrval2) const;

///Test if this excludes the value t.
///  ^ꂽlg̃C^[o͈͊Oɂ邩ǂԋpB͈͊O
///   true(1) ԋpBg̃C^[o empty ̏ꍇ true(1) 
///  ԋpB
//bool operator<< (const MGEReal& t) const;

///Test if two intervals are equal(tolerance included)
///  Q̃C^[oꂩǂ𔻒肷B
///  ł鎞 true(1) ԋp
bool operator== (const MGInterval&) const;

///Test if two intervals are not equal(tolerance included)
///  Q̃C^[oꂩǂ𔻒肷B
///  łȂ  true(1) ԋp
bool operator!= (const MGInterval&) const;

///Arithmatic comparison operation:ZpIr
bool operator< (const MGInterval&) const;
bool operator> (const MGInterval&) const;
bool operator<= (const MGInterval&) const;
bool operator>= (const MGInterval&) const;

bool operator< (const MGEReal& ) const;
bool operator> (const MGEReal&) const;
bool operator<= (const MGEReal&) const;
bool operator>= (const MGEReal&) const;

///Return true if m_high<t;
bool operator< (double t) const;

///Return true if m_low>t;
bool operator> (double t) const;

///Chnage the interval range to [t0, t1] if t0<t1,
///                          to [t1, t0] if t1<t0.
void change_range(
	double t0,	///<new range, start.
	double t1	///<new range, end.
);

/// Reference
///Test if empty:C^[o empty ǂ肷B
///***Interval of m_low==m_high(including error) is not empty.***
bool empty() const;

///Expand the interval so that this includes the doube val.
void expand(const MGEReal& val);

///Test if finite.
///  C^[oLǂ肷B
bool finite() const;

///Test if finite above(infinite below).
///  C^[oLǂ肷B
bool finite_above() const;
 
///Test if finite below(infinite above).
///  C^[oLǂ肷B
bool finite_below() const;

///Get the high value.
const MGEReal& high() const {return m_high;};
MGEReal& high() {return m_high;};

///Return high point of the interval. When infinite_above(),
///mgInfiniteVal will be returned.@When empty, return 0.
///lԋpB
double high_point() const;

///Test if this finite interval includes the value,
///taking the relative error of the interval into the account.
bool includes(double t)const;

///Test if infinite.
///  C^[oǂ肷B
bool infinite() const;

///Test if infinite_above.
bool infinite_above() const;

///Test if infinite_below.
bool infinite_below() const;

///Compute interpolated point of the interval. Given parameter t,
///the interpolated point=(1-t)*low_point()+t*high_point().
///t may be t<0 or t>1. Valid only when finite().
///  C^[o̕ԂԋpB^ꂽ param ɑ΂āA
///  (1-t)*low_point()+t*high_point() ԋpB
///  ( 0 > t, 1 < t ̎͊O )
double interpolate(double t) const;

///Test if this is empty interval.
bool is_null() const{return empty();};

///Compute the length of the interval.
///When empty, length()<=0.0.
///[Ɖ[̍قԋpBempty ͔񐳒li[܂͕̒lj
///ԋpB
MGEReal length() const;

///Get the low value.
const MGEReal& low() const {return m_low;};
MGEReal& low() {return m_low;};

///Return low point of the interval. When infinite_below(),
/// -mgInfiniteVal will be returned. When empty, return 0.
///  C^[oL̎LŉlԋpB̎ 0.0 
///  ԋpB
double low_point() const;

///Return mid point of the interval. Valid only when finite().
///  C^[oL̎LŒ_ԋpB
double mid_point() const;

///Compute relative_error*length();
double relative_error() const;

///Round the input value into this interval's range, that is:
///round_into_interval(t)=(*this)[0] if t<=(*this)[0],
///round_into_interval(t)=(*this)[1] if t>=(*this)[1].
double round_into_interval(double t)const;

///Set the interval length to len so that the midpoint does not change.
void set_length(double len);

///Set this as null(empty) interval
void set_null(double t=0.);
void set_empty(double t=0.){set_null(t);};

///Update high point of the interval. When given point is less than
///low point, the interval becomes empty.
///  g̃C^[ȍlύXB^ꂽll菬
///  ꍇAg̃C^[o empty ɂȂA㉺l̓ꊷ͂ȂB
void set_high_point(const MGEReal&);
void set_high(const MGEReal& t){set_high_point(t);};

///Update low point of the interval. When given point is greater than
///high point, the interval becomes empty.
///  g̃C^[ỏlύXB^ꂽll傫
///  ꍇAg̃C^[o empty ɂȂA㉺l̓ꊷ͂ȂB
void set_low_point(const MGEReal&);
void set_low(const MGEReal& t){set_low_point(t);};

///Return interval type.
///  C^[õ^CvԋpB
MGINTERVAL_TYPE type() const;

///Dump Functions.
///Calculate dump size
int dump_size() const;

///Dump Function
int dump(MGOfstream& ) const;

///Restore Function
int restore(MGIfstream& );

private:

//////////// Member data.////////////

	///  Two points on real line. ̂Q_ 
	MGEReal m_high;	///<larger one.
	MGEReal m_low;	///<smaller one.

};

/** @} */ // end of BASE group
#endif
