/********************************************************************/
/* Copyright (c) 2019 System fugen G.K. and Yuzi Mizuno          */
/* All rights reserved.                                             */
/********************************************************************/
#ifndef _MGSSisect_list_HH_
#define _MGSSisect_list_HH_
/** @file */
/** @addtogroup IsectContainer
 *  @{
 */

#include <list>
#include "mg/SSisect.h"

//Forward class declaration.
class MGFSurface;

/// MGSSisect_list defines linked list of MGSSisect.

/// This is value based list.
/// Used to represent intersection lines of two surfaces.
class MG_DLL_DECLR MGSSisect_list{
//We cannot use inheritance of std::list<MYELM> to make DLL.

public:
	using MYELM = MGSSisect;
	using MYLIST = std::list<MYELM>;

	typedef MYLIST::reference              reference;
	typedef MYLIST::const_reference        const_reference;
	typedef MYLIST::iterator               iterator;
	typedef MYLIST::const_iterator         const_iterator;
	typedef MYLIST::size_type              size_type;
	typedef MYLIST::difference_type        difference_type;
	typedef MYLIST::value_type             value_type;
	typedef MYLIST::allocator_type         allocator_type;
	typedef allocator_type::pointer       pointer;
	typedef allocator_type::const_pointer const_pointer;
	typedef MYLIST::reverse_iterator       reverse_iterator;
	typedef MYLIST::const_reverse_iterator const_reverse_iterator;

	MYLIST m_list;

	iterator begin() { return m_list.begin(); };
	const_iterator begin()const { return m_list.begin(); };
	iterator end() { return m_list.end(); };
	const_iterator end()const { return m_list.end(); };
	reverse_iterator rbegin() noexcept { return m_list.rbegin(); };
	reverse_iterator rend() noexcept { return m_list.rend(); };
	const_reverse_iterator rbegin() const noexcept { return m_list.rbegin(); };
	const_reverse_iterator rend() const noexcept { return m_list.rend(); };
	void clear() { m_list.clear(); };
	template <class... Args>
	void emplace_back(Args&&... args) { m_list.emplace_back(std::forward<Args>(args)...); };
	bool empty() const { return m_list.empty(); };
	iterator erase(iterator x) { return m_list.erase(x); };
	iterator erase(iterator first, iterator last) { return m_list.erase(first, last); };
	const MYELM& front() const { return m_list.front(); };
	MYELM& front() { return m_list.front(); };
	const MYELM& back() const { return m_list.back(); };
	MYELM& back() { return m_list.back(); };
	iterator insert(const_iterator it, MYELM&& x) { return m_list.insert(it, std::move(x)); };
	template <class InputIterator>
	iterator insert(const_iterator it, InputIterator first, InputIterator last) {
		return m_list.insert(it, first, last);
	};
	void pop_back() { m_list.pop_back(); };
	void pop_front() { m_list.pop_front(); };
	void push_back(MYELM&& x) { m_list.push_back(std::move(x)); };//Only move operation is allowed.
	void push_front(MYELM&& x) { m_list.push_front(std::move(x)); };//Only move operation is allowed.
	size_t size() const { return m_list.size(); };
	void sort() { m_list.sort(); };

	////////Special member functions/////////
	//MGSSisect_list() = default;
	~MGSSisect_list() = default;
	MGSSisect_list(const MGSSisect_list&) = delete;
	MGSSisect_list& operator=(const MGSSisect_list&) = delete;
	MGSSisect_list(MGSSisect_list&&) = default;
	MGSSisect_list& operator=(MGSSisect_list&&) = default;

///String stream Function.
MG_DLL_DECLR friend std::ostream & operator << (std::ostream&, const MGSSisect_list& );

//////////// Constructor ////////////
explicit MGSSisect_list(const MGFSurface *s1=NULL, const MGFSurface *s2=NULL)
: m_surface1(s1), m_surface2(s2){;};


/// Adds the MGSSisect to the end of the list.

///isect transfer the ownership of the curves in isect to this list.
void append(MGSSisect&& isect);
void append(MGSSisect_list&& isectlist);

///Add one intersection line to the list.

///iline, param1, and param2 must be newed objects, and their ownership
///are transfered to MGSSisect_list.
void append(
	MGCurve* iline,
	MGCurve* param1,
	MGCurve* param2,
	const MGSSRELATION r1=MGSSREL_UNKNOWN);

///Add one intersection line to the list.

///this append copies the three curves.
void append(
	const MGCurve& iline,
	const MGCurve& param1,
	const MGCurve& param2,
	const MGSSRELATION r1=MGSSREL_UNKNOWN);

///Find where in this ssi2  have common parts (in line_zero()) in 
///their world representation.

///Fucntion's return value is the iterator of this that had the common.
///		!=end():have common part. 
///		==end():no common part(except a point) found.
iterator find_common(const MGSSisect& ssi2);

///Return the pointer to surface1.
const MGFSurface* surface1() const {return m_surface1;}

///Return the pointer to surface2.
const MGFSurface* surface2() const {return m_surface2;}

/// Return the number of items that are in the list.
int entries() const{return int(size());};

///Insert MGSSisect at the index position i.

///This position must be between zero and the number of items in the list,
/// or behavior is undefined.
void insertAt(iterator i, MGSSisect&& isect){insert(i, std::move(isect));};

///Return true (1) if there are no items in the list,
/// false(0) otherwise.
bool isEmpty() const{return empty();};

/// Return(but does not remove) lst element in the list.
/// If list is empty, behavior is undefined.
const MGSSisect& last() const{return back();};

/// Adds the MGSSisect to the beginning of the list.
///isect transfer the ownership of the curves in isect to this list.
void prepend(MGSSisect&& isect){push_front(std::move(isect));};

///Remove the MGSSisect and return the MGSSisect. If i is no valid, 
/// behavior is undefined.
MGSSisect removeAt(iterator i);

///Remove the first MGSSisect int the list and return the MGSSisect.
///If i is not valid, behavior is undefined.
MGSSisect removeFirst();

///Remove the last MGSSisect in the list and return the MGSSisect.
///If i is not valid, behavior is undefined.
MGSSisect removeLast();

///Replace first and second order of surface 1 and 2.
MGSSisect_list& replace12() ;

private:
	const MGFSurface *m_surface1;	///< Surface 1.
	const MGFSurface *m_surface2;	///< Surface 2.
};

/** @} */ // end of IsectContainer group
#endif
