/********************************************************************/
/* Copyright (c) 2019 System fugen G.K. and Yuzi Mizuno          */
/* All rights reserved.                                             */
/********************************************************************/
#ifndef _MGTrimmedCurve_HH_
#define _MGTrimmedCurve_HH_

#include <vector>
#include "mg/Default.h"
#include "mg/Interval.h"
#include "mg/Box.h"
#include "mg/Position.h"
#include "mg/Position_list.h"
#include "mg/LBRep.h"
#include "mg/RLBRep.h"
#include "mg/Ellipse.h"
#include "mg/Straight.h"
#include "mg/BSumCurve.h"

//Define MGTrimmedCurve Class.

class MGInterval;
class MGBox;
class MGVector;
class MGUnit_vector;
class MGPosition;
class MGTransf;
class MGCParam_list;
class MGStraight;
class MGEllipse;
class MGCompositeCurve;
class MGKnotVector;
class MGLBRep;
class MGRLBRep;
class MGSurface;
class MGCCisect_list;
class MGCSisect_list;
class MGIfstream;
class MGOfstream;
/** @file */
/** @addtogroup GEO
 *  @{
 */

///MGTrimmedCurve is a part of an original curve of a limitted parameter range.

///MGTrimmedCurve is a part of original curve that has limitted parameter range.
///MGTrimmedCurve is a temporal curve, and does not have update functions.
class MG_DLL_DECLR MGTrimmedCurve:public MGCurve{

public:

MG_DLL_DECLR friend MGTrimmedCurve operator+ (const MGVector& v, const MGTrimmedCurve& cv2);
MG_DLL_DECLR friend MGTrimmedCurve operator* (double scale, const MGTrimmedCurve& cv2);


////////Special member functions/////////
MGTrimmedCurve();
~MGTrimmedCurve()=default;
MGTrimmedCurve(const MGTrimmedCurve&);///<Copy constructor.
MGTrimmedCurve& operator= (const MGTrimmedCurve&);///<Copy assignment.
MGTrimmedCurve(MGTrimmedCurve&&)=default;		///<Move constructor.
MGTrimmedCurve& operator= (MGTrimmedCurve&&)=default;///<Move assignment.
MGTrimmedCurve(const MGCurve& crv, const MGInterval range);

///subcurve of the input curve crv from t1 to t2.
///t1 may be >t2. In this case, the curve will be from t2 to t1.
MGTrimmedCurve(const MGCurve& crv, double t1, double t2);

///Assignment.
///When the leaf object of this and crv2 are not equal, this assignment
///does nothing.
MGTrimmedCurve& operator=(const MGGel& gel2);
MGTrimmedCurve& operator=(MGGel&& gel2);

///Logical operator overload(_Zqd`)
/// Q̋Ȑǂr肷B
///Comparison of two curves.
bool is_same_curve(const MGCurve& curve2)const;
bool operator==(const MGTrimmedCurve& gel2)const;
bool operator==(const MGCompositeCurve& gel2)const;
bool operator==(const MGGel& gel2)const;
bool operator<(const MGTrimmedCurve& gel2)const;
bool operator<(const MGGel& gel2)const;

///Approximate this curve as a MGLBRep curve
///within the tolerance MGTolerance::line_zero().
///When parameter_normalization=0, reparameterization will not done, and
///the evaluation at the same parameter has the same values before and after
///of approximate_as_LBRep.
void approximate_as_LBRep(
	MGLBRep& lb,///<Approximated obrep will be set.
	int ordr=0,	///<new order. When this is MGLBRep, if ordr=0,
				///<ordr=order() will be assumed, else ordr=4 is assumed.
	int parameter_normalization=0,
		///<Indicates how the parameter normalization be done:
		///<  =0: no parameter normalization.
		///<  =1: normalize to range=(0., 1.);
		///<  =2: normalize to make the average length of the 1st derivative 
		///<      is as equal to 1. as possible.
	bool neglectMulti=false///<Indicates if multiple knots be kept.
		///< true: multiplicity is removed.
		///< false: multiplicity is kept.
)const override;

///Returns B-Rep Dimension.
int bdim() const;

///Return minimum box that includes the curve of parameter interval.
/// ͂̃p[^͈͂̋Ȑ͂ރ{bNXԂB
MGBox box_limitted(
	const MGInterval& ///< Parameter Range of the curve.
) const;

///Changing this object's space dimension.
void change_dimension(
	int dim,		///< new space dimension
	int start1, 	///< Destination order of new object.
	int start2 	///< Source order of this object.
){
	assert(false);//This cannot be used.
}
///Change parameter range, be able to change the direction by providing
///t1 greater than t2.
void change_range(
	double t1,	///<Parameter value for the start of original. 
	double t2	///<Parameter value for the end of original.
){
	assert(false);//The use is prohibitted.
}

///Construct new curve object by copying to newed area.
///User must delete this copied object by "delete".
MGCurve* clone() const;

///^ꂽȐƎg̋ʕ邩ǂׂB

///F
///		const MGCurve&			curve2,		(I/ )	^Ȑ
///		std::vector<double>&	vecComSpan	( /O)	ʕ̃p[^͈
///		 4n̔zŁAvecComSpan(4*i+0),vecComSpan(4*i+1)g̃p[^͈
///					(vecComSpan(4*i+0) < vecComSpan(4*i+1))A
///				 vecComSpan(4*i+2),vecComSpan(4*i+3)curve2̃p[^͈
///		MGCCisect_list&			isect		( /O)	_
///߂lF
///		3:_ʕ܂
///		2:_݂̂܂
///		1:ʕ݂̂܂
///		0:_ʕȂ
///		-1:ʃGbW̎vZG[
///		-2:ʃGbWSȏ㋁܂(̂ĂȂƌȂ)
///ǋLF
///	Ȑʂǂ̌덷ɂline_zero()Ap[^͈͂̎vZ
///	덷ɂ́Ap[^͈*rc_zero()gp
int common(
	const MGCurve& curve2,
	std::vector<double>& vecComSpan,
	MGCCisect_list& isect
) const;

///^ꂽȐƎg̋ʕ邩ǂׂB

///F
///		const MGCurve&			curve2,		(I/ )	^Ȑ
///		std::vector<double>&	vecComSpan	( /O)	ʕ̃p[^͈
///		 4n̔zŁAvecComSpan(4*i+0),vecComSpan(4*i+1)g̃p[^͈
///					(vecComSpan(4*i+0) < vecComSpan(4*i+1))A
///				 vecComSpan(4*i+2),vecComSpan(4*i+3)curve2̃p[^͈
///߂lF
///		ʕ̐:	ʕ܂
///		0:				ʕȂ
///		-1:				ʃGbW̎vZG[
///		-2:				ʃGbWSȏ㋁܂(̂ĂȂƌȂ)
///ǋLF
///	Ȑʂǂ̌덷ɂline_zero()Ap[^͈͂̎vŽ덷ɂ́A
///  p[^͈*rc_zero()gp
int common(
	const MGCurve& curve2,
	std::vector<double>& vecComSpan
) const;

///copy as a newed curve.

///The new curve will be MGLBRep or MGRLBRep.
///When original curve was a MGRLBRep, the new curve will be a MGRLBRep.
///Otherwise,  the new curve will be a MGLBRep.
///Returned object must be deleted.
MGCurve* copy_as_nurbs() const override;

///Construct new curve object by changing
///the original object's space dimension.
///User must delete this copied object by "delete".
MGCurve* copy_change_dimension(
	int sdim,			///< new space dimension
	int start1=0, 		///< Destination order of new line.
	int start2=0 		///< Source order of this line.
)const;

///Construct new curve object by copying to newed area,
///and limitting the parameter range to prange.
///Returned is newed object and must be deleted.
///Returned curve is not TrimmedCurve, but a part copy of the original
///curve.
MGCurve* copy_limitted(const MGInterval& prange) const;
MGCurve* copy_limitted() const;

/// Return parameter curve's pointer
const MGCurve* base_curve() const {return m_curve;};

///Compute curvilinear integral of the 1st two coordinates.

///This integral can be used to compute area sorounded by the curve.
///(ϕj߂B
///curvilinear_integral from t1 to t2 can be obtained by
///Integral of (x*dy-y*dx) about t, where curve is expressed by
///f(t)=(x(t),y(t)), dx=dx/dt, and dy=dy/dt.
double curvilinear_integral(double t1, double t2) const;

///Divide this curve at the designated knot multiplicity point.
///Function's return value is the number of the curves after divided.
int divide_multi(
	std::vector<UniqueCurve>& crv_list,///<divided curves are appended.
	int multiplicity=-1	///<designates the multiplicity of the knot to divide at,
						///<When multiplicity<=0, order()-1 is assumed,
						///<When multiplicity>=order(), order() is assumed.
)const override;

///Draw using vbo.
void drawSE(
	mgVBO& vbo,///<The target graphic object.
	double t0,			///<Start parameter value of the curve.
	double t1			///<End parameter value of the curve,
						///<Draw will be performed from t0 to t1.
)const;

/// Evaluate n'th derivative data. n=0 means positional data evaluation.
MGVector eval(
	double t,			///< Parameter value.
	int nderiv=0,	///< Order of Derivative.
	int left=0			///<Left continuous(left=true)
						///<or right continuous(left=false).
)const{return m_curve->eval(t,nderiv,left);};

///Extrapolate this curve by an (approximate) chord length.
///The extrapolation is C2 continuous.
void extend(
	double length,	///<approximate chord length to extend. 
	bool start=false///<Flag of which point to extend, start or end point of the line,
					///<If start is true extend on the start point.
);

/// Return This object's typeID
long identify_type() const;

///Test if input parameter value is inside parameter range of the line.
bool in_range(double t) const{return m_range.includes(t);}

///Provide divide number of curve span for function intersect.
int intersect_dnum() const;

///Test if this cure is planar or not.
///MGPlane expression will be out to plane if this is planar.
///Function's return value is true if planar.
bool is_planar(MGPlane& plane)const;

///<Test if this parameter range is the same as the original m_curve's.
bool is_same_range()const{return m_sameRange;};

///Intersection of Curve and Surface
MGCCisect_list isect(const MGCurve& curve2) const;
MGCCisect_list isect(const MGStraight& curve2)const;
MGCCisect_list isect(const MGRLBRep& curve2)const;
MGCCisect_list isect(const MGEllipse& curve2)const;
MGCCisect_list isect(const MGLBRep& curve2)const;
MGCCisect_list isect(const MGSurfCurve& curve2)const;
MGCCisect_list isect(const MGBSumCurve& curve2)const;
MGCCisect_list isect(const MGTrimmedCurve& curve2)const;
MGCCisect_list isect(const MGCompositeCurve& curve2)const;

///Intersection with a Surface
MGCSisect_list isect(const MGSurface& surf) const;
MGCSisect_list isect(const MGPlane& surf) const;
MGCSisect_list isect(const MGSphere& surf)const;
MGCSisect_list isect(const MGCylinder& surf)const;
MGCSisect_list isect(const MGSBRep& surf)const;
MGCSisect_list isect(const MGRSBRep& surf)const;
MGCSisect_list isect(const MGBSumSurf& surf)const;

///Access to i-th element of knot
double knot(int i) const;
	
///Returns the knot vector.
const MGKnotVector& knot_vector() const;

///Cmpute curve length of the interval.

///If t1 is greater than t2, return negative value.
/// ^ꂽp[^lԂ̋Ȑ̒ԂB
/// p[^ŗ^ꂽƂ͐lA~̂Ƃ͕lԂB
double length(double t1, double t2) const;

///Inverse function of length. Compute the point that is away from
///the point t by length len.
/// length̋t֐Bwp[^tŎ_w苗len
/// Ȑɉėꂽ_p[^lԂB
double length_param( double t, double len) const;

///Update this by limiting the parameter range of the curve.
/// gɎw肵p[^͈͂̂B
void limit(const MGInterval& );

///Obtain parameter value if this curve is negated by "negate()".
double negate_param(double t)const{return m_curve->negate_param(t);};

///Returns the order.
int order() const{return m_curve->order();}

/// Output function.
std::ostream& out(std::ostream&) const;

///IGES output function.
int out_to_IGES(
	MGIgesOfstream& igesfile,
	int SubordinateEntitySwitch=0
)const;

/// Return ending parameter value.
double param_e() const;

///Normalize parameter value t to the nearest knot if their distance is
///within tolerance.
double param_normalize(double t) const;

///Return parameter range of the curve(p[^͈͂Ԃ)
MGInterval param_range() const{return m_range;}

/// Return starting parameter value.
double param_s() const;
	
///Compute part of this curve from parameter t1 to t2.

///Returned is the pointer to newed object, and so should be deleted
///by calling program, or memory leaked.
MGCurve* part(
	double t1,///<target parameter range from t1,
	double t2,///< to t2.
	int multiple=0	///<Indicates if start and end knot multiplicities
					///<are necessary. =0:unnecessary, !=0:necessary.
) const;
	
///Compute all foot points of the perpendicular line from point to the curve.

/// ^|CgȐ։낵̑́CȐ̃p[^l
/// ׂċ߂B
MGCParam_list perps(
	const MGPosition& P		///<Point(w_)
)const;

///Compute all the perpendicular points of this curve and the second one.

///If f(s) and g(t) are the points of the two curves f and g,
///then obtains points where the following conditions are satisfied:
///  fs*(f-g)=0.    gt*(g-f)=0.
///Here fs and gt are 1st derivatives at s and t of f and g.
///MGPosition P in the MGPosition_list contains this and crv's parameter
///as:     P(0)=this curve's parameter, P(1)=crv2's parameter value.
MGPosition_list perps(const MGCurve& crv2)const{return perps_in_range(crv2);};
MGPosition_list perps(const MGStraight& crv2)const{return perps_in_range(crv2);};
MGPosition_list perps(const MGRLBRep& crv2)const{return perps_in_range(crv2);};
MGPosition_list perps(const MGEllipse& crv2)const{return perps_in_range(crv2);};
MGPosition_list perps(const MGLBRep& crv2)const{return perps_in_range(crv2);};
MGPosition_list perps(const MGSurfCurve& crv2)const;
MGPosition_list perps(const MGBSumCurve& crv2)const{return perps_in_range(crv2);};


///Obtain the projected curve of a curve onto the surface.

///The direction of the projection is along the vector vec if the vec is not NULL,
///and normal to the surface if the vec is NULL.
///Output of 'project' is two kind of curves:
///one is general world coordinate curves('vec_crv'), and the other is (u,v) curves of
///the parameter space of the surfaces(vec_crv_uv).
///vec_crv_uv.size() is equal to vec_crv.size(). Let the size be n, then
/// (vec_crv_uv[i], vec_crv[i]) is one pair for 0<=i<n.
///Function's return value is:
/// >=0: number of curves obtained, <0 : Some error detected.
int project(
	const MGFSurface& surf,	//given surface.
	std::vector<UniqueCurve>& vec_crv_uv,	//uv projection curve will be appended.
	std::vector<UniqueCurve>& vec_crv,	//3d projection curve will be appended.
	const MGVector& vec	//projection vector.
						//if vec = NULL then calculate perpendicular project.
)const;

///Round t into curve's parameter range.
double range(double t) const;

///Return space dimension
int sdim() const{return m_curve->sdim();};

///Return sweep surface from crv.

///Returned is a newed MGSurface, must be deleted.
///The sweep surface is defined as:
///This curve(say c(t)) is the rail and the straight line segments from
///C(t)+start_dist*uvec to C(t)+end_dist*uvec are the generatrix.
MGSurface* sweep(
	const MGUnit_vector& uvec,	///<Sweep Direction.
	double start_dist,			///<distance to start edge.
	double end_dist				///<distance to end edge.
)const;				

///Return curve type.
MGCURVE_TYPE type() const{return MGCURVE_TRIMMED;}

///Unlimit parameter range of the curve.
MGCurve& unlimit();

///Unlimit parameter range of the curve to the end point direction
MGCurve& unlimit_end();

///Unlimit parameter range of the curve to the start point direction
MGCurve& unlimit_start();

///Get the name of the class.
std::string whoami()const{return "TrimmedCurve";};

protected:

///Compute intersection point of 1D sub curve of original curve.
///Parameter values of intersection point will be returned.
MGCParam_list intersect_1D(						
	double f,			///< Coordinate value
	int coordinate=0	///< Coordinate kind of the data f(from 0).
)const;	

///Obtain so transformed 1D curve expression of this curve that
///f(t)={sum(xi(t)*g[i]) for i=0(x), 1(y), 2(z)}-g[3], where f(t) is the output
///of oneD and xi(t) is i-th coordinate expression of this curve.
///This is used to compute intersections with a plane g[4].
std::unique_ptr<MGCurve> oneD(
	const double g[4]			///<Plane expression(a,b,c,d) where ax+by+cz=d.
) const;

///of[^ǂݍފ֐
/// ߂lbool͐ɓǂݏootrueAsfalseɂȂ
void ReadMembers(MGIfstream& buf);

///of[^ފ֐
/// ߂lbool͐ɏ݂otrueAsfalseɂȂ
void WriteMembers(MGOfstream& buf) const;
	
private:

	const MGCurve* m_curve;	///<Curve pointer.
	mutable bool m_sameRange;///<Indicates if m_range is the same as the original m_curve's.
	MGInterval     m_range; ///<Parameter range of the above m_curve.
	mutable std::unique_ptr<MGKnotVector> m_knotV;///<When knot_vector() is invoked, the knot vector is set.

///Compute the box of the whole of the curve.
///Returned is a newed object pointer.
MGBox* compute_box()const;

//////////////////////////////////////////////////////////////////////////////////////
///Following functions are defined to prohibit their use in TrimmedCurve.///
//////////////////////////////////////////////////////////////////////////////////////

///Exchange ordering of the coordinates.
///Exchange coordinates (i) and (j).
void coordinate_exchange(int i, int j);

///Intersection of Curve.
MGCCisect_list isect_in_range(const MGCurve& curve2)const;

///Compute intersections with MGLBRep curve2 that does not have C0 continuity in it.
MGCCisect_list isect_withC1LB(const MGLBRep& curve2)const;

///Intersection with a surface.
MGCSisect_list isect_in_range(const MGSurface& surf)const;

///Exclusive function for common.
///When this curve is a TrimmedCurve of MGComposite, narrow the parameter range
///by this m_range.
void narrow_into_range(
	const MGCurve& curve2,
	std::vector<double>& vecComSpan
)const;

///Negate the curve direction(Ȑ̕𔽓])
void negate();

///Compute all the perpendicular points of this curve and the second one.
///That is, if f(s) and g(t) are the points of the two curves f and g,
///then obtains points where the following conditions are satisfied:
///  fs*(f-g)=0.    gt*(g-f)=0.
///Here fs and gt are 1st derivatives at s and t of f and g.
///MGPosition P in the MGPosition_list contains this and crv's parameter
///as:     P(0)=this curve's parameter, P(1)=crv's parameter value.
MGPosition_list perps_in_range(
	const MGCurve& crv2		///<The second curve
)const;

///mbg폜֐(B\Ȑ̂)
///gXline_zerogpB̃mbĝׂقǍ폜₷
///߂ĺA폜mbg̐
///removal knot. line_zero tolerance is used.
void remove_knot();

///Object transformation.
MGTrimmedCurve& operator+=(const MGVector& v);
MGTrimmedCurve& operator-=(const MGVector& v);
MGTrimmedCurve& operator*=(double scale);
MGTrimmedCurve& operator*=(const MGMatrix& mat);
MGTrimmedCurve& operator*=(const MGTransf& tr);

///Transformation object construction(NOT ALLOWED TO USE).
MGTrimmedCurve operator+ (const MGVector& v) const;
MGTrimmedCurve operator- (const MGVector& v) const;
MGTrimmedCurve operator* (double scale) const;
MGTrimmedCurve operator* (const MGMatrix& mat) const;
MGTrimmedCurve operator* (const MGTransf& tr) const;

friend class MGSurfCurve;
friend class MGLBRep;

};

/** @} */ // end of GEO group
#endif
