/********************************************************************/
/* Copyright (c) 2019 System fugen G.K. and Yuzi Mizuno          */
/* All rights reserved.                                             */
/********************************************************************/
#ifndef _MGisects_HH_
#define _MGisects_HH_
/** @file */
/** @addtogroup IsectContainer
 *  @{
 */

#include <memory>
#include <vector>
#include "mg/isect.h"

//Forward class declaration.
class MGisect;
class MGObject;
class MGCCisect_list;
class MGCSisect_list;
class MGSSisect_list;
class MGCFisect_vector;
class MGFFisect;
class MGHHisect;
class MGHHisect_vector;

///MGisects defines a vector of MGisect.

///MGisects is used to represent an array of intersection lines of 
///two objects.

///Intersections are obtained from two objects, which are known using
///the member functions object1() and object2().
///****NOTE****
///When two objects' manifold dimension are the same, object1 is this object
///at the invocation of MGObject::intersection(), and object2 is the argument
///object.
///However, their manifold dimension are not the same, object1 is always
///the lower dimension's object and object2 is the higer dimension's object.
class MG_DLL_DECLR MGisects{
using MYELM = std::unique_ptr<MGisect>;
using MYVEC=std::vector<MYELM>;

private:
	MYVEC m_vec;
const MGObject* m_object1;///< Object 1.
const MGObject* m_object2;///< Object 2.

public:

using iterator= MYVEC::iterator;
using const_iterator= MYVEC::const_iterator;
using reverse_iterator= MYVEC::reverse_iterator;
using const_reverse_iterator = MYVEC::const_reverse_iterator;
using reference= MYVEC::reference;
using const_reference= MYVEC::const_reference;
using size_type= MYVEC::size_type;

///String stream Function
MG_DLL_DECLR friend std::ostream& operator << (std::ostream& ostrm, const MGisects& is);


////////Special member functions/////////
~MGisects()=default;//Destructor.
MGisects(const MGisects& rhs)=delete;//Copy constructor.
MGisects& operator=(const MGisects& rhs)=delete;//Copy assignment.
MGisects(MGisects&& rhs)=default;//Move constructor.
MGisects& operator=(MGisects&& rhs)=default;//Move assignment.

///Constructor(of size 0)
MGisects(
	const MGObject* obj1=nullptr,
	const MGObject* obj2=nullptr
);

///Construct from MGCCisect_list.
MGisects(const MGCCisect_list& ccis);

///Construct from MGCSisect_list.
MGisects(const MGCSisect_list& csis);

///Construct from MGCSisect_list.
///(Only move operation is allowed)
MGisects(MGSSisect_list&& ssis);

///Construct from MGCFisect_vector.
MGisects(const MGCFisect_vector& cfis);

///Construct from MGHHisect.
///(Only move operation is allowed)
MGisects(MGHHisect&& hhi);

///Construct from MGCFisect_vector.
///(Only move operation is allowed)
MGisects(MGHHisect_vector&& hhis);

/////MYVEC's member function/////
//We cannot use inheritance of std::vector<MYELM> to make DLL.
iterator begin();
const_iterator begin()const;
iterator end();
const_iterator end()const;
reverse_iterator rbegin() noexcept;
const_reverse_iterator rbegin() const noexcept;
void clear();
bool empty() const;

template <class... Args>
void emplace_back(Args&&... args) {
	m_vec.emplace_back(std::forward<Args>(args)...); 
};

iterator erase(iterator x);
iterator erase(iterator first, iterator last);
const MYELM& front() const;
MYELM& front();
const MYELM& back() const;
MYELM& back();
iterator insert(iterator it, MYELM&& x);
void pop_back();
void push_back(MYELM&& x);//Only move operation is allowed.
void reserve(size_type n);
size_t size() const;
MYELM& operator[](size_t i);
const MYELM& operator[](size_t i)const;

///Exchange first and second order of MGisect.
void exchange12();

///Get the 1st object pointer of the i-th intersection.
///Generally objects are different for each intersection.
///Ex. in the case of Shell to Shell intersection, different Face pointer
///will be returned.
const MGObject* object1(int i)const;

///Get the 2nd object pointer of the i-th intersection.
///Generally objects are different for each intersection.
///Ex. in the case of Shell to Shell intersection, different Face pointer
///will be returned.
const MGObject* object2(int i)const;

///Adds one MGisect* to the end of the vector.
///Transfers the ownership of the isect to this vector.
void push_back(MGisect* isect);

///append all the member of isects to the end of the vector.
///Transfers the ownership of the isect in isects to this vector.
void push_back(MGisects&& isects);

};

/** @} */ // end of IsectContainer group
#endif