package jp.sourceforge.talisman.pcp.ui.swing;

/*
 * $Id: PcpPane.java 132 2008-07-15 03:30:47Z tama3 $
 */

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.JLabel;
import javax.swing.JLayeredPane;
import javax.swing.JPanel;

import jp.sourceforge.talisman.i18n.MessageManager;
import jp.sourceforge.talisman.i18n.Messages;
import jp.sourceforge.talisman.i18n.ResourceNotFoundException;
import jp.sourceforge.talisman.pcp.Metrics;
import jp.sourceforge.talisman.pcp.MetricsDomain;
import jp.sourceforge.talisman.pcp.Sample;
import jp.sourceforge.talisman.pcp.Samples;
import jp.sourceforge.talisman.pcp.ui.PcpGraphSetting;

/**
 * This swing component shows a PCP (Parallel coordinates plot). 
 * 
 * @author Haruaki Tamada
 * @version $Revision: 132 $ 
 */
public class PcpPane extends JPanel implements MessageManager{
    private static final long serialVersionUID = 7536699600763579247L;

    private Samples samples;
    private MetricsDomain[] domains;
    private JLayeredPane layer = new JLayeredPane();
    private PcpAxisPane borderPane;
    private PcpMainPane mainPane;
    private PcpSelectedAreaPane selectionPane;
    private PcpGraphSetting setting;
    private Messages messages;

    public PcpPane(Samples samples) throws ResourceNotFoundException{
        this(samples, new PcpGraphSetting());
    }

    public PcpPane(Samples samples, Messages messages){
        this(samples, new PcpGraphSetting(), messages);
    }

    public PcpPane(Samples samples, PcpGraphSetting setting) throws ResourceNotFoundException{
        this(samples, setting, new Messages("jp.sourceforge.talisman.pcp.messages"));
    }

    public PcpPane(Samples samples, PcpGraphSetting setting, Messages messages){
        this.samples = samples;
        this.messages = messages;
        setSetting(setting);

        initData();
        initLayouts();
    }

    public PcpGraphSetting getSetting(){
        return setting;
    }

    public void setSetting(PcpGraphSetting setting){
        if(setting == null){
            throw new NullPointerException();
        }
        this.setting = setting;
        setting.addPropertyChangeListener(new PropertyChangeListener(){
            public void propertyChange(PropertyChangeEvent evt){
                repaint();
            }
        });
    }

    /**
     * clears current selection.
     */
    public void clearSelection(){
        selectionPane.clearSelection();
        repaint();
    }

    private void initData(){
        Metrics[] metric = samples.getAvailableMetrics();
        MetricsDomain[] domains = new MetricsDomain[metric.length];
        for(int i = 0; i < domains.length; i++){
            domains[i] = new MetricsDomain(metric[i]);
        }

        for(int i = 0; i < metric.length; i++){
            for(Sample sample: samples){
                if(sample.hasValue(metric[i])){
                    int v = sample.getValue(metric[i]);
                    if(domains[i].getMaximum() < v){
                        domains[i].setMaximum(v);
                    }
                    if(domains[i].getMinimum() > v){
                        domains[i].setMinimum(v);
                    }
                }
            }
        }
        this.domains = domains;
    }

    private void initLayouts(){
        JPanel labelPane = new JPanel(new GridLayout(1, domains.length, 0, 0));
        for(int i = 0; i < domains.length; i++){
            Metrics metrics = domains[i].getMetrics();
            JLabel label = new JLabel(metrics.getName(), JLabel.CENTER);
            if(metrics.getDescription() != null){
                label.setToolTipText(metrics.getDescription());
            }
            labelPane.add(label);
        }

        borderPane = new PcpAxisPane(domains, getSetting(), getMessages());
        mainPane = new PcpMainPane(samples, domains, getSetting());
        selectionPane = new PcpSelectedAreaPane(samples, domains, getSetting());

        layer.add(borderPane, JLayeredPane.DEFAULT_LAYER);
        layer.add(mainPane, JLayeredPane.PALETTE_LAYER);
        layer.add(selectionPane, JLayeredPane.MODAL_LAYER);

        layer.addComponentListener(new ComponentAdapter(){
            @Override
            public void componentResized(ComponentEvent e){
                Component comp = e.getComponent();
                Dimension dim = comp.getSize();
                borderPane.setSize(dim);
                mainPane.setSize(dim);
                selectionPane.setSize(dim);
            }
        });
        layer.setPreferredSize(new Dimension(800, 400));

        setLayout(new BorderLayout());
        add(layer, BorderLayout.CENTER);
        add(labelPane, BorderLayout.SOUTH);

        setFont(getFont());
    }

    @Override
    public void setFont(Font font){
        super.setFont(font);
        if(mainPane != null){
            mainPane.setFont(font);
        }
        if(selectionPane != null){
            selectionPane.setFont(font);
        }
        if(borderPane != null){
            borderPane.setFont(font);
        }
    }

    public Messages getMessages(){
        return messages;
    }
}