package jp.sourceforge.talisman.i18n;

/*
 * $Id: MessagesTest.java 140 2008-07-15 04:01:32Z tama3 $
 */

import java.util.Locale;

import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;


/**
 * Test class of Messages.
 * 
 * @author Haruaki Tamada
 * @version $Revision: 140 $
 */
public class MessagesTest{
    private Messages messages;

    /**
     * set default locale as Japan.
     */
    @BeforeClass
    public static void initializeDefaultLocale(){
        Locale.setDefault(Locale.JAPAN);
    }

    /**
     * build instance of Messages class.
     */
    @Before
    public void buildMessages(){
        try{
            messages = new Messages(Locale.JAPANESE, "resources.test");
        } catch(ResourceNotFoundException e){
        }
    }

    /**
     * ResourceNotFoundException will be thrown because resource file is not exist.
     */
    @Test(expected=ResourceNotFoundException.class)
    public void loadFail() throws Exception{
        new Messages("test");
    }

    @Test
    public void escapeTest(){
        Assert.assertEquals("$${notescape}", messages.get(Locale.JAPANESE, "test.escape1"));
        Assert.assertEquals("$notescape",    messages.get(Locale.JAPANESE, "test.escape2"));
        Assert.assertEquals("${notescape",   messages.get(Locale.JAPANESE, "test.escape3"));
        Assert.assertEquals("$notescape}",   messages.get(Locale.JAPANESE, "test.escape4"));
    }

    @Test
    public void basicTest(){
        messages.setLocale(Locale.GERMAN);
        Assert.assertEquals(Locale.GERMAN, messages.getLocale());
    }

    @Test
    public void nullTest() throws Exception{
        Assert.assertFalse(messages.hasValue(Locale.JAPANESE, "missing.key"));
        Assert.assertNull(messages.get(Locale.JAPANESE, "missing.key"));
        Assert.assertNull(messages.format(Locale.JAPANESE, "missing.key", "arg1", "arg2"));

        Assert.assertTrue(messages.hasValue(Locale.JAPANESE, "test1"));
        Assert.assertEquals("default value",
                            messages.get(Locale.JAPANESE, "missing.key", "default value"));
        Assert.assertEquals(0, messages.getArray(Locale.JAPANESE, "missing.key").length);
    }

    @Test
    public void arrayTest1() throws Exception{
        String[] array = messages.getArray(Locale.GERMAN, "test.array1");
        Assert.assertNotNull(array);
        Assert.assertEquals(3, array.length);
        Assert.assertEquals("aaa", array[0]);
        Assert.assertEquals("bbb", array[1]);
        Assert.assertEquals("ccc", array[2]);
    }

    @Test
    public void arrayTest2() throws Exception{
        String[] array = messages.getArray(Locale.GERMAN, "test.array2");
        Assert.assertNotNull(array);
        Assert.assertEquals(3, array.length);
        Assert.assertEquals("aaa", array[0]);
        Assert.assertEquals("bbb", array[1]);
        Assert.assertEquals("ccc", array[2]);
    }

    @Test
    public void testJapanese(){
        Assert.assertEquals("テスト1", messages.get(Locale.JAPANESE, "test1"));
        Assert.assertEquals("テスト2", messages.get(Locale.JAPANESE, "test2"));
        Assert.assertEquals("テスト3", messages.get(Locale.JAPANESE, "test3"));
        Assert.assertEquals("test4", messages.get(Locale.JAPANESE, "test4"));

        Assert.assertEquals("日本語 foo bar",
                            messages.format(Locale.JAPANESE, "test.template", "foo", "bar"));
        Assert.assertEquals("default テスト1 テスト2",
                            messages.get(Locale.JAPANESE, "test.replace"));
    }

    @Test
    public void testEnglish(){
        Assert.assertEquals("test1", messages.get(Locale.ENGLISH, "test1"));
        Assert.assertEquals("test2", messages.get(Locale.ENGLISH, "test2"));
        Assert.assertEquals("test3", messages.get(Locale.ENGLISH, "test3"));
        Assert.assertEquals("test4", messages.get(Locale.ENGLISH, "test4"));

        Assert.assertEquals("english foo bar",
                            messages.format(Locale.ENGLISH, "test.template", "foo", "bar"));
        Assert.assertEquals("default test1 test2",
                            messages.get(Locale.ENGLISH, "test.replace"));
    }

    @Test
    public void testGerman(){
        Assert.assertEquals("test1 (default locale)", messages.get(Locale.GERMAN, "test1"));
        Assert.assertEquals("test2 (default locale)", messages.get(Locale.GERMAN, "test2"));
        Assert.assertEquals("test3 (default locale)", messages.get(Locale.GERMAN, "test3"));
        Assert.assertEquals("test4", messages.get(Locale.GERMAN, "test4"));

        Assert.assertEquals("default foo bar",
                            messages.format(Locale.GERMAN, "test.template", "foo", "bar"));
        Assert.assertEquals("default test1 (default locale) test2 (default locale)",
                            messages.get(Locale.GERMAN, "test.replace"));
    }

    @Test
    public void testFrench(){
        Assert.assertEquals("テスト1", messages.get(Locale.FRENCH, "test1"));
        Assert.assertEquals("テスト2", messages.get(Locale.FRENCH, "test2"));
        Assert.assertEquals("テスト3", messages.get(Locale.FRENCH, "test3"));
        Assert.assertEquals("test4", messages.get(Locale.FRENCH, "test4"));

        Assert.assertEquals("日本語 foo bar",
                            messages.format(Locale.FRENCH, "test.template", "foo", "bar"));
        Assert.assertEquals("default テスト1 テスト2",
                            messages.get(Locale.FRENCH, "test.replace"));
    }
}
