require 'date'

module TapKit

	class Date
		include Comparable

		attr_accessor :year, :month, :day

		class << self
			def new_with_date( date )
				Date.new(date.year, date.month, date.day)
			end

			def new_with_time( time )
				Date.new(time.year, time.month, time.day)
			end

			def today
				Date.new_with_date ::Date.today
			end

			def parse( string )
				new_with_date DateTime.parse(string)
			end
		end

		def initialize( year = 0, month = 0, day = 0 )
			@year  = year
			@month = month
			@day   = day
		end

		def +( other )
			date = to_date + other
			Date.new_with_date date
		end

		alias >> +

		def -( other )
			case other
			when Date
				to_date - other.to_date
			when ::Date
				to_date - other
			else
				Date.new_with_date((to_date - other))
			end
		end

		def <=>( other )
			case other
			when Date
				to_date <=> other.to_date
			else
				to_date <=> other
			end
		end

		def ===( other )
			if (@year == other.year) and (@month == other.mon) \
				and (@day == other.mday) then
				true
			else
				false
			end
		end

		def to_time
			::Time.local(@year, @month, @day, 0, 0, 0)
		end

		def to_date
			::Date.new(@year, @month, @day)
		end

		def to_s
			sprintf("%04d-%02d-%02d", @year, @month, @day)
		end

		alias mon   month
		alias mon=  month=
		alias mday  day
		alias mday= day=
		alias inspect to_s
	end


	class Time
		include Comparable

		attr_accessor :hour, :minute, :second

		class << self
			def new_with_time( time )
				Time.new(time.hour, time.min, time.sec)
			end

			def now
				Time.new_with_time ::Time.now
			end

			def parse( string )
				require 'time'
				new_with_time ::Time.parse(string)
			end
		end

		def initialize( hour = 0, minute = 0, second = 0 )
			@hour   = hour
			@minute = minute
			@second = second
		end

		def +( other )
			time = to_time + other
			Time.new_with_time time
		end

		def -( other )
			case other
			when Time
				to_time - other.to_time
			when ::Time
				to_time - other
			else
				Time.new_with_time((to_time - other))
			end
		end

		def <=>( other )
			to_time <=> other.to_time
		end

		def am?
			if @hour <= 11 then
				true
			else
				false
			end
		end

		def pm?
			if am? == true then
				false
			else
				true
			end
		end

		def to_time
			t = ::Time.now
			::Time.local(t.year, t.month, t.day, @hour, @minute, @second)
		end

		def to_s( meridiem = false )
			if meridiem == false then
				sprintf("%02d:%02d:%02d", @hour, @minute, @second)
			else
				if am? then
					meridiem = 'AM'
					hour = @hour
				else
					meridiem = 'PM'
					hour = @hour - 12
				end
				sprintf("%02d:%02d #{meridiem}", hour, @minute)
			end
		end

		alias min  minute
		alias min= minute=
		alias sec  second
		alias sec= second=
		alias inspect to_s
	end


	class Timestamp
		include Comparable

		attr_accessor :year, :month, :day, :hour, :minute, :second, :zone

		class << self
			def new_with_datetime( datetime )
				Timestamp.new(datetime.year, datetime.month, datetime.day,
					datetime.hour, datetime.min, datetime.sec, datetime.zone)
			end

			def now
				Timestamp.new_with_datetime ::Time.now
			end

			def parse( string )
				new_with_datetime DateTime.parse(string)
			end
		end
  
		def initialize( year = 0, month  = 0, day    = 0,
		                hour = 0, minute = 0, second = 0, zone = nil )
			@year     = year
			@month    = month
			@day      = day
			@hour     = hour
			@minute   = minute
			@second   = second
			@zone     = zone
		end

		def +( other )
			Timestamp.new_with_datetime((to_time + other))
		end

		def -( other )
			case other
			when Timestamp
				to_time - other.to_time
			when ::Time
				to_time - other
			else
				Timestamp.new_with_datetime((to_time - other))
			end
		end

		def <=>( other )
			case other
			when Timestamp
				to_time <=> other.to_time
			else
				to_time <=> other
			end
		end

		def to_s
			s = sprintf("%04d-%02d-%02d %02d:%02d:%02d",
				@year, @month, @day, @hour, @minute, @second) 

			if @zone.nil?
				s
			else
				s + " #@zone"
			end
		end

		def to_time
			::Time.local(@year, @month, @day, @hour, @minute, @second)
		end

		def to_date
			::Date.new(@year, @month, @day)
		end

		def to_datetime
			DateTime.parse to_s
		end

		alias mon   month
		alias mon=  month=
		alias mday  day
		alias mday= day=
		alias min   minute
		alias min=  minute=
		alias sec   second
		alias sec=  second=
		alias inspect to_s
	end

end


def Date(object); TapKit::Date.parse(object.to_s); end
def Time(object); TapKit::Time.parse(object.to_s); end
def Timestamp(object); TapKit::Timestamp.parse(object.to_s); end
