class TestQualifier < Test::Unit::TestCase
	include TapKit

	# operators

	def test_operators
		expected = ['=', '==', '!=', '<', '<=', '>', '>=', 'like', 'cilike']
		assert_equal(expected, Qualifier.operators)
	end

	def test_relational_operators
		expected = ['=', '==', '!=', '<', '<=', '>', '>=']
		assert_equal(expected, Qualifier.relational_operators)
	end

	def test_operator_symbol
		expected = { 'dummylike' => nil, '=' => Qualifier::EQUAL,
			'=='   => Qualifier::EQUAL,   '!='     => Qualifier::NOT_EQUAL, 
			'>'    => Qualifier::GREATER, '>='     => Qualifier::GREATER_OR_EQUAL,
			'<'    => Qualifier::LESS,    '<='     => Qualifier::LESS_OR_EQUAL,
			'like' => Qualifier::LIKE,    'cilike' => Qualifier::CI_LIKE }

		expected.each do |string, symbol|
			assert_equal(symbol, Qualifier.operator_symbol(string))
		end
	end

	def test_operator_string
		expected = { 'dummy' => nil,
			Qualifier::EQUAL   => '=' ,   Qualifier::NOT_EQUAL        => '!=',
			Qualifier::GREATER => '>',    Qualifier::GREATER_OR_EQUAL => '>=',
			Qualifier::LESS    => '<',    Qualifier::LESS_OR_EQUAL    => '<=',
			Qualifier::LIKE    => 'like', Qualifier::CI_LIKE          => 'cilike' }

		expected.each do |symbol, string|
			assert_equal(string, Qualifier.operator_string(symbol))
		end
	end


	# qualifier

	def test_new_to_match_all_values
		values    = {'title' => 'test', 'number' => 1}
		expected  = "((number = 1) AND (title = 'test'))"
		qualifier = Qualifier.new_to_match_all_values values

		assert_equal(expected, qualifier.to_s)
		assert_instance_of(AndQualifier, qualifier)
	end

	def test_new_to_match_any_value
		values    = {'title' => 'test', 'number' => 1}
		expected  = "((number = 1) OR (title = 'test'))"
		qualifier = Qualifier.new_to_match_any_value values

		assert_equal(expected, qualifier.to_s)
		assert_instance_of(OrQualifier, qualifier)
	end

	def test_new_with_format
		qualifier = Qualifier.new_with_format "title like '*'"
		assert_instance_of(KeyValueQualifier, qualifier)
	end


	# filter

	def test_filter_equal
		qualifier = Qualifier.format "title = 'test'"
		app       = Application.new MODELS
		ec        = app.create_editing_context
		object1 = ec.create 'cd'
		object2 = ec.create 'cd'
		object3 = ec.create 'cd'
		object1['title'] = 'test'
		object2['title'] = 'dummy'
		object3['title'] = 'test'
		array    = [object1, object2, object3]
		expected = [object1, object3]
		assert_equal(expected, Qualifier.filter(array, qualifier))
	end

	def test_filter_not_equal
		qualifier = Qualifier.format "title != 'test'"
		app       = Application.new MODELS
		ec        = app.create_editing_context
		object1 = ec.create 'cd'
		object2 = ec.create 'cd'
		object3 = ec.create 'cd'
		object1['title'] = 'test'
		object2['title'] = 'dummy'
		object3['title'] = 'test'
		array    = [object1, object2, object3]
		expected = [object2]
		assert_equal(expected, Qualifier.filter(array, qualifier))
	end

	def test_filter_greater
		qualifier = Qualifier.format "release > 50"
		app       = Application.new MODELS
		ec        = app.create_editing_context
		object1 = ec.create 'cd'
		object2 = ec.create 'cd'
		object3 = ec.create 'cd'
		object1['release'] = 10
		object2['release'] = 50
		object3['release'] = 100
		array    = [object1, object2, object3]
		expected = [object3]
		assert_equal(expected, Qualifier.filter(array, qualifier))
	end

	def test_filter_greater_or_equal
		qualifier = Qualifier.format "release >= 50"
		app       = Application.new MODELS
		ec        = app.create_editing_context
		object1 = ec.create 'cd'
		object2 = ec.create 'cd'
		object3 = ec.create 'cd'
		object1['release'] = 10
		object2['release'] = 50
		object3['release'] = 100
		array    = [object1, object2, object3]
		expected = [object2, object3]
		assert_equal(expected, Qualifier.filter(array, qualifier))
	end

	def test_filter_less
		qualifier = Qualifier.format "release < 50"
		app       = Application.new MODELS
		ec        = app.create_editing_context
		object1 = ec.create 'cd'
		object2 = ec.create 'cd'
		object3 = ec.create 'cd'
		object1['release'] = 10
		object2['release'] = 50
		object3['release'] = 100
		array    = [object1, object2, object3]
		expected = [object1]
		assert_equal(expected, Qualifier.filter(array, qualifier))
	end

	def test_filter_less_or_equal
		qualifier = Qualifier.format "release <= 50"
		app       = Application.new MODELS
		ec        = app.create_editing_context
		object1 = ec.create 'cd'
		object2 = ec.create 'cd'
		object3 = ec.create 'cd'
		object1['release'] = 10
		object2['release'] = 50
		object3['release'] = 100
		array    = [object1, object2, object3]
		expected = [object1, object2]
		assert_equal(expected, Qualifier.filter(array, qualifier))
	end

	def test_filter_like
		qualifier = Qualifier.format "title like 'a*'"
		app       = Application.new MODELS
		ec        = app.create_editing_context
		object1 = ec.create 'cd'
		object2 = ec.create 'cd'
		object3 = ec.create 'cd'
		object1['title'] = 'abcde'
		object2['title'] = 'bcdea'
		object3['title'] = 'ABCDE'
		array    = [object1, object2, object3]
		expected = [object1]
		assert_equal(expected, Qualifier.filter(array, qualifier))
	end

	def test_filter_cilike
		qualifier = Qualifier.format "title cilike 'a*'"
		app       = Application.new MODELS
		ec        = app.create_editing_context
		object1 = ec.create 'cd'
		object2 = ec.create 'cd'
		object3 = ec.create 'cd'
		object1['title'] = 'abcde'
		object2['title'] = 'bcdea'
		object3['title'] = 'ABCDE'
		array    = [object1, object2, object3]
		expected = [object1, object3]
		assert_equal(expected, Qualifier.filter(array, qualifier))
	end



	# validate

	def test_qualifier_keys
		qualifier = Qualifier.new
		assert qualifier.qualifier_keys.empty?
	end

	def test_validate
		qualifier =  Qualifier.new_with_format "dummy == 'error'"
		app = Application.new MODELS
		description = app.entity('Book').class_description

		assert_raises(Qualifier::UnknownKeyError) { qualifier.validate(description) }
	end


	# format

	def test_parser_for_literals
		left = "title like"
		tokens = { "'*'" => String, '"*"' => String, '100' => Fixnum, '0.1' => Float,
			'true' => TrueClass, 'false' => FalseClass, 'nil' => NilClass,
			'true_dummy' => String, "(Integer)'1'" => Fixnum,
			"(Date)'2003-12-15'"=>TapKit::Date, "(Time)'12:30:30'"=>TapKit::Time,
			"(Timestamp)'2003-12-15 12:30:30 +0900'"=>TapKit::Timestamp }

		tokens.each do |right, expected|
			format = "#{left} #{right}"
			q = Qualifier.new_with_format format

			case q
			when KeyValueQualifier
				assert_instance_of(expected, q.value)
			when KeyComparisonQualifier
				assert_instance_of(expected, q.right)
			end
		end
	end

	def test_parser_for_operator
		operators = {
			'='    => Qualifier::EQUAL,
			'=='   => Qualifier::EQUAL,   '!='     => Qualifier::NOT_EQUAL, 
			'>'    => Qualifier::GREATER, '>='     => Qualifier::GREATER_OR_EQUAL,
			'<'    => Qualifier::LESS,    '<='     => Qualifier::LESS_OR_EQUAL,
			'like' => Qualifier::LIKE,    'cilike' => Qualifier::CI_LIKE }

		operators.each do |operator, expected|
			format = "title #{operator} '*'"
			q = Qualifier.new_with_format format

			assert_equal(expected, q.symbol)
		end
	end

	# %s, %d, %f, %@, %K, %%
	def test_parser_for_format_strings
		# bindings, convert
		tests = [
			{:convert => '%s', :bindings => 'test',  :expected => 'test', :key => false},
			{:convert => '%s', :bindings => '100',   :expected => '100',  :key => false},
			{:convert => '%s', :bindings => "'",     :expected => "\\'",  :key => false},
			{:convert => '%d', :bindings => '100',   :expected => 100,    :key => false},
			{:convert => '%d', :bindings => 100,     :expected => 100,    :key => false},
			{:convert => '%f', :bindings => '0.1',   :expected => 0.1,    :key => false},
			{:convert => '%f', :bindings => 0.1,     :expected => 0.1,    :key => false},
			{:convert => '%@', :bindings => 'test',  :expected => 'test', :key => false},
			{:convert => '%@', :bindings => "'",     :expected => "\\'",  :key => false},
			{:convert => '%@', :bindings => 100,     :expected => 100,    :key => false},
			{:convert => '%@', :bindings => 0.1,     :expected => 0.1,    :key => false},
			{:convert => '%@', :bindings => true,    :expected => true,   :key => false},
			{:convert => '%@', :bindings => false,   :expected => false,  :key => false},
			{:convert => '%@', :bindings => nil,     :expected => nil,    :key => false},
			{:convert => '%K', :bindings => 'test',  :expected => 'test', :key => true},
			{:convert => '%K', :bindings => 'k.p',   :expected => 'k.p',  :key => true},
			{:convert => '%K', :bindings => 100,     :expected => 100,    :key => false},
			{:convert => '%K', :bindings => 0.1,     :expected => 0.1,    :key => false},
			{:convert => '%K', :bindings => true,    :expected => true,   :key => false},
			{:convert => '%K', :bindings => false,   :expected => false,  :key => false},
			{:convert => '%K', :bindings => nil,     :expected => nil,    :key => false},
			{:convert => "'%%'", :bindings => 'dummy', :expected => '%',  :key => false},
			{:convert => '%@', :bindings => Date.new(2003,12,15),
				:expected => Date.new(2003,12,15), :key => false},
			{:convert => '%@', :bindings => Time.new(12,30,30),
				:expected => Time.new(12,30,30), :key => false},
			{:convert => '%@', :bindings => Timestamp.new(2003,12,15,12,30,30),
				:expected => Timestamp.new(2003,12,15,12,30,30), :key => false},
			{:convert => '%@', :bindings => Array(1),
				:expected => Array('1'), :key => false}
		]

		tests.each do |test|
			format = "title like #{test[:convert]}"
			qualifier = Qualifier.new_with_format(format, [test[:bindings]])

			if test[:key] == true then
				assert_instance_of(KeyComparisonQualifier, qualifier)
				assert_equal(test[:expected], qualifier.right)
			else
				assert_instance_of(KeyValueQualifier, qualifier)
				assert_equal(test[:expected], qualifier.value)
			end
		end
	end

	def test_parser_for_key_value
		expected_format = "(title like '*')"
		expected_qualifier = KeyValueQualifier.new('title', Qualifier::LIKE, '*')
		qualifier = Qualifier.new_with_format expected_format

		assert_equal(expected_qualifier, qualifier)
		assert_equal(expected_format, qualifier.to_s)
	end

	def test_parser_for_key_comparison
		expected_format = "(salary > manager.salary)"
		expected_qualifier = KeyComparisonQualifier.new( \
			'salary', Qualifier::GREATER, 'manager.salary')
		qualifier = Qualifier.new_with_format expected_format

		assert_equal(expected_qualifier, qualifier)
		assert_equal(expected_format, qualifier.to_s)
	end

	def test_parser_for_and
		expected_format = "((title like '*') AND (name = 'John'))"
		keyvalue1 = KeyValueQualifier.new('title', Qualifier::LIKE, '*')
		keyvalue2 = KeyValueQualifier.new('name', Qualifier::EQUAL, 'John')
		expected_qualifier = AndQualifier.new [keyvalue1, keyvalue2]
		qualifier = Qualifier.new_with_format expected_format

		assert_equal(expected_qualifier, qualifier)
		assert_equal(expected_format, qualifier.to_s)
	end

	def test_parser_for_or
		expected_format = "((title like '*') OR (name = 'John'))"
		keyvalue1 = KeyValueQualifier.new('title', Qualifier::LIKE, '*')
		keyvalue2 = KeyValueQualifier.new('name', Qualifier::EQUAL, 'John')
		expected_qualifier = OrQualifier.new [keyvalue1, keyvalue2]
		qualifier = Qualifier.new_with_format expected_format

		assert_equal(expected_qualifier, qualifier)
		assert_equal(expected_format, qualifier.to_s)
	end

	def test_parser_for_not
		expected_format = "(NOT ((title like '*') AND (name = 'John')))"
		keyvalue1 = KeyValueQualifier.new('title', Qualifier::LIKE, '*')
		keyvalue2 = KeyValueQualifier.new('name', Qualifier::EQUAL, 'John')
		and_qualifier = AndQualifier.new [keyvalue1, keyvalue2]
		expected_qualifier = NotQualifier.new and_qualifier
		qualifier = Qualifier.new_with_format expected_format

		assert_equal(expected_qualifier, qualifier)
		assert_equal(expected_format, qualifier.to_s)
	end

end


# equal, to_s, eval?, add_qualifier_keys
class TestKeyValueQualifier < Test::Unit::TestCase
	include TapKit

	def setup
		@qualifier = KeyValueQualifier.new('key', Qualifier::EQUAL, 'value')
	end

	def test_qualifier_keys
		expected = Set.new ['key']

		assert_equal(expected, @qualifier.qualifier_keys)
	end

	def test_equal
		qualifier1 = KeyValueQualifier.new('key', Qualifier::EQUAL, 'value')
		qualifier2 = KeyValueQualifier.new('key', Qualifier::EQUAL, 'value')

		assert_equal(qualifier1, qualifier2)
	end

	def test_eval?
		app       = Application.new MODELS
		ec        = app.create_editing_context
		object    = ec.create 'Book'
		qualifier = Qualifier.format "title like 'ab*'"

		object['title'] = 'abcde'
		assert qualifier.eval?(object)

		object['title'] = 'cde'
		assert_equal(false, qualifier.eval?(object))
	end

	def test_to_s
		expected = "(key = 'value')"

		assert_equal(expected, @qualifier.to_s)
	end

end


class TestKeyComparisonQualifier < Test::Unit::TestCase
	include TapKit

	def setup
		@qualifier = KeyComparisonQualifier.new('left', Qualifier::EQUAL, 'right.path')
	end

	def test_qualifier_keys
		expected = Set.new ['left']

		assert_equal(expected, @qualifier.qualifier_keys)
	end

	def test_equal
		qualifier1 = KeyComparisonQualifier.new('left', Qualifier::EQUAL, 'right.path')
		qualifier2 = KeyComparisonQualifier.new('left', Qualifier::EQUAL, 'right.path')

		assert_equal(qualifier1, qualifier2)
	end

	def test_eval?
		app       = Application.new MODELS
		ec        = app.create_editing_context
		qualifier = Qualifier.format "title like publisher.name"

		book      = ec.create 'Book'
		publisher = ec.create 'Publisher'

		book['publisher'] = publisher
		book['title']     = 'test'
		publisher['name'] = 'test'
		assert qualifier.eval?(book)

		book['title'] = 'dummy'
		assert_equal(false, qualifier.eval?(book))
	end

	def test_to_s
		expected  = "(left = right.path)"

		assert_equal(expected, @qualifier.to_s)
	end
end


class TestAndQualifier < Test::Unit::TestCase
	include TapKit

	def setup
		q1 = KeyValueQualifier.new('key1', Qualifier::EQUAL, 'value1')
		q2 = KeyValueQualifier.new('key2', Qualifier::EQUAL, 'value2')
		array = [q1, q2]
		@qualifier = AndQualifier.new array
	end

	def test_qualifier_keys
		expected = Set.new ['key1', 'key2']

		assert_equal(expected, @qualifier.qualifier_keys)
	end

	def test_equal
		keyvalue   = KeyValueQualifier.new('key', Qualifier::EQUAL, 'value')
		qualifier1 = AndQualifier.new [keyvalue]
		qualifier2 = AndQualifier.new [keyvalue]

		assert_equal(qualifier1, qualifier2)
	end

	def test_eval?
		app       = Application.new MODELS
		ec        = app.create_editing_context
		qualifier = Qualifier.format "(title == 'test') and (publisher.name == 'dummy')"

		book      = ec.create 'Book'
		publisher = ec.create 'Publisher'

		book['publisher'] = publisher
		book['title']     = 'test'
		publisher['name'] = 'dummy'
		assert qualifier.eval?(book)

		publisher['name'] = 'failure'
		assert_equal(false, qualifier.eval?(book))
	end

	def test_to_s
		expected = "((key1 = 'value1') AND (key2 = 'value2'))"

		assert_equal(expected, @qualifier.to_s)
	end
end


class TestOrQualifier < Test::Unit::TestCase
	include TapKit

	def setup
		q1 = KeyValueQualifier.new('key1', Qualifier::EQUAL, 'value1')
		q2 = KeyValueQualifier.new('key2', Qualifier::EQUAL, 'value2')
		array = [q1, q2]
		@qualifier = OrQualifier.new array
	end

	def test_qualifier_keys
		expected = Set.new ['key1', 'key2']

		assert_equal(expected, @qualifier.qualifier_keys)
	end

	def test_equal
		keyvalue   = KeyValueQualifier.new('key', Qualifier::EQUAL, 'value')
		qualifier1 = OrQualifier.new [keyvalue]
		qualifier2 = OrQualifier.new [keyvalue]

		assert_equal(qualifier1, qualifier2)
	end

	def test_eval?
		app       = Application.new MODELS
		ec        = app.create_editing_context
		qualifier = Qualifier.format "(title == 'test') or (publisher.name == 'dummy')"

		book      = ec.create 'Book'
		publisher = ec.create 'Publisher'

		book['publisher'] = publisher
		book['title']     = 'dummy'
		publisher['name'] = 'dummy'
		assert qualifier.eval?(book)

		publisher['name'] = 'failure'
		assert_equal(false, qualifier.eval?(book))
	end

	def test_to_s
		expected = "((key1 = 'value1') OR (key2 = 'value2'))"

		assert_equal(expected, @qualifier.to_s)
	end
end


class TestNotQualifier < Test::Unit::TestCase
	include TapKit

	def setup
		keyvalue   = KeyValueQualifier.new('key', Qualifier::EQUAL, 'value')
		@qualifier = NotQualifier.new keyvalue
	end

	def test_qualifier_keys
		expected = Set.new ['key']

		assert_equal(expected, @qualifier.qualifier_keys)
	end

	def test_equal
		keyvalue   = KeyValueQualifier.new('key', Qualifier::EQUAL, 'value')
		qualifier1 = NotQualifier.new keyvalue
		qualifier2 = NotQualifier.new keyvalue

		assert_equal(qualifier1, qualifier2)
	end

	def test_eval?
		app       = Application.new MODELS
		ec        = app.create_editing_context
		object    = ec.create 'Book'
		qualifier = Qualifier.format "not (title like 'ab*')"

		object['title'] = 'cde'
		assert qualifier.eval?(object)

		object['title'] = 'abcde'
		assert_equal(false, qualifier.eval?(object))
	end

	def test_to_s
		expected  = "(NOT (key = 'value'))"

		assert_equal(expected, @qualifier.to_s)
	end
end


class TestQualifierComparisonSupport < Test::Unit::TestCase
	include TapKit

	# ==, !=, >, >=, <, <=, like, cilike
	def test_compare
		tests = [ \
		{:symbol=>Qualifier::EQUAL, :left=>'yes', :right=>'yes', :expected=>true},
		{:symbol=>Qualifier::EQUAL, :left=>'yes', :right=>'no',   :expected=>false},
		{:symbol=>Qualifier::NOT_EQUAL, :left=>'yes', :right=>'no', :expected=>true},
		{:symbol=>Qualifier::NOT_EQUAL, :left=>'yes', :right=>'yes', :expected=>false},
		{:symbol=>Qualifier::GREATER, :left=>2, :right=>1, :expected=>true},
		{:symbol=>Qualifier::GREATER, :left=>2, :right=>2, :expected=>false},
		{:symbol=>Qualifier::GREATER_OR_EQUAL, :left=>2, :right=>1, :expected=>true},
		{:symbol=>Qualifier::GREATER_OR_EQUAL, :left=>2, :right=>2, :expected=>true},
		{:symbol=>Qualifier::GREATER_OR_EQUAL, :left=>1, :right=>2, :expected=>false},
		{:symbol=>Qualifier::LESS, :left=>1, :right=>2, :expected=>true},
		{:symbol=>Qualifier::LESS, :left=>1, :right=>1, :expected=>false},
		{:symbol=>Qualifier::LESS_OR_EQUAL, :left=>1, :right=>2, :expected=>true},
		{:symbol=>Qualifier::LESS_OR_EQUAL, :left=>1, :right=>1, :expected=>true},
		{:symbol=>Qualifier::LESS_OR_EQUAL, :left=>2, :right=>1, :expected=>false},
		{:symbol=>Qualifier::LIKE, :left=>'yes', :right=>'yes', :expected=>true},
		{:symbol=>Qualifier::LIKE, :left=>'yes', :right=>'y*', :expected=>true},
		{:symbol=>Qualifier::LIKE, :left=>'yes', :right=>'y?s', :expected=>true},
		{:symbol=>Qualifier::LIKE, :left=>'yes', :right=>'YES', :expected=>false},
		{:symbol=>Qualifier::CI_LIKE, :left=>'yes', :right=>'yes', :expected=>true},
		{:symbol=>Qualifier::CI_LIKE, :left=>'yes', :right=>'YES', :expected=>true},
		{:symbol=>Qualifier::CI_LIKE, :left=>'yes', :right=>'y*', :expected=>true},
		{:symbol=>Qualifier::CI_LIKE, :left=>'yes', :right=>'Y*', :expected=>true},
		{:symbol=>Qualifier::CI_LIKE, :left=>'yes', :right=>'y?s', :expected=>true},
		{:symbol=>Qualifier::CI_LIKE, :left=>'yes', :right=>'Y?s', :expected=>true},
		{:symbol=>Qualifier::CI_LIKE, :left=>'yes', :right=>'no', :expected=>false},
		]

		tests.each do |test|
			assert_equal(test[:expected], \
				Qualifier.compare(test[:left], test[:right], test[:symbol]))
		end
	end
end
