unit Frame_ServerOfficialGraph;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, MainDatastore, DB, DBClient, Provider, IBCustomDataSet, IBQuery,
  StdCtrls, Grids, DBCtrls, IBSQL, IBDatabase, DBLocal, DBLocalI, Math,
  ImgList, DateUtils, StrUtils, ActnList,
  Misc_Constants, Misc_Utilities, Base_Frame;

type
  TServerPacket = record
    instanceKey : Integer;
    code, name, nameRead : String;
    birthday : TDate;
    sex : Integer;
    phone, address : String;
    score : Currency;
    scheduleIndex : Integer;
    overlapped : Boolean;
  end;

  TServerOfficialGraphFrame = class(TFrameTemplate)
    Region: TIBQuery;
    DataSourceOf_Region: TDataSource;

    Branch: TIBQuery;
    DataSourceOf_Branch: TDataSource;

    Servers: TIBQuery;
    Schedule: TIBClientDataSet;

    Selection: TIBClientDataSet;
    DataSourceOf_Selection: TDataSource;

    iglIcon: TImageList;

    cboRegion: TDBLookupComboBox;
    cboBranch: TDBLookupComboBox;

    txtProfileInfo: TEdit;
    btnSelectProfile: TButton;
    btnLookupProfile: TButton;

    cboSelection: TDBLookupComboBox;
    btnSelectSchedule: TButton;
    btnDeleteSchedule: TButton;

    gridGraph: TStringGrid;

    Shortcuts: TActionList;
    Shortcut_LookupProfile: TAction;
    radOrderByName: TRadioButton;
    radOrderByScore: TRadioButton;
    radOrderByScoreAscend: TRadioButton;

    procedure cboRegion_Click(Sender: TObject);
    procedure cboBranch_Click(Sender: TObject);

    procedure btnSelectProfile_Click(Sender: TObject);
    procedure btnLookupProfile_Click(Sender: TObject);

    procedure cboSelection_Click(Sender: TObject);
    procedure btnSelectSchedule_Click(Sender: TObject);
    procedure btnDeleteSchedule_Click(Sender: TObject);

    procedure gridGraph_DrawCell(Sender: TObject; ACol, ARow: Integer; Rect: TRect; State: TGridDrawState);

    procedure gridGraph_SelectCell(Sender: TObject; ACol, ARow: Integer; var CanSelect: Boolean);
    procedure gridGraph_MouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure gridGraph_DblClick(Sender: TObject);
    procedure gridGraph_KeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);

    procedure gridGraph_DragOver(Sender, Source: TObject; X, Y: Integer; State: TDragState; var Accept: Boolean);
    procedure gridGraph_DragDrop(Sender, Source: TObject; X, Y: Integer);

    procedure radOrder_Click(Sender: TObject);
  private
    { Private 錾 }
    FServerKey : Integer;
    FScheduleAt : TDate;
    FObsolete : Boolean;

    FEncloseFrom : Integer;
    FEncloseTo : Integer;

    FCursorRow : Integer;

    FServers : array of TServerPacket;
    FSchedules : array of TSchedulePacket;

    sv_keyInstance : Integer;
    sv_strServerCode, sv_strServerName, sv_strServerRead : Integer;
    sv_datServerBirthday, sv_typServerSex : Integer;
    sv_strServerPhone, sv_strServerAddress : Integer;

    sc_keyInstance, sc_typSchedule, sc_typProgress : Integer;
    sc_refSubject, sc_typSubject, sc_intSubjectAt, sc_intSubjectFor : Integer;
    sc_strService, sc_intMinutesFrom, sc_intMinutesTo : Integer;
    sc_typExchange, sc_curAmountPerCount, sc_curAmountPerHour : Integer;
    sc_typAddressFrom, sc_refAddressFrom, sc_strAddressFrom, sc_refResidentFrom, sc_intFromX, sc_intFromY : Integer;
    sc_typAddressTo, sc_refAddressTo, sc_strAddressTo, sc_refResidentTo, sc_intToX, sc_intToY : Integer;
    sc_refServer, sc_strServerCode, sc_strServerName, sc_strServerRead, sc_datServerBirthday, sc_typServerSex, sc_strServerAddress, sc_strServerPhone : Integer;
    sc_refRecipient, sc_refClient, sc_strClient : Integer;
    sc_strOpponent : Integer;

    property CursorRow : Integer read FCursorRow;

    function GetBranchKey() : Integer;
    function GetRegionKey() : Integer;
    function GetTopHour() : Integer;
  protected
    { Protected 錾 }
    function MinutesToSurface( Minutes : Integer) : Integer; overload;
    function MinutesToSurface( Minutes, Col : Integer) : Integer; overload;
    function ColFromX( X : Integer) : Integer;
    function RowFromY( Y : Integer) : Integer;
    function RowFromKey( keyProfile : Integer) : Integer;
    function PacketFromPos( X, Y : Integer) : TSchedulePacket;

    procedure ResetSelection( Row, keySchedule : Integer);
  public
    { Public 錾 }
    constructor Create(AOwner: TComponent); override;

    property ServerKey : Integer read FServerKey;
    property ScheduleAt : TDate read FScheduleAt;
    property Obsolete : Boolean read FObsolete write FObsolete;

    property EncloseFrom : Integer read FEncloseFrom;
    property EncloseTo : Integer read FEncloseTo;

    property BranchKey : Integer read GetBranchKey;
    property RegionKey : Integer read GetRegionKey;
    property TopHour : Integer read GetTopHour;

    procedure Reset( datAt : TDate);
    procedure Enclose( intMinutesFrom, intMinutesTo : Integer);

    procedure Lookup( keyProfile : Integer);
  end;

var
  ServerOfficialGraphFrame: TServerOfficialGraphFrame;

implementation

{$R *.dfm}

uses Piece_JobPalette, Chooser_Server, Set_Server, Set_Care, Set_Menage, Set_Migration, Set_Transfer, Misc_RouteBrowser;

{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

constructor TServerOfficialGraphFrame.Create(AOwner: TComponent);
begin
	inherited Create( AOwner);

	FServerKey := -1;
	FScheduleAt := Today;
	FObsolete := false;

	FEncloseFrom := -1;
	FEncloseTo := -1;

	FCursorRow := -1;

	gridGraph.ColWidths[ 0] := 120;
	gridGraph.LeftCol := 7;

	sc_keyInstance := -1; // see procedure Reset( datAt : TDate);

	Selection.Open;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TServerOfficialGraphFrame.GetBranchKey() : Integer;
begin
	result := cboBranch.KeyValue;
end;

function TServerOfficialGraphFrame.GetRegionKey() : Integer;
begin
	result := cboRegion.KeyValue;
end;

function TServerOfficialGraphFrame.GetTopHour() : Integer;
begin
	result := gridGraph.LeftCol;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TServerOfficialGraphFrame.MinutesToSurface( Minutes : Integer) : Integer;
begin
	result := Minutes * ( gridGraph.DefaultColWidth + 1) div 60;
end;

function TServerOfficialGraphFrame.MinutesToSurface( Minutes, Col : Integer) : Integer;
begin
	result := MinutesToSurface( Minutes) - Col * ( gridGraph.DefaultColWidth + 1);
end;

function TServerOfficialGraphFrame.ColFromX( X : Integer) : Integer;
begin
	with gridGraph do result := Min( ColCount - 1, Max( 0, LeftCol + ( X - ColWidths[ 0]) div ( DefaultColWidth + 1)));
end;

function TServerOfficialGraphFrame.RowFromY( Y : Integer) : Integer;
begin
	with gridGraph do result := Min( RowCount - 1, Max( 0, TopRow + Y div ( DefaultRowHeight + 1)));
end;

function TServerOfficialGraphFrame.RowFromKey( keyProfile : Integer) : Integer;
var
	index : Integer;
begin
	result := -1;
	for index := Low( FServers) to High( FServers) do
	begin
		if FServers[ index].instanceKey = keyProfile then
		begin
			result := index;
			exit;
		end;
	end;
end;

function TServerOfficialGraphFrame.PacketFromPos( X, Y : Integer) : TSchedulePacket;
var
	clickedX, currSchedule : Integer;
begin
	result.instanceKey := -1;
	result.scheduleType := -1;
	result.progress := -1;
	result.presence := -1;
	result.minutesFrom := -1;
	result.minutesTo := -1;
	result.subjectAt := -1;
	result.serverReady := false;
	result.subjectKey := -1;
	result.nextIndex := -1;

	if Schedule.IsEmpty then exit;

	with gridGraph do clickedX := ( X - ColWidths[ 0]) + ( gridGraph.DefaultColWidth + 1) * LeftCol;

	currSchedule := FServers[ RowFromY( Y)].scheduleIndex;
	while currSchedule >= 0 do
	begin
		with FSchedules[ currSchedule] do
		begin
			case scheduleType of
			CARE_SCHEDULE, MENAGE_SCHEDULE :
				if ( clickedX >= MinutesToSurface( minutesFrom)) and ( clickedX <= MinutesToSurface( minutesTo)) then
					result := FSchedules[ currSchedule];

			MIGRATION_SCHEDULE, TRANSFER_SCHEDULE :
				if ( clickedX >= MinutesToSurface( minutesFrom)) and ( clickedX <= MinutesToSurface( minutesFrom) + 20) then
					result := FSchedules[ currSchedule];
			end;
		end;
		currSchedule := FSchedules[ currSchedule].nextIndex;
	end;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TServerOfficialGraphFrame.ResetSelection( Row, keySchedule : Integer);

	procedure InvalidateRow( Row : Integer);
	var
		rectRedraw : TRect;
	begin
		rectRedraw := gridGraph.CellRect( 0, Row);
		rectRedraw.Left := 0;
		rectRedraw.Right := gridGraph.Width;
		InvalidateRect( gridGraph.Handle, @rectRedraw, false);
	end;
var
	currSchedule : Integer;
	addressFrom, addressTo, client, summary : String;
begin
	Row := Min( Row, High( FServers));

	if Row <> CursorRow then
	begin
		InvalidateRow( CursorRow);
		FCursorRow := Row;
		InvalidateRow( CursorRow);
	end;

	if Row < 0 then
	begin
		txtProfileInfo.Text := 'XXXXX / 񋟎ҁ@ / Ă傤@߂ / dbԍP / Z';
		btnSelectProfile.Enabled := false;
		FServerKey := -1;
		currSchedule := -1;
	end
	else
	begin
		with FServers[ Row] do txtProfileInfo.Text := Format( '%s / %s / %s / %s / %s', [ code, name, nameRead, phone, address]);
		btnSelectProfile.Enabled := true;
		FServerKey := FServers[ Row].instanceKey;
		currSchedule := FServers[ Row].scheduleIndex;
	end;

	Selection.DisableControls;
	Selection.Close;
	Selection.Open;
	while currSchedule >= 0 do
	begin
		with Schedule, FSchedules[ currSchedule] do
		begin
			RecNo := recordNo;

			summary := Format( '%.2d:%.2d ` %s', [ minutesFrom div 60, minutesFrom mod 60, Fields[ sc_strService].AsString]);

			if ( Fields[ sc_typAddressFrom].AsInteger = PROFILE_ADDRESS)
			    and ( Fields[ sc_refResidentFrom].AsInteger = Fields[ sc_refRecipient].AsInteger) then
				addressFrom := 'p҂̎'
			else
				addressFrom := Fields[ sc_strAddressFrom].AsString;

			if ( Fields[ sc_typAddressTo].AsInteger = PROFILE_ADDRESS)
			    and ( Fields[ sc_refResidentTo].AsInteger = Fields[ sc_refRecipient].AsInteger) then
				addressTo := 'p҂̎'
			else
				addressTo := Fields[ sc_strAddressTo].AsString;

			if not Fields[ sc_refClient].IsNull then
				client := Format( ' ~ %s', [ Fields[ sc_strClient].AsString])
			else
				client := '';

			case Fields[ sc_typSchedule].AsInteger of
			CARE_SCHEDULE, MENAGE_SCHEDULE :
				summary := Format( '%s %s%s', [ summary, addressFrom, client]);

			MIGRATION_SCHEDULE, TRANSFER_SCHEDULE :
				if presence = BEFORE_PRESENCE then
					summary := Format( '%s y%s  %sz%s', [ summary, addressTo, addressFrom, client])
				else
					summary := Format( '%s y%s  %sz%s', [ summary, addressFrom, addressTo, client]);
			end;

			case Fields[ sc_typSchedule].AsInteger of
			CARE_SCHEDULE, MENAGE_SCHEDULE :
				summary := Format( '%s ` %.2d:%.2d', [ summary, minutesTo div 60, minutesTo mod 60]);

			MIGRATION_SCHEDULE :
				if presence = BEFORE_PRESENCE then
					summary := Format( '%s ` %.2d:%.2d', [ summary, subjectAt div 60, subjectAt mod 60]);
			end;

			Selection.Append;
			Selection.FieldByName( 'keyRow').AsInteger := Row;
			Selection.FieldByName( 'keyInstance').AsInteger := instanceKey;
			Selection.FieldByName( 'typSchedule').AsInteger := scheduleType;
			Selection.FieldByName( 'intMinutesAt').AsInteger := minutesFrom;
			Selection.FieldByName( 'intMinutesFor').AsInteger := minutesTo - minutesFrom;
			Selection.FieldByName( 'strCaption').AsString := summary;
			Selection.Post;
			currSchedule := nextIndex;
		end;
	end;
	Selection.IndexFieldNames := 'intMinutesAt;intMinutesFor;strCaption';
	Selection.EnableControls;

	cboSelection.KeyValue := ZeroToNull( 0);
	cboSelection.KeyValue := keySchedule;
	btnSelectSchedule.Enabled := cboSelection.Text <> '';
	btnDeleteSchedule.Enabled := cboSelection.Text <> '';
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TServerOfficialGraphFrame.Reset( datAt : TDate);
var
	defaultBranch, defaultRegion : Integer;
	defaultRow, defaultSchedule : Integer;

	currServer, nextServer : Integer;
	currSchedule, nextSchedule : Integer;
begin
	defaultBranch := VarToInt( cboBranch.KeyValue);
	defaultRegion := VarToInt( cboRegion.KeyValue);
	defaultRow := Selection.FieldByName( 'keyRow').AsInteger;
	defaultSchedule := Selection.FieldByName( 'keyInstance').AsInteger;
	// cursor preservation

	FScheduleAt := datAt;
	// property update

	Branch.Close;
	Branch.ParamByName( 'datAt').AsDate := ScheduleAt;
	Branch.Open;
	Branch.FetchAll;
	SelectComboWithoutFail( cboBranch, defaultBranch);
	defaultBranch := cboBranch.KeyValue;

	Region.Close;
	Region.ParamByName( 'datAt').AsDate := ScheduleAt;
	Region.Open;
	Region.FetchAll;
	SelectComboWithoutFail( cboRegion, defaultRegion);
	defaultRegion := cboRegion.KeyValue;
	// Branch update

	Servers.Close;
	Servers.ParamByName( 'datAvailableAt').AsDate := ScheduleAt;
	Servers.ParamByName( 'optAvailability').AsInteger := 1 shl ( DayOfWeek( ScheduleAt) - 1);
	Servers.ParamByName( 'optToday').AsInteger := 1 shl ( DayOfTheMonth( datAt) - 1);
	Servers.ParamByName( 'keyBranch').AsInteger := defaultBranch;
	Servers.ParamByName( 'keyRegion').AsInteger := defaultRegion;
	Servers.Open;
	Servers.FetchAll;

	Schedule.Close;
	Schedule.Params.ParamByName( 'datSchedule').AsDate := ScheduleAt;
	Schedule.Params.ParamByName( 'keyBranch').AsInteger := defaultBranch;
	Schedule.Params.ParamByName( 'keyRegion').AsInteger := defaultRegion;
	Schedule.Open;
	// Schedule update

	if sc_keyInstance < 0 then
	begin
		sv_keyInstance := Servers.FieldByName( 'keyInstance').Index;
		sv_strServerCode := Servers.FieldByName( 'strServerCode').Index;
		sv_strServerName := Servers.FieldByName( 'strServerName').Index;
		sv_strServerRead := Servers.FieldByName( 'strServerRead').Index;
		sv_datServerBirthday := Servers.FieldByName( 'datServerBirthday').Index;
		sv_typServerSex := Servers.FieldByName( 'typServerSex').Index;
		sv_strServerPhone := Servers.FieldByName( 'strServerPhone').Index;
		sv_strServerAddress := Servers.FieldByName( 'strServerAddress').Index;

		sc_keyInstance := Schedule.FieldByName( 'keyInstance').Index;
		sc_typSchedule := Schedule.FieldByName( 'typSchedule').Index;
		sc_typProgress := Schedule.FieldByName( 'typProgress').Index;
		sc_refSubject := Schedule.FieldByName( 'refSubject').Index;
		sc_typSubject := Schedule.FieldByName( 'typSubject').Index;
		sc_intSubjectAt := Schedule.FieldByName( 'intSubjectAt').Index;
		sc_intSubjectFor := Schedule.FieldByName( 'intSubjectFor').Index;
		sc_strService := Schedule.FieldByName( 'strService').Index;
		sc_intMinutesFrom := Schedule.FieldByName( 'intMinutesFrom').Index;
		sc_intMinutesTo := Schedule.FieldByName( 'intMinutesTo').Index;
		sc_typExchange := Schedule.FieldByName( 'typExchange').Index;
		sc_curAmountPerCount := Schedule.FieldByName( 'curAmountPerCount').Index;
		sc_curAmountPerHour := Schedule.FieldByName( 'curAmountPerHour').Index;
		sc_typAddressFrom := Schedule.FieldByName( 'typAddressFrom').Index;
		sc_refAddressFrom := Schedule.FieldByName( 'refAddressFrom').Index;
		sc_strAddressFrom := Schedule.FieldByName( 'strAddressFrom').Index;
		sc_refResidentFrom := Schedule.FieldByName( 'refResidentFrom').Index;
		sc_intFromX := Schedule.FieldByName( 'intFromX').Index;
		sc_intFromY := Schedule.FieldByName( 'intFromY').Index;
		sc_typAddressTo := Schedule.FieldByName( 'typAddressTo').Index;
		sc_refAddressTo := Schedule.FieldByName( 'refAddressTo').Index;
		sc_strAddressTo := Schedule.FieldByName( 'strAddressTo').Index;
		sc_refResidentTo := Schedule.FieldByName( 'refResidentTo').Index;
		sc_intToX := Schedule.FieldByName( 'intToX').Index;
		sc_intToY := Schedule.FieldByName( 'intToY').Index;
		sc_refServer  := Schedule.FieldByName( 'refServer').Index;
		sc_strServerCode := Schedule.FieldByName( 'strServerCode').Index;
		sc_strServerName := Schedule.FieldByName( 'strServerName').Index;
		sc_strServerRead := Schedule.FieldByName( 'strServerRead').Index;
		sc_datServerBirthday := Schedule.FieldByName( 'datServerBirthday').Index;
		sc_typServerSex := Schedule.FieldByName( 'typServerSex').Index;
		sc_strServerPhone := Schedule.FieldByName( 'strServerPhone').Index;
		sc_strServerAddress := Schedule.FieldByName( 'strServerAddress').Index;
		sc_refRecipient := Schedule.FieldByName( 'refRecipient').Index;
		sc_refClient := Schedule.FieldByName( 'refClient').Index;
		sc_strClient := Schedule.FieldByName( 'strClient').Index;
		sc_strOpponent := Schedule.FieldByName( 'strOpponent').Index;
	end;
	// prepare index cache

	FServers := nil;
	nextServer := 0;
	SetLength( FServers, Max( 1, Servers.RecordCount + Schedule.RecordCount));
	FServers[ 0].scheduleIndex := -1;

	FSchedules := nil;
	nextSchedule := 0;
	SetLength( FSchedules, Schedule.RecordCount);

	Servers.First;
	with Servers do while not EOF do
	begin
		currServer := RowFromKey( Fields[ sv_keyInstance].AsInteger);
		if currServer < 0 then
		begin
			currServer := nextServer;
			with FServers[ currServer] do
			begin
				instanceKey := Fields[ sv_keyInstance].AsInteger;
				code := Fields[ sv_strServerCode].AsString;
				name := Fields[ sv_strServerName].AsString;
				nameRead := Fields[ sv_strServerRead].AsString;
				birthday := Fields[ sv_datServerBirthday].AsDateTime;
				sex := Fields[ sv_typServerSex].AsInteger;
				phone := Fields[ sv_strServerPhone].AsString;
				address := Fields[ sv_strServerAddress].AsString;
				score := 0;
				scheduleIndex := -1;
			end;
			Inc( nextServer);
		end;
		Next;
	end;

	Schedule.First;
	with Schedule do while not EOF do
	begin
		currServer := RowFromKey( Fields[ sc_refServer].AsInteger);
		if currServer < 0 then
		begin
			currServer := nextServer;
			with FServers[ currServer] do
			begin
				instanceKey := Fields[ sc_refServer].AsInteger;
				code := Fields[ sc_strServerCode].AsString;
				name := Fields[ sc_strServerName].AsString;
				nameRead := Fields[ sc_strServerRead].AsString;
				birthday := Fields[ sc_datServerBirthday].AsDateTime;
				sex := Fields[ sc_typServerSex].AsInteger;
				phone := Fields[ sc_strServerPhone].AsString;
				address := Fields[ sc_strServerAddress].AsString;
				score := 0;
				scheduleIndex := -1;
				overlapped := false;
			end;
			Inc( nextServer);
		end;

		with FServers[ currServer] do
		begin
			score := score + AmountOf(
				Fields[ sc_typExchange].AsInteger,
				Fields[ sc_curAmountPerCount].AsCurrency,
				Fields[ sc_curAmountPerHour].AsCurrency,
				Fields[ sc_intMinutesTo].AsInteger - Fields[ sc_intMinutesFrom].AsInteger
			);
		end;

		if FServers[ currServer].scheduleIndex >= 0 then
		begin
			currSchedule := FServers[ currServer].scheduleIndex;
			while true do
			begin
				case Fields[ sc_typSchedule].AsInteger of CARE_SCHEDULE, MENAGE_SCHEDULE :
					if IntersectionOf(
						FSchedules[ currSchedule].minutesFrom, FSchedules[ currSchedule].minutesTo,
						Fields[ sc_intMinutesFrom].AsInteger, Fields[ sc_intMinutesTo].AsInteger
					) > 5 then
					begin
						FServers[ currServer].overlapped := true;
					end;
				else
					if ( FSchedules[ currSchedule].scheduleType = Fields[ sc_typSchedule].AsInteger)
					and ( FSchedules[ currSchedule].minutesFrom = Fields[ sc_intMinutesFrom].AsInteger)
					and ( FSchedules[ currSchedule].minutesTo = Fields[ sc_intMinutesTo].AsInteger)
					then
					begin
						FServers[ currServer].overlapped := true;
					end;
				end;
				if FSchedules[ currSchedule].nextIndex < 0 then break;
				currSchedule := FSchedules[ currSchedule].nextIndex;
			end;
			FSchedules[ currSchedule].nextIndex := nextSchedule;
		end
		else
		begin
			FServers[ currServer].scheduleIndex := nextSchedule;
		end;

		with FSchedules[ nextSchedule] do
		begin
			instanceKey := Fields[ sc_keyInstance].AsInteger;
			scheduleType := Fields[ sc_typSchedule].AsInteger;
			progress := Fields[ sc_typProgress].AsInteger;
			minutesFrom := Fields[ sc_intMinutesFrom].AsInteger;
			minutesTo := Fields[ sc_intMinutesTo].AsInteger;
			fromX := Fields[ sc_intFromX].AsInteger;
			fromY := Fields[ sc_intFromY].AsInteger;
			toX := Fields[ sc_intToX].AsInteger;
			toY := Fields[ sc_intToY].AsInteger;
			subjectAt := Fields[ sc_intSubjectAt].AsInteger;
			opponent := Fields[ sc_strOpponent].AsString;
			serverReady := true;
			subjectKey := Fields[ sc_refSubject].AsInteger;
			case scheduleType of
			MIGRATION_SCHEDULE :
				if minutesFrom < subjectAt + Fields[ sc_intSubjectFor].AsInteger div 2 then
					presence := BEFORE_PRESENCE
				else
					presence := AFTER_PRESENCE;

			TRANSFER_SCHEDULE :
				if ( subjectKey > 0) and ( Fields[ sc_typSubject].AsInteger <> TRANSFER_SCHEDULE) then
				begin
					if minutesFrom < subjectAt + Fields[ sc_intSubjectFor].AsInteger div 2 then
						presence := BEFORE_PRESENCE
					else
						presence := AFTER_PRESENCE;
				end
				else
					presence := FULL_PRESENCE;
			else
				presence := RAW_PRESENCE;
			end;
			recordNo := RecNo;
			nextIndex := -1;
		end;
		Inc( nextSchedule);

		Next;
	end;
	SetLength( FServers, Max( nextServer, 1));
	// Servers and Packets setup

	gridGraph.RowCount := Length( FServers);
	radOrder_Click( self);
	// grid setup

	ResetSelection( defaultRow, defaultSchedule);
	// Selection update

	btnLookupProfile.Enabled := not Schedule.IsEmpty;
	Obsolete := false;
end;

procedure TServerOfficialGraphFrame.Enclose( intMinutesFrom, intMinutesTo : Integer);
var
	leftBase : Integer;
	rectRedraw : TRect;
begin
        leftBase := - gridGraph.ColWidths[ 0] + gridGraph.LeftCol * ( gridGraph.DefaultColWidth + 1);

	rectRedraw.Top := 0;
	rectRedraw.Bottom := gridGraph.Height;
	rectRedraw.Left := MinutesToSurface( FEncloseFrom) - leftBase - 2;
	rectRedraw.Right := MinutesToSurface( FEncloseTo) - leftBase + 2;
	InvalidateRect( gridGraph.Handle, @rectRedraw, false);

	FEncloseFrom := intMinutesFrom;
	FEncloseTo := intMinutesTo;

	rectRedraw.Top := 0;
	rectRedraw.Bottom := gridGraph.Height;
	rectRedraw.Left := MinutesToSurface( FEncloseFrom) - leftBase - 2;
	rectRedraw.Right := MinutesToSurface( FEncloseTo) - leftBase + 2;
	InvalidateRect( gridGraph.Handle, @rectRedraw, false);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TServerOfficialGraphFrame.Lookup( keyProfile : Integer);
var
	ARow : Integer;
begin
	ARow := RowFromKey( keyProfile);
	if ARow < 0 then exit;
	gridGraph.TopRow := ARow;
	ResetSelection( ARow, -1);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TServerOfficialGraphFrame.cboRegion_Click(Sender: TObject);
begin
	Reset( ScheduleAt);
end;

procedure TServerOfficialGraphFrame.cboBranch_Click(Sender: TObject);
begin
	Reset( ScheduleAt);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TServerOfficialGraphFrame.btnSelectProfile_Click(Sender: TObject);
begin
	TServerSet.Prepare.Select( ServerKey);
end;

procedure TServerOfficialGraphFrame.btnLookupProfile_Click(Sender: TObject);
var
	keyServer : Integer;
begin
	if ChooseServer( DataStore.MainTransaction, ScheduleAt, keyServer) then Lookup( keyServer);
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TServerOfficialGraphFrame.cboSelection_Click(Sender: TObject);
begin
	btnSelectSchedule.Enabled := cboSelection.Text <> '';
	btnDeleteSchedule.Enabled := cboSelection.Text <> '';
end;

procedure TServerOfficialGraphFrame.btnSelectSchedule_Click(Sender: TObject);
begin
	case Selection.FieldByName( 'typSchedule').AsInteger of
	     CARE_SCHEDULE : TCareSet.Prepare.Select( Selection.FieldByName( 'keyInstance').AsInteger);
	   MENAGE_SCHEDULE : TMenageSet.Prepare.Select( Selection.FieldByName( 'keyInstance').AsInteger);
	MIGRATION_SCHEDULE : TMigrationSet.Prepare.Select( Selection.FieldByName( 'keyInstance').AsInteger);
	 TRANSFER_SCHEDULE : TTransferSet.Prepare.Select( Selection.FieldByName( 'keyInstance').AsInteger);
	end;
end;

procedure TServerOfficialGraphFrame.btnDeleteSchedule_Click(Sender: TObject);
begin
	case Selection.FieldByName( 'typSchedule').AsInteger of
	     CARE_SCHEDULE : TCareSet.Prepare.Delete( Selection.FieldByName( 'keyInstance').AsInteger);
	   MENAGE_SCHEDULE : TMenageSet.Prepare.Delete( Selection.FieldByName( 'keyInstance').AsInteger);
	MIGRATION_SCHEDULE : TMigrationSet.Prepare.Delete( Selection.FieldByName( 'keyInstance').AsInteger);
	 TRANSFER_SCHEDULE : TTransferSet.Prepare.Delete( Selection.FieldByName( 'keyInstance').AsInteger);
	end;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TServerOfficialGraphFrame.gridGraph_DrawCell(Sender: TObject; ACol, ARow: Integer; Rect: TRect; State: TGridDrawState);
var
	defBrushColor, defPenColor : TColor;
	defBrushStyle : TBrushStyle;
	defPenStyle : TPenStyle;
	defClipRect : TRect;
	strScore : String;
	rectEnclosure : TRect;
	leftLimit, rightLimit : Integer;
	currSchedule : Integer;
begin
	with gridGraph.Canvas do
	begin
		defBrushColor := Brush.Color;
		defBrushStyle := Brush.Style;
		defPenColor := Pen.Color;
		defPenStyle := Pen.Style;
		defClipRect := ClipRect;
		IntersectClipRect( Handle, Rect.Left, Rect.Top, Rect.Right, Rect.Bottom);
	end;
	// backup default status

	with gridGraph.Canvas do if ACol = 0 then
	begin
		Font.Color := clBtnText;
		Font.Size := 8;
		if not FServers[ ARow].overlapped then
		begin
			if ARow = CursorRow then
				Brush.Color := TColor( $AAAAAA)
			else
				Brush.Color := clSilver;
		end
		else
		begin
			if ARow = CursorRow then
				Brush.Color := TColor( $9999CC)
			else
				Brush.Color := TColor( $AAAAEE);
		end;
		Brush.Style := bsSolid;
		if FServers[ ARow].instanceKey > 0 then
		begin
			TextRect( Rect, Rect.Left + 2, Rect.Top + 1, FServers[ ARow].code);
			TextOut( Rect.Left + 2, Rect.Top + 12, FServers[ ARow].name);
			TextOut( Rect.Left + 2, Rect.Top + 24, Format( '%d %s', [
				YearsBetween( Today, FServers[ ARow].birthday),
				aSex[ FServers[ ARow].sex]
			]));

			if FServers[ ARow].score > 0 then
			begin
				Font.Size := 12;
				strScore := Format( '%d', [ Round( FServers[ ARow].score)]);
				leftLimit := Rect.Right - 2 - TextWidth( strScore);
				SetBkMode( Handle, TRANSPARENT);
				TextOut( leftLimit, Rect.Top + 10, strScore);
				TextOut( leftLimit - 1, Rect.Top + 10, strScore);
				TextOut( leftLimit, Rect.Top + 11, strScore);
				TextOut( leftLimit - 1, Rect.Top + 11, strScore);
			end;
		end
		else
		begin
			FillRect( Rect);
		end;
		// draw server information on header column
	end
	else
	begin
		if ARow = CursorRow then
			Brush.Color := TColor( $DDDDDD)
		else
			Brush.Color := gridGraph.Color;
		Brush.Style := bsSolid;
		FillRect( Rect);

		if ( ( FEncloseFrom div 60) <= ACol) and ( ACol <= ( FEncloseTo div 60)) then
		begin
			if ARow = CursorRow then
				Brush.Color := TColor( $E0D0D0)
			else
				Brush.Color := TColor( $F0E0E0);
			rectEnclosure := Rect;
			rectEnclosure.Left := Rect.Left + MinutesToSurface( FEncloseFrom, ACol);
			rectEnclosure.Right := Rect.Left + MinutesToSurface( FEncloseTo, ACol);
			FillRect( rectEnclosure);
		end;
		// paint backcolor

		Pen.Color := clLtGray;
		Pen.Style := psDot;
		MoveTo( Rect.Left, Rect.Top + 18);
		LineTo( Rect.Right, Rect.Top + 18);
		// draw schedule axis

		SetBkMode( Handle, TRANSPARENT);
		if ( ACol mod 4 = 0) or ( ARow mod 4 = 0) then
		begin
			Font.Color := clWhite;
			Font.Color := clGray;
			Font.Size := 8;
			TextOut( Rect.Left + 1, Rect.Top + 1,
				IfThen( ( ACol div 12) mod 2 = 0,
					Format( 'AM%d:00', [ ACol mod 12]),
					Format( 'PM%d:00', [ ACol mod 12])
				)
			);
		end;
		// draw time label

		Font.Color := clBlack;
		Font.Size := 8;
		if gridGraph.Objects[ ACol, ARow] <> nil then
		begin
			currSchedule := FServers[ ARow].scheduleIndex;
			while currSchedule >= 0 do
			begin
				with FSchedules[ currSchedule] do case scheduleType of
				CARE_SCHEDULE, MENAGE_SCHEDULE :
					begin
						Brush.Color := IfThen( scheduleType = CARE_SCHEDULE, clCare, clMenage);
						Brush.Style := bsSolid;
						if progress = UNCLEAR_PROGRESS then Brush.Style := bsDiagCross;

						Pen.Color := Brush.Color;
						Pen.Style := psSolid;

						leftLimit := Rect.Left + MinutesToSurface( minutesFrom, ACol);
						rightLimit := Rect.Left + MinutesToSurface( minutesTo, ACol);
						Rectangle( leftLimit, Rect.Top + 13, rightLimit, Rect.Top + 24);
					end;

				MIGRATION_SCHEDULE :
					begin
						leftLimit := Rect.Left + MinutesToSurface( minutesFrom, ACol);
						rightLimit := leftLimit + 22;
						iglIcon.Draw( gridGraph.Canvas, leftLimit, Rect.Top + 8, IfThen( progress <> UNCLEAR_PROGRESS, 4, 5));
					end;

				TRANSFER_SCHEDULE :
					if presence = FULL_PRESENCE then
					begin
						leftLimit := Rect.Left + MinutesToSurface( minutesFrom, ACol);
						rightLimit := leftLimit + 22;
						iglIcon.Draw( gridGraph.Canvas, leftLimit, Rect.Top + 8, IfThen( progress <> UNCLEAR_PROGRESS, 0, 1));
					end
					else
					begin
						leftLimit := Rect.Left + MinutesToSurface( minutesFrom, ACol);
						rightLimit := leftLimit + 22;
						iglIcon.Draw( gridGraph.Canvas, leftLimit, Rect.Top + 8, IfThen( progress <> UNCLEAR_PROGRESS, 2, 3));
					end;
				end;

				SetBkMode( Handle, TRANSPARENT);
				TextOut(
					( ( leftLimit + rightLimit) - TextWidth( FSchedules[ currSchedule].opponent)) div 2,
					Rect.Top + 26,
					FSchedules[ currSchedule].opponent
				);

				currSchedule := FSchedules[ currSchedule].nextIndex;
			end;
		end;
		// draw schedule

		if ( ACol mod 24) = HourOf( Now) then
		begin
			Pen.Color := clRed;
			Pen.Style := psSolid;
			MoveTo( Rect.Left + MinutesToSurface( MinuteOf( Now), 0), Rect.Top);
			LineTo( Rect.Left + MinutesToSurface( MinuteOf( Now), 0), Rect.Bottom);
		end;
		// draw current time indicator

		if ACol = ( FEncloseFrom div 60) then
		begin
			Pen.Color := clNavy;
			Pen.Style := psSolid;
			MoveTo( Rect.Left + MinutesToSurface( FEncloseFrom, ACol), Rect.Top);
			LineTo( Rect.Left + MinutesToSurface( FEncloseFrom, ACol), Rect.Bottom);
		end;

		if ACol = ( FEncloseTo div 60) then
		begin
			Pen.Color := clNavy;
			Pen.Style := psSolid;
			MoveTo( Rect.Left + MinutesToSurface( FEncloseTo, ACol), Rect.Top);
			LineTo( Rect.Left + MinutesToSurface( FEncloseTo, ACol), Rect.Bottom);
		end;
		// draw request time indicator
	end;

	with gridGraph.Canvas do
	begin
		Brush.Color := defBrushColor;
		Brush.Style := defBrushStyle;
		Pen.Color := defPenColor;
		Pen.Style := defPenStyle;
		SelectClipRgn( Handle, 0);
		IntersectClipRect( Handle, defClipRect.Left, defClipRect.Top, defClipRect.Right, defClipRect.Bottom);
	end;
	// restore default status
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TServerOfficialGraphFrame.gridGraph_SelectCell(Sender: TObject; ACol, ARow: Integer; var CanSelect: Boolean);
begin
	CanSelect := false;
end;

procedure TServerOfficialGraphFrame.gridGraph_MouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
	if Button = mbLeft then ResetSelection( RowFromY( Y), PacketFromPos( X, Y).instanceKey);
end;

procedure TServerOfficialGraphFrame.gridGraph_DblClick(Sender: TObject);
var
	clickedPos : TPoint;
	scheduleKey : Integer;
begin
	clickedPos := gridGraph.ScreenToClient( Mouse.CursorPos);

	with gridGraph do if ( TopRow + clickedPos.Y div ( DefaultRowHeight + 1)) >= RowCount then exit;

	if clickedPos.X < gridGraph.ColWidths[ 0] then
	begin
		btnSelectProfile.Click;
		exit;
	end;

	scheduleKey := PacketFromPos( clickedPos.X, clickedPos.Y).instanceKey;
	if scheduleKey > 0 then
	begin
		cboSelection.KeyValue := ZeroToNull( 0);
		cboSelection.KeyValue := scheduleKey;
		btnSelectSchedule.Enabled := cboSelection.Text <> '';
		btnDeleteSchedule.Enabled := cboSelection.Text <> '';
		if btnSelectSchedule.Enabled then btnSelectSchedule.Click;
		exit;
	end;

	if clickedPos.X < gridGraph.ColWidths[ 0] then btnSelectProfile.Click;
end;

procedure TServerOfficialGraphFrame.gridGraph_KeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
	case Key of
	 VK_LEFT : gridGraph.LeftCol := Max( 1, gridGraph.LeftCol - 1);
	VK_RIGHT : gridGraph.LeftCol := Min( gridGraph.ColCount - gridGraph.VisibleColCount, gridGraph.LeftCol + 1);
	   VK_UP : gridGraph.TopRow := Max( 0, gridGraph.TopRow - 1);
	 VK_DOWN : gridGraph.TopRow := Min( gridGraph.RowCount - gridGraph.VisibleRowCount, gridGraph.TopRow + 1);
	end;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TServerOfficialGraphFrame.gridGraph_DragOver(Sender, Source: TObject; X, Y: Integer; State: TDragState; var Accept: Boolean);
var
	currSchedule : Integer;
begin
	if FCursorRow <> RowFromY( Y) then
	begin
		RouteBrowser.StartRefresh;
		currSchedule := FServers[ RowFromY( Y)].scheduleIndex;
		while currSchedule >= 0 do
		begin
			with FSchedules[ currSchedule] do RouteBrowser.RegisterSchedule(
				minutesFrom, minutesTo - minutesFrom,
				fromX, fromY,
				toX, toY
			);
			currSchedule := FSchedules[ currSchedule].nextIndex;
		end;
		RouteBrowser.FinishRefresh;
	end;

	gridGraph_MouseDown( Sender, mbLeft, [], X, Y);

	Accept := false;

	if FServers[ 0].instanceKey <= 0 then exit;
	with gridGraph do if ( TopRow + Y div ( DefaultRowHeight + 1)) >= RowCount then exit;

	Accept := true;
end;

procedure TServerOfficialGraphFrame.gridGraph_DragDrop(Sender, Source: TObject; X, Y: Integer);
begin
	with DragSourceToJobPalette( Source) do case JobType of
	     CARE_SCHEDULE : TCareSet.Arrange( FServers[ RowFromY( Y)].instanceKey).Select( JobKey);
	   MENAGE_SCHEDULE : TMenageSet.Arrange( FServers[ RowFromY( Y)].instanceKey).Select( JobKey);
	MIGRATION_SCHEDULE : TMigrationSet.Arrange( FServers[ RowFromY( Y)].instanceKey).Select( JobKey);
	 TRANSFER_SCHEDULE : TTransferSet.Arrange( FServers[ RowFromY( Y)].instanceKey).Select( JobKey);
	end;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TServerOfficialGraphFrame.radOrder_Click(Sender: TObject);

	procedure SortServer( leftMost, rightMost : Integer);
	var
		leftEdge, rightEdge : Integer;
		criterion, tempServer : TServerPacket;

		function CompareServer( const Server1, Server2 : TServerPacket) : Integer;
		begin
			if radOrderByScore.Checked then
			begin
				result := Round( Server2.score - Server1.score);
				if result <> 0 then exit;
			end;

			if radOrderByScoreAscend.Checked then
			begin
				result := Round( Server1.score - Server2.score);
				if result <> 0 then exit;
			end;

			result := CompareStr( Server1.nameRead, Server2.nameRead);
			if result <> 0 then exit;

			result := CompareStr( Server1.name, Server2.name);
			if result <> 0 then exit;

			result := Server1.instanceKey - Server2.instanceKey;
		end;
	begin
		if leftMost >= rightMost then exit;
		criterion := FServers[ ( leftMost + rightMost) div 2];
		leftEdge := leftMost;
		rightEdge := rightMost;

		while true do
		begin
			while ( leftEdge < rightEdge)
				and ( CompareServer( FServers[ leftEdge], criterion) < 0)
					do Inc( leftEdge);

			while ( rightEdge > leftEdge)
				and ( CompareServer( FServers[ rightEdge], criterion) > 0)
					do Dec( rightEdge);

			if leftEdge = rightEdge then break;

			tempServer := FServers[ leftEdge];
			FServers[ leftEdge] := FServers[ rightEdge];
			FServers[ rightEdge] := tempServer;
		end;

		SortServer( leftMost, leftEdge);
		SortServer( leftEdge + 1, rightMost);
	end;
var
	ARow, ACol : Integer;
	currSchedule : Integer;
begin
	SortServer( Low( FServers), High( FServers));

	for ARow := 0 to gridGraph.RowCount - 1 do
		for ACol := 0 to gridGraph.ColCount - 1 do
			gridGraph.Objects[ ACol, ARow] := nil;

	if not Schedule.IsEmpty then
	begin
		for ARow := 0 to gridGraph.RowCount - 1 do
		begin
			currSchedule := FServers[ ARow].scheduleIndex;
			while currSchedule >= 0 do
                        begin
				with FSchedules[ currSchedule] do case scheduleType of CARE_SCHEDULE, MENAGE_SCHEDULE :
					for ACol := minutesFrom div 60 to ( minutesTo - 1) div 60 do
						gridGraph.Objects[ ACol, ARow] := Schedule;
				else
					for ACol := minutesFrom div 60 to ( minutesTo + 22) div 60 do
						gridGraph.Objects[ ACol, ARow] := Schedule;
				end;
				currSchedule := FSchedules[ currSchedule].nextIndex;
			end;
		end;
	end;

	gridGraph.Invalidate;
end;

end.
