unit Frame_ServerPrivateGraph;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, MainDatastore, DB, DBClient, Provider, IBCustomDataSet, IBQuery,
  StdCtrls, Grids, DBCtrls, IBSQL, IBDatabase, DBLocal, DBLocalI, Math, ImgList,
  DateUtils, StrUtils, ActnList, Misc_Constants, Misc_Utilities, Base_Frame;

type
  TServerPrivateGraphFrame = class(TFrameTemplate)
    Profile: TIBQuery;
    Absence: TIBQuery;
    Schedule: TIBClientDataSet;
    Selection: TIBClientDataSet;
    DataSourceOf_Selection: TDataSource;

    iglIcon: TImageList;

    txtProfileInfo: TEdit;
    btnSelectProfile: TButton;
    btnChooseProfile: TButton;

    cboSelection: TDBLookupComboBox;
    btnSelectSchedule: TButton;
    btnDeleteSchedule: TButton;

    gridGraph: TStringGrid;

    Shortcuts: TActionList;
    Shortcut_SelectProfile: TAction;

    procedure btnSelectProfile_Click(Sender: TObject);
    procedure btnChooseProfile_Click(Sender: TObject);

    procedure cboSelection_Click(Sender: TObject);
    procedure btnSelectSchedule_Click(Sender: TObject);
    procedure btnDeleteSchedule_Click(Sender: TObject);

    procedure gridGraph_DrawCell(Sender: TObject; ACol, ARow: Integer; Rect: TRect; State: TGridDrawState);

    procedure gridGraph_SelectCell(Sender: TObject; ACol, ARow: Integer; var CanSelect: Boolean);
    procedure gridGraph_MouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure gridGraph_DblClick(Sender: TObject);
    procedure gridGraph_KeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);

    procedure gridGraph_DragOver(Sender, Source: TObject; X, Y: Integer; State: TDragState; var Accept: Boolean);
    procedure gridGraph_DragDrop(Sender, Source: TObject; X, Y: Integer);
  private
    { Private 錾 }
    FServerKey : Integer;
    FScheduleFrom : TDate;
    FScheduleTo : TDate;
    FObsolete : Boolean;

    FEncloseFrom : Integer;
    FEncloseTo : Integer;

    FCursorRow : Integer;

    FCalendars : array of TCalendarPacket;
    FSchedules : array of TSchedulePacket;

    FAvailability : Integer;

    sc_keyInstance, sc_typProgress : Integer;
    sc_datSchedule, sc_typSchedule, sc_strService, sc_intMinutesFrom, sc_intMinutesTo : Integer;
    sc_refSubject, sc_typSubject, sc_intSubjectAt, sc_intSubjectFor : Integer;
    sc_typAddressFrom, sc_refAddressFrom, sc_strAddressFrom, sc_refResidentFrom : Integer;
    sc_typAddressTo, sc_refAddressTo, sc_strAddressTo, sc_refResidentTo : Integer;
    sc_refRecipient : Integer;
    sc_refClient, sc_strClient : Integer;
    sc_refServer, sc_strServer : Integer;
    sc_strOpponent : Integer;

    property CursorRow : Integer read FCursorRow;
  protected
    { Protected 錾 }
    function MinutesToSurface( Minutes : Integer) : Integer; overload;
    function MinutesToSurface( Minutes, Col : Integer) : Integer; overload;
    function ColFromX( X : Integer) : Integer;
    function RowFromY( Y : Integer) : Integer;
    function DateFromRow( ARow : Integer) : TDate;
    function RowFromDate( datAt : TDate) : Integer;
    function PacketFromPos( X, Y : Integer) : TSchedulePacket;

    procedure ResetSelection( Row, keySchedule : Integer);
  public
    { Public 錾 }
    constructor Create(AOwner: TComponent); override;

    property ServerKey : Integer read FServerKey;
    property ScheduleFrom : TDate read FScheduleFrom;
    property ScheduleTo : TDate read FScheduleTo;
    property Obsolete : Boolean read FObsolete write FObsolete;

    function ScheduleAt() : TDate;

    procedure Reset( keyProfile : Integer; datFrom, datTo : TDate);
    procedure Enclose( intMinutesFrom, intMinutesTo : Integer);
  end;

var
  ServerPrivateGraphFrame: TServerPrivateGraphFrame;

implementation

{$R *.dfm}

uses Piece_JobPalette, Chooser_Server, Set_Server, Set_Care, Set_Menage, Set_Migration, Set_Transfer;

{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

constructor TServerPrivateGraphFrame.Create(AOwner: TComponent);
begin
	inherited Create( AOwner);

	FServerKey := -1;
	FScheduleFrom := Today;
	FScheduleTo := Today;
	FObsolete := false;

	FEncloseFrom := -1;
	FEncloseTo := -1;

	FCursorRow := -1;

	gridGraph.ColWidths[ 0] := 80;
	gridGraph.LeftCol := 7;

	sc_keyInstance := -1; // see procedure Reset( keyProfile : Integer; datFrom, datTo : TDate);

	Selection.Open;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TServerPrivateGraphFrame.ScheduleAt() : TDate;
begin
	if Selection.IsEmpty then
		result := ScheduleFrom
	else
		result := DateFromRow( Selection.FieldByName( 'keyRow').AsInteger);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TServerPrivateGraphFrame.MinutesToSurface( Minutes : Integer) : Integer;
begin
	result := Minutes * ( gridGraph.DefaultColWidth + 1) div 60;
end;

function TServerPrivateGraphFrame.MinutesToSurface( Minutes, Col : Integer) : Integer;
begin
	result := MinutesToSurface( Minutes) - Col * ( gridGraph.DefaultColWidth + 1);
end;

function TServerPrivateGraphFrame.ColFromX( X : Integer) : Integer;
begin
	with gridGraph do result := Min( ColCount - 1, Max( 0, LeftCol + ( X - ColWidths[ 0]) div ( DefaultColWidth + 1)));
end;

function TServerPrivateGraphFrame.RowFromY( Y : Integer) : Integer;
begin
	with gridGraph do result := Min( RowCount - 1, Max( 0, TopRow + Y div ( DefaultRowHeight + 1)));
end;

function TServerPrivateGraphFrame.DateFromRow( ARow : Integer) : TDate;
begin
	result := Max( ScheduleFrom, Min( IncDay( ScheduleFrom, ARow), ScheduleTo));
end;

function TServerPrivateGraphFrame.RowFromDate( datAt : TDate) : Integer;
var
	index : Integer;
begin
	result := -1;
	for index := Low( FCalendars) to High( FCalendars) do
	begin
		if FCalendars[ index].calendar = datAt then
		begin
			result := index;
			exit;
		end;
	end;
end;

function TServerPrivateGraphFrame.PacketFromPos( X, Y : Integer) : TSchedulePacket;
var
	clickedX, currSchedule : Integer;
begin
	result.instanceKey := -1;
	result.scheduleType := -1;
	result.progress := -1;
	result.presence := -1;
	result.minutesFrom := -1;
	result.minutesTo := -1;
	result.subjectAt := -1;
	result.serverReady := false;
	result.subjectKey := -1;
	result.nextIndex := -1;

	if Schedule.IsEmpty then exit;

	with gridGraph do clickedX := ( X - ColWidths[ 0]) + ( gridGraph.DefaultColWidth + 1) * LeftCol;

	currSchedule := FCalendars[ RowFromY( Y)].scheduleIndex;
	while currSchedule >= 0 do
	begin
		with FSchedules[ currSchedule] do
		begin
			case scheduleType of
			CARE_SCHEDULE, MENAGE_SCHEDULE :
				if ( clickedX >= MinutesToSurface( minutesFrom)) and ( clickedX <= MinutesToSurface( minutesTo)) then
					result := FSchedules[ currSchedule];

			MIGRATION_SCHEDULE, TRANSFER_SCHEDULE :
				if ( clickedX >= MinutesToSurface( minutesFrom)) and ( clickedX <= MinutesToSurface( minutesFrom) + 20) then
					result := FSchedules[ currSchedule];
			end;
		end;
		currSchedule := FSchedules[ currSchedule].nextIndex;
	end;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TServerPrivateGraphFrame.ResetSelection( Row, keySchedule : Integer);

	procedure InvalidateRow( Row : Integer);
	var
		rectRedraw : TRect;
	begin
		rectRedraw := gridGraph.CellRect( 0, Row);
		rectRedraw.Left := 0;
		rectRedraw.Right := gridGraph.Width;
		InvalidateRect( gridGraph.Handle, @rectRedraw, false);
	end;
var
	currSchedule : Integer;
	addressFrom, addressTo, client, summary : String;
begin
	Row := Max( Low( FCalendars), Min( Row, High( FCalendars)));

	if Row <> CursorRow then
	begin
		InvalidateRow( CursorRow);
		FCursorRow := Row;
		InvalidateRow( CursorRow);
	end;

	Selection.DisableControls;
	Selection.Close;
	Selection.Open;
	currSchedule := FCalendars[ Row].scheduleIndex;
	while currSchedule >= 0 do
	begin
		with Schedule, FSchedules[ currSchedule] do
		begin
			RecNo := recordNo;

			summary := Format( '%.2d:%.2d ` %s', [ minutesFrom div 60, minutesFrom mod 60, Fields[ sc_strService].AsString]);

			if ( Fields[ sc_typAddressFrom].AsInteger = PROFILE_ADDRESS)
			    and ( Fields[ sc_refResidentFrom].AsInteger = Fields[ sc_refRecipient].AsInteger) then
				addressFrom := 'p҂̎'
			else
				addressFrom := Fields[ sc_strAddressFrom].AsString;

			if ( Fields[ sc_typAddressTo].AsInteger = PROFILE_ADDRESS)
			    and ( Fields[ sc_refResidentTo].AsInteger = Fields[ sc_refRecipient].AsInteger) then
				addressTo := 'p҂̎'
			else
				addressTo := Fields[ sc_strAddressTo].AsString;

			if not Fields[ sc_refClient].IsNull then
				client := Format( ' ~ %s', [ Fields[ sc_strClient].AsString])
			else
				client := '';

			case Fields[ sc_typSchedule].AsInteger of
			CARE_SCHEDULE, MENAGE_SCHEDULE :
				summary := Format( '%s %s%s', [ summary, addressFrom, client]);

			MIGRATION_SCHEDULE, TRANSFER_SCHEDULE :
				if presence = BEFORE_PRESENCE then
					summary := Format( '%s y%s  %sz%s', [ summary, addressTo, addressFrom, client])
				else
					summary := Format( '%s y%s  %sz%s', [ summary, addressFrom, addressTo, client]);
			end;

			case Fields[ sc_typSchedule].AsInteger of
			CARE_SCHEDULE, MENAGE_SCHEDULE :
				summary := Format( '%s ` %.2d:%.2d', [ summary, minutesTo div 60, minutesTo mod 60]);

			MIGRATION_SCHEDULE :
				if presence = BEFORE_PRESENCE then
					summary := Format( '%s ` %.2d:%.2d', [ summary, subjectAt div 60, subjectAt mod 60]);
			end;

			Selection.Append;
			Selection.FieldByName( 'keyRow').AsInteger := Row;
			Selection.FieldByName( 'keyInstance').AsInteger := instanceKey;
			Selection.FieldByName( 'typSchedule').AsInteger := scheduleType;
			Selection.FieldByName( 'intMinutesAt').AsInteger := minutesFrom;
			Selection.FieldByName( 'intMinutesFor').AsInteger := minutesTo - minutesFrom;
			Selection.FieldByName( 'strCaption').AsString := summary;
			Selection.Post;
			currSchedule := nextIndex;
		end;
	end;
	Selection.IndexFieldNames := 'intMinutesAt;intMinutesFor;strCaption';
	Selection.EnableControls;

	cboSelection.KeyValue := ZeroToNull( 0);
	cboSelection.KeyValue := keySchedule;
	btnSelectSchedule.Enabled := cboSelection.Text <> '';
	btnDeleteSchedule.Enabled := cboSelection.Text <> '';
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TServerPrivateGraphFrame.Reset( keyProfile : Integer; datFrom, datTo : TDate);
var
	defaultRow, defaultSchedule : Integer;

	dayIndex, optAvailability : Integer;
	ARow, ACol : Integer;
	currCalendar, currSchedule, nextSchedule : Integer;
begin
	defaultRow := Selection.FieldByName( 'keyRow').AsInteger;
	defaultSchedule := Selection.FieldByName( 'keyInstance').AsInteger;
	// cursor preservation

	FServerKey := keyProfile;
	FScheduleFrom := datFrom;
	FScheduleTo := datTo;
	// property update

	Profile.Close;
	Profile.ParamByName( 'keyInstance').AsInteger := ServerKey;
	Profile.ParamByName( 'datAvailableAt').AsDateTime := ScheduleTo;
	Profile.Open;

	if not Profile.IsEmpty then with Profile do
		txtProfileInfo.Text := Format( '%s / %s / %s / %s / %s/ %s', [
			Trim( FieldByName( 'strBranch').AsString),
			Trim( FieldByName( 'strRegion').AsString),
			Trim( FieldByName( 'strCode').AsString),
			Trim( FieldByName( 'strName').AsString),
			Trim( FieldByName( 'strNameRead').AsString),
			Trim( FieldByName( 'strPhoneNumber1').AsString),
			Trim( FieldByName( 'strAddress').AsString)
		])
	else
		txtProfileInfo.Text := 'o^敪 / Zn / o^ԍ / o^@ / ݁@ / dbԍP/ Z';
	// Profile update

	FAvailability := 0;
	optAvailability := Profile.FieldByName( 'optAvailability').AsInteger;
	for dayIndex := 0 to 30 do
	begin
		if ( ( 1 shl ( DayOfWeek( IncDay( ScheduleFrom, dayIndex) - 1))) and optAvailability) <> 0 then
			FAvailability := FAvailability or ( 1 shl dayIndex);
	end;

	Absence.Close;
	Absence.ParamByName( 'keyProfile').AsInteger := ServerKey;
	Absence.ParamByName( 'datMonthFirstday').AsDateTime := StartOfTheMonth( ScheduleTo);
	Absence.Open;
	with Absence do while not Eof do
	begin
		FAvailability := FAvailability and ( not FieldByName( 'optAbsence').AsInteger);
		Next;
	end;
	// Absence update

	Schedule.Close;
	Schedule.Params.ParamByName( 'keySupplier').AsInteger := ServerKey;
	Schedule.Params.ParamByName( 'datFrom').AsDate := ScheduleFrom;
	Schedule.Params.ParamByName( 'datTo').AsDate := ScheduleTo;
	Schedule.Open;
	// Schedule update

	if sc_keyInstance < 0 then
	begin
		sc_keyInstance := Schedule.FieldByName( 'keyInstance').Index;
		sc_typProgress := Schedule.FieldByName( 'typProgress').Index;
		sc_datSchedule := Schedule.FieldByName( 'datSchedule').Index;
		sc_typSchedule := Schedule.FieldByName( 'typSchedule').Index;
		sc_strService := Schedule.FieldByName( 'strService').Index;
		sc_intMinutesFrom := Schedule.FieldByName( 'intMinutesFrom').Index;
		sc_intMinutesTo  := Schedule.FieldByName( 'intMinutesTo').Index;
		sc_refSubject := Schedule.FieldByName( 'refSubject').Index;
		sc_typSubject := Schedule.FieldByName( 'typSubject').Index;
		sc_intSubjectAt  := Schedule.FieldByName( 'intSubjectAt').Index;
		sc_intSubjectFor  := Schedule.FieldByName( 'intSubjectFor').Index;
		sc_typAddressFrom := Schedule.FieldByName( 'typAddressFrom').Index;
		sc_refAddressFrom := Schedule.FieldByName( 'refAddressFrom').Index;
		sc_strAddressFrom := Schedule.FieldByName( 'strAddressFrom').Index;
		sc_refResidentFrom := Schedule.FieldByName( 'refResidentFrom').Index;
		sc_typAddressTo := Schedule.FieldByName( 'typAddressTo').Index;
		sc_refAddressTo := Schedule.FieldByName( 'refAddressTo').Index;
		sc_strAddressTo := Schedule.FieldByName( 'strAddressTo').Index;
		sc_refResidentTo := Schedule.FieldByName( 'refResidentTo').Index;
		sc_refRecipient := Schedule.FieldByName( 'refRecipient').Index;
		sc_refClient := Schedule.FieldByName( 'refClient').Index;
		sc_strClient := Schedule.FieldByName( 'strClient').Index;
		sc_refServer := Schedule.FieldByName( 'refServer').Index;
		sc_strServer := Schedule.FieldByName( 'strServer').Index;
		sc_strOpponent := Schedule.FieldByName( 'strOpponent').Index;
	end;
	// prepare index cache

	FCalendars := nil;
	SetLength( FCalendars, DaysBetween( ScheduleFrom, ScheduleTo) + 1);
	for ARow := Low( FCalendars) to High( FCalendars) do
	begin
		FCalendars[ ARow].calendar := IncDay( ScheduleFrom, ARow);
		FCalendars[ ARow].overlapped := false;
		FCalendars[ ARow].scheduleIndex := -1;
	end;

	FSchedules := nil;
	nextSchedule := 0;
	SetLength( FSchedules, Schedule.RecordCount);

	Schedule.First;
	with Schedule do while not EOF do
	begin
		currCalendar := DaysBetween( ScheduleFrom, Fields[ sc_datSchedule].AsDateTime);
		if FCalendars[ currCalendar].scheduleIndex >= 0 then
		begin
			currSchedule := FCalendars[ currCalendar].scheduleIndex;
			while true do
			begin
				case Fields[ sc_typSchedule].AsInteger of CARE_SCHEDULE, MENAGE_SCHEDULE :
					if IntersectionOf(
						FSchedules[ currSchedule].minutesFrom, FSchedules[ currSchedule].minutesTo,
						Fields[ sc_intMinutesFrom].AsInteger, Fields[ sc_intMinutesTo].AsInteger
					) > 5 then
					begin
						FCalendars[ currCalendar].overlapped := true;
					end;
				else
					if ( FSchedules[ currSchedule].scheduleType = Fields[ sc_typSchedule].AsInteger)
					and ( FSchedules[ currSchedule].minutesFrom = Fields[ sc_intMinutesFrom].AsInteger)
					and ( FSchedules[ currSchedule].minutesTo = Fields[ sc_intMinutesTo].AsInteger)
					then
					begin
						FCalendars[ currCalendar].overlapped := true;
					end;
				end;
				if FSchedules[ currSchedule].nextIndex < 0 then break;
				currSchedule := FSchedules[ currSchedule].nextIndex;
			end;
			FSchedules[ currSchedule].nextIndex := nextSchedule;
		end
		else
		begin
			FCalendars[ currCalendar].scheduleIndex := nextSchedule;
		end;

		with FSchedules[ nextSchedule] do
		begin
			instanceKey := Fields[ sc_keyInstance].AsInteger;
			scheduleType := Fields[ sc_typSchedule].AsInteger;
			progress := Fields[ sc_typProgress].AsInteger;
			minutesFrom := Fields[ sc_intMinutesFrom].AsInteger;
			minutesTo := Fields[ sc_intMinutesTo].AsInteger;
			subjectAt := Fields[ sc_intSubjectAt].AsInteger;
			opponent := Fields[ sc_strOpponent].AsString;
			serverReady := true;
			subjectKey := Fields[ sc_refSubject].AsInteger;
			case scheduleType of
			MIGRATION_SCHEDULE :
				if minutesFrom < subjectAt + Fields[ sc_intSubjectFor].AsInteger div 2 then
					presence := BEFORE_PRESENCE
				else
					presence := AFTER_PRESENCE;

			TRANSFER_SCHEDULE :
				if ( subjectKey > 0) and ( Fields[ sc_typSubject].AsInteger <> TRANSFER_SCHEDULE) then
				begin
					if minutesFrom < subjectAt + Fields[ sc_intSubjectFor].AsInteger div 2 then
						presence := BEFORE_PRESENCE
					else
						presence := AFTER_PRESENCE;
				end
				else
					presence := FULL_PRESENCE;
			else
				presence := RAW_PRESENCE;
			end;
			recordNo := RecNo;
			nextIndex := -1;
		end;
		Inc( nextSchedule);

		Next;
	end;
	// Calendars and Packets setup

	gridGraph.RowCount := DaysBetween( ScheduleFrom, ScheduleTo) + 1;
	for ARow := 0 to gridGraph.RowCount - 1 do
		for ACol := 0 to gridGraph.ColCount - 1 do
			gridGraph.Objects[ ACol, ARow] := nil;

	if not Schedule.IsEmpty then
	begin
		for ARow := 0 to gridGraph.RowCount - 1 do
		begin
			currSchedule := FCalendars[ ARow].scheduleIndex;
			while currSchedule >= 0 do
                        begin
				with FSchedules[ currSchedule] do case scheduleType of CARE_SCHEDULE, MENAGE_SCHEDULE :
					for ACol := minutesFrom div 60 to ( minutesTo - 1) div 60 do
						gridGraph.Objects[ ACol, ARow] := Schedule;
				else
					for ACol := minutesFrom div 60 to ( minutesTo + 22) div 60 do
						gridGraph.Objects[ ACol, ARow] := Schedule;
				end;
				currSchedule := FSchedules[ currSchedule].nextIndex;
			end;
		end;
	end;
	// grid setup

	ResetSelection( defaultRow, defaultSchedule);
	// Selection update

	Obsolete := false;
end;

procedure TServerPrivateGraphFrame.Enclose( intMinutesFrom, intMinutesTo : Integer);
var
	leftBase : Integer;
	rectRedraw : TRect;
begin
        leftBase := - gridGraph.ColWidths[ 0] + gridGraph.LeftCol * ( gridGraph.DefaultColWidth + 1);

	rectRedraw.Top := 0;
	rectRedraw.Bottom := gridGraph.Height;
	rectRedraw.Left := MinutesToSurface( FEncloseFrom) - leftBase - 2;
	rectRedraw.Right := MinutesToSurface( FEncloseTo) - leftBase + 2;
	InvalidateRect( gridGraph.Handle, @rectRedraw, false);

	FEncloseFrom := intMinutesFrom;
	FEncloseTo := intMinutesTo;

	rectRedraw.Top := 0;
	rectRedraw.Bottom := gridGraph.Height;
	rectRedraw.Left := MinutesToSurface( FEncloseFrom) - leftBase - 2;
	rectRedraw.Right := MinutesToSurface( FEncloseTo) - leftBase + 2;
	InvalidateRect( gridGraph.Handle, @rectRedraw, false);
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TServerPrivateGraphFrame.btnSelectProfile_Click(Sender: TObject);
begin
	TServerSet.Prepare.Select( ServerKey);
end;

procedure TServerPrivateGraphFrame.btnChooseProfile_Click(Sender: TObject);
var
	keyServer : Integer;
begin
	if ChooseServer( DataStore.MainTransaction, ScheduleTo, keyServer) then
	begin
		Reset( keyServer, ScheduleFrom, ScheduleTo);
	end;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TServerPrivateGraphFrame.cboSelection_Click(Sender: TObject);
begin
	btnSelectSchedule.Enabled := cboSelection.Text <> '';
	btnDeleteSchedule.Enabled := cboSelection.Text <> '';
end;

procedure TServerPrivateGraphFrame.btnSelectSchedule_Click(Sender: TObject);
begin
	case Selection.FieldByName( 'typSchedule').AsInteger of
	     CARE_SCHEDULE : TCareSet.Prepare.Select( Selection.FieldByName( 'keyInstance').AsInteger);
	   MENAGE_SCHEDULE : TMenageSet.Prepare.Select( Selection.FieldByName( 'keyInstance').AsInteger);
	MIGRATION_SCHEDULE : TMigrationSet.Prepare.Select( Selection.FieldByName( 'keyInstance').AsInteger);
	 TRANSFER_SCHEDULE : TTransferSet.Prepare.Select( Selection.FieldByName( 'keyInstance').AsInteger);
	end;
end;

procedure TServerPrivateGraphFrame.btnDeleteSchedule_Click(Sender: TObject);
begin
	case Selection.FieldByName( 'typSchedule').AsInteger of
	     CARE_SCHEDULE : TCareSet.Prepare.Delete( Selection.FieldByName( 'keyInstance').AsInteger);
	   MENAGE_SCHEDULE : TMenageSet.Prepare.Delete( Selection.FieldByName( 'keyInstance').AsInteger);
	MIGRATION_SCHEDULE : TMigrationSet.Prepare.Delete( Selection.FieldByName( 'keyInstance').AsInteger);
	 TRANSFER_SCHEDULE : TTransferSet.Prepare.Delete( Selection.FieldByName( 'keyInstance').AsInteger);
	end;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TServerPrivateGraphFrame.gridGraph_DrawCell(Sender: TObject; ACol, ARow: Integer; Rect: TRect; State: TGridDrawState);
var
	defBrushColor, defPenColor : TColor;
	defBrushStyle : TBrushStyle;
	defPenStyle : TPenStyle;
	defClipRect : TRect;
        rectEnclosure : TRect;
        leftLimit, rightLimit : Integer;
	currSchedule : Integer;
begin
	with gridGraph.Canvas do
	begin
		defBrushColor := Brush.Color;
		defBrushStyle := Brush.Style;
		defPenColor := Pen.Color;
		defPenStyle := Pen.Style;
		defClipRect := ClipRect;
		IntersectClipRect( Handle, Rect.Left, Rect.Top, Rect.Right, Rect.Bottom);
	end;
	// backup default status

	with gridGraph.Canvas do if ACol = 0 then
	begin
		Font.Color := clBtnText;
		Font.Size := 8;
		if not FCalendars[ ARow].overlapped then
		begin
			if ARow = CursorRow then
			begin
				case DayOfTheWeek( DateFromRow( ARow)) of
				DaySaturday : Brush.Color := TColor( $CCAAAA);
				DaySunday : Brush.Color := TColor( $AAAACC);
				else Brush.Color := TColor( $AAAAAA);
				end;
			end
			else
			begin
				case DayOfTheWeek( DateFromRow( ARow)) of
				DaySaturday : Brush.Color := TColor( $DCC0C0);
				DaySunday : Brush.Color := TColor( $C0C0DC);
				else Brush.Color := clSilver;
				end;
			end;
		end
		else
		begin
			if ARow = CursorRow then
				Brush.Color := TColor( $9999CC)
			else
				Brush.Color := TColor( $AAAAEE);
		end;
		if FCalendars[ ACol].overlapped then Brush.Color := TColor( $9999EE);
		Brush.Style := bsSolid;
                Rect.Right := Rect.Left + 80;
		TextRect( Rect, Rect.Left + 2, Rect.Top + 2,
			FormatDateTime( 'M"/"dd"("ddd")"', DateFromRow( ARow))
		);
		// draw calendar information on header column

		if ( FAvailability and ( 1 shl ( DayOfTheMonth( FCalendars[ ARow].calendar) - 1))) = 0 then
		begin
			Font.Color := clBlack;
			Font.Size := 20;
			SetBkMode( Handle, TRANSPARENT);
			TextOut( Rect.Left + 53, Rect.Top + 10, '~');
		end;
		// draw available marker on header column
	end
	else
	begin
		if ( FAvailability and ( 1 shl ( DayOfTheMonth( FCalendars[ ARow].calendar) - 1))) <> 0 then
		begin
			if ARow = CursorRow then
				Brush.Color := TColor( $DDDDDD)
			else
				Brush.Color := gridGraph.Color;
			Brush.Style := bsSolid
		end
		else
		begin
			Brush.Color := clGray;
			Brush.Style := bsBDiagonal;
			if ARow = CursorRow then
				SetBkColor( Handle, TColor( $DDDDDD))
			else
				SetBkColor( Handle, gridGraph.Color);
		end;
		FillRect( Rect);

		if ( ( FEncloseFrom div 60) <= ACol) and ( ACol <= ( FEncloseTo div 60)) then
		begin
			if ARow = CursorRow then
				Brush.Color := TColor( $E0D0D0)
			else
				Brush.Color := TColor( $F0E0E0);
			rectEnclosure := Rect;
			rectEnclosure.Left := Rect.Left + MinutesToSurface( FEncloseFrom, ACol);
			rectEnclosure.Right := Rect.Left + MinutesToSurface( FEncloseTo, ACol);
			FillRect( rectEnclosure);
		end;

		Pen.Color := clLtGray;
		Pen.Style := psDot;
		MoveTo( Rect.Left, Rect.Top + 18);
		LineTo( Rect.Right, Rect.Top + 18);
		// draw schedule axis

		SetBkMode( Handle, TRANSPARENT);
		if ( ACol mod 4 = 0) or ( ARow mod 4 = 0) then
		begin
			Font.Color := clWhite;
			Font.Color := clGray;
			Font.Size := 8;
			TextOut( Rect.Left + 1, Rect.Top + 1,
				IfThen( ( ACol div 12) mod 2 = 0,
					Format( 'AM%d:00', [ ACol mod 12]),
					Format( 'PM%d:00', [ ACol mod 12])
				)
			);
		end;
		// draw time label

		Font.Color := clBlack;
		Font.Size := 8;
		if gridGraph.Objects[ ACol, ARow] <> nil then
		begin
			currSchedule := FCalendars[ ARow].scheduleIndex;
			while currSchedule >= 0 do
			begin
				with FSchedules[ currSchedule] do case scheduleType of
				CARE_SCHEDULE, MENAGE_SCHEDULE :
					begin
						Brush.Color := IfThen( scheduleType = CARE_SCHEDULE, clCare, clMenage);
						Brush.Style := bsSolid;
						if progress = UNCLEAR_PROGRESS then Brush.Style := bsDiagCross;

						Pen.Color := Brush.Color;
						Pen.Style := psSolid;

						leftLimit := Rect.Left + MinutesToSurface( minutesFrom, ACol);
						rightLimit := Rect.Left + MinutesToSurface( minutesTo, ACol);
						Rectangle( leftLimit, Rect.Top + 13, rightLimit, Rect.Top + 24);
					end;

				MIGRATION_SCHEDULE :
					begin
						leftLimit := Rect.Left + MinutesToSurface( minutesFrom, ACol);
						rightLimit := leftLimit + 22;
						iglIcon.Draw( gridGraph.Canvas, leftLimit, Rect.Top + 8, IfThen( progress <> UNCLEAR_PROGRESS, 4, 5));
					end;

				TRANSFER_SCHEDULE :
					if presence = FULL_PRESENCE then
					begin
						leftLimit := Rect.Left + MinutesToSurface( minutesFrom, ACol);
						rightLimit := leftLimit + 22;
						iglIcon.Draw( gridGraph.Canvas, leftLimit, Rect.Top + 8, IfThen( progress <> UNCLEAR_PROGRESS, 0, 1));
					end
					else
					begin
						leftLimit := Rect.Left + MinutesToSurface( minutesFrom, ACol);
						rightLimit := leftLimit + 22;
						iglIcon.Draw( gridGraph.Canvas, leftLimit, Rect.Top + 8, IfThen( progress <> UNCLEAR_PROGRESS, 2, 3));
					end;
				end;

				SetBkMode( Handle, TRANSPARENT);
				TextOut(
					( ( leftLimit + rightLimit) - TextWidth( FSchedules[ currSchedule].opponent)) div 2,
					Rect.Top + 26,
					FSchedules[ currSchedule].opponent
				);

				currSchedule := FSchedules[ currSchedule].nextIndex;
			end;
		end;
		// draw schedule

		if ( ACol mod 24) = HourOf( Now) then
		begin
			Pen.Color := clRed;
			Pen.Style := psSolid;
			MoveTo( Rect.Left + MinutesToSurface( MinuteOf( Now), 0), Rect.Top);
			LineTo( Rect.Left + MinutesToSurface( MinuteOf( Now), 0), Rect.Bottom);
		end;
		// draw current time indicator

		if ACol = ( FEncloseFrom div 60) then
		begin
			Pen.Color := clNavy;
			Pen.Style := psSolid;
			MoveTo( Rect.Left + MinutesToSurface( FEncloseFrom, ACol), Rect.Top);
			LineTo( Rect.Left + MinutesToSurface( FEncloseFrom, ACol), Rect.Bottom);
		end;

		if ACol = ( FEncloseTo div 60) then
		begin
			Pen.Color := clNavy;
			Pen.Style := psSolid;
			MoveTo( Rect.Left + MinutesToSurface( FEncloseTo, ACol), Rect.Top);
			LineTo( Rect.Left + MinutesToSurface( FEncloseTo, ACol), Rect.Bottom);
		end;
		// draw request time indicator
	end;

	with gridGraph.Canvas do
	begin
		Brush.Color := defBrushColor;
		Brush.Style := defBrushStyle;
		Pen.Color := defPenColor;
		Pen.Style := defPenStyle;
		SelectClipRgn( Handle, 0);
		IntersectClipRect( Handle, defClipRect.Left, defClipRect.Top, defClipRect.Right, defClipRect.Bottom);
	end;
	// restore default status
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TServerPrivateGraphFrame.gridGraph_SelectCell(Sender: TObject; ACol, ARow: Integer; var CanSelect: Boolean);
begin
	CanSelect := false;
end;

procedure TServerPrivateGraphFrame.gridGraph_MouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
	if Button = mbLeft then ResetSelection( RowFromY( Y), PacketFromPos( X, Y).instanceKey);
end;

procedure TServerPrivateGraphFrame.gridGraph_DblClick(Sender: TObject);
var
	clickedPos : TPoint;
	scheduleKey : Integer;
begin
	clickedPos := gridGraph.ScreenToClient( Mouse.CursorPos);

	with gridGraph do if ( TopRow + clickedPos.Y div ( DefaultRowHeight + 1)) >= RowCount then exit;

	scheduleKey := PacketFromPos( clickedPos.X, clickedPos.Y).instanceKey;
	if scheduleKey > 0 then
	begin
		cboSelection.KeyValue := ZeroToNull( 0);
		cboSelection.KeyValue := scheduleKey;
		btnSelectSchedule.Enabled := cboSelection.Text <> '';
		btnDeleteSchedule.Enabled := cboSelection.Text <> '';
		if btnSelectSchedule.Enabled then btnSelectSchedule.Click;
	end;
end;

procedure TServerPrivateGraphFrame.gridGraph_KeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
	case Key of
	 VK_LEFT : gridGraph.LeftCol := Max( 1, gridGraph.LeftCol - 1);
	VK_RIGHT : gridGraph.LeftCol := Min( gridGraph.ColCount - gridGraph.VisibleColCount, gridGraph.LeftCol + 1);
	   VK_UP : gridGraph.TopRow := Max( 0, gridGraph.TopRow - 1);
	 VK_DOWN : gridGraph.TopRow := Min( gridGraph.RowCount - gridGraph.VisibleRowCount, gridGraph.TopRow + 1);
	end;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TServerPrivateGraphFrame.gridGraph_DragOver(Sender, Source: TObject; X, Y: Integer; State: TDragState; var Accept: Boolean);
begin
	gridGraph_MouseDown( Sender, mbLeft, [], X,
		( DaysBetween( ScheduleFrom, DragSourceToJobPalette( Source).ScheduleAt) - gridGraph.TopRow)
			* ( gridGraph.DefaultRowHeight + 1)
				+ 10
	);

	Accept := not Profile.IsEmpty;
end;

procedure TServerPrivateGraphFrame.gridGraph_DragDrop(Sender, Source: TObject; X, Y: Integer);
begin
	with DragSourceToJobPalette( Source) do case JobType of
	     CARE_SCHEDULE : TCareSet.Arrange( ServerKey).Select( JobKey);
	   MENAGE_SCHEDULE : TMenageSet.Arrange( ServerKey).Select( JobKey);
	MIGRATION_SCHEDULE : TMigrationSet.Arrange( ServerKey).Select( JobKey);
	 TRANSFER_SCHEDULE : TTransferSet.Arrange( ServerKey).Select( JobKey);
	end;
end;

end.
