unit Misc_RouteBrowser;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, ExtCtrls, Math, Misc_Utilities;

type
  TRoutingPoint = record
    identifier : Integer;
    minutesFrom, minutesTo : Integer;
    fromX, fromY : Integer;
    toX, toY : Integer;
    primary : Boolean;
  end;

  TRouteBrowser = class(TForm)
    MapImage: TPaintBox;

    procedure Form_Create(Sender: TObject);
    procedure Form_Destroy(Sender: TObject);
    procedure Form_Show(Sender: TObject);

    procedure MapImage_DragOver(Sender, Source: TObject; X, Y: Integer; State: TDragState; var Accept: Boolean);
    procedure MapImage_Paint(Sender: TObject);

    procedure WMEraseBkgnd( var Message: TWMERASEBKGND);message WM_ERASEBKGND;
  private
    { Private 錾 }
    FMapPicture : TBitmap;

    FMinutesFrom : Integer;
    FMinutesTo : Integer;
    FFromX : Integer;
    FFromY : Integer;
    FToX : Integer;
    FToY : Integer;

    FRoutingPoints : array of TRoutingPoint;
    FNextPointer : Integer;
  public
    { Public 錾 }
    procedure StartHovering( intMinutesAt, intMinutesFor, intFromX, intFromY, intToX, intToY : Integer);
    procedure FinishHovering();

    procedure StartRefresh();
    procedure RegisterSchedule( intMinutesAt, intMinutesFor, intFromX, intFromY, intToX, intToY : Integer);
    procedure FinishRefresh();
  end;

var
  RouteBrowser: TRouteBrowser;

implementation

{$R *.dfm}

{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TRouteBrowser.StartHovering( intMinutesAt, intMinutesFor, intFromX, intFromY, intToX, intToY : Integer);
begin
	if FMapPicture.Empty then exit;

	FMinutesFrom := intMinutesAt;
	FMinutesTo := intMinutesAt + intMinutesFor;
	FFromX := intFromX;
	FFromY := intFromY;
	FToX := intToX;
	FToY := intToY;

	StartRefresh;
	Visible := true;
end;

procedure TRouteBrowser.FinishHovering();
begin
	Visible := false;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TRouteBrowser.StartRefresh();
begin
	SetLength( FRoutingPoints, 1);
	FNextPointer := 1;

	with FRoutingPoints[ 0] do
	begin
		identifier := 0;
		minutesFrom := FMinutesFrom;
		minutesTo := FMinutesTo;
		fromX := FFromX;
		fromY := FFromY;
		toX := FToX;
		toY := FToY;
		primary := true;
	end;
end;

procedure TRouteBrowser.RegisterSchedule( intMinutesAt, intMinutesFor, intFromX, intFromY, intToX, intToY : Integer);
begin
	SetLength( FRoutingPoints, FNextPointer + 1);

	with FRoutingPoints[ FNextPointer] do
	begin
		identifier := FNextPointer;
		minutesFrom := intMinutesAt;
		minutesTo := intMinutesAt + intMinutesFor;
		fromX := intFromX;
		fromY := intFromY;
		toX := intToX;
		toY := intToY;
		primary := false;
	end;

	Inc( FNextPointer);
end;

procedure TRouteBrowser.FinishRefresh();

	procedure SortPoint( leftMost, rightMost : Integer);
	var
		leftEdge, rightEdge : Integer;
		criterion, tempPoint : TRoutingPoint;

		function ComparePoint( const Point1, Point2 : TRoutingPoint) : Integer;
		begin
			result := Point1.minutesFrom - Point2.minutesFrom;
			if result <> 0 then exit;

			result := Point1.minutesTo - Point2.minutesTo;
			if result <> 0 then exit;

			result := Point1.identifier - Point2.identifier;
		end;
	begin
		if leftMost >= rightMost then exit;
		criterion := FRoutingPoints[ ( leftMost + rightMost) div 2];
		leftEdge := leftMost;
		rightEdge := rightMost;

		while true do
		begin
			while ( leftEdge < rightEdge)
				and ( ComparePoint( FRoutingPoints[ leftEdge], criterion) < 0)
					do Inc( leftEdge);

			while ( rightEdge > leftEdge)
				and ( ComparePoint( FRoutingPoints[ rightEdge], criterion) > 0)
					do Dec( rightEdge);

			if leftEdge = rightEdge then break;

			tempPoint := FRoutingPoints[ leftEdge];
			FRoutingPoints[ leftEdge] := FRoutingPoints[ rightEdge];
			FRoutingPoints[ rightEdge] := tempPoint;
		end;

		SortPoint( leftMost, leftEdge);
		SortPoint( leftEdge + 1, rightMost);
	end;
begin
	SortPoint( Low( FRoutingPoints), High( FRoutingPoints));
	MapImage.Invalidate;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TRouteBrowser.Form_Create(Sender: TObject);
var
	PictureName : String;
begin
	FMapPicture := TBitmap.Create;
	PictureName := ExtractFilePath( Application.ExeName) + 'RouteMap.bmp';
	if FileExists( PictureName) then FMapPicture.LoadFromFile( PictureName);
end;

procedure TRouteBrowser.Form_Destroy(Sender: TObject);
begin
	FMapPicture.Free;
end;

procedure TRouteBrowser.Form_Show(Sender: TObject);
begin
	ClientWidth := FMapPicture.Width;
	ClientHeight := FMapPicture.Height;
	Top := IfThen( Mouse.CursorPos.Y > ( Screen.Height div 2), 0, Screen.Height - Height);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TRouteBrowser.MapImage_DragOver(Sender, Source: TObject; X, Y: Integer; State: TDragState; var Accept: Boolean);
begin
	Top := IfThen( Mouse.CursorPos.Y > ( Screen.Height div 2), 0, Screen.Height - Height);
end;

procedure TRouteBrowser.MapImage_Paint(Sender: TObject);
var
	index, primaryIndex : Integer;
	markerSize : Integer;
	fromMarker, toMarker : String;
	fromLabel, atLabel, toLabel : String;
begin
	with MapImage.Canvas do
	begin
		Draw( 0, 0, FMapPicture);

		Pen.Color := clBlack;
		Pen.Width := 2;
		for index := 0 to High( FRoutingPoints) do with FRoutingPoints[ index] do
		begin
			if index = 0 then MoveTo( fromX, fromY) else LineTo( fromX, fromY);

			if primary then
			begin
				Font.Color := TColor( $0000C0);
				Font.Height := 16;
				Pen.Color := Font.Color;
				fromMarker := '~';
				toMarker := '';
				primaryIndex := index;
			end
			else
			begin
				Font.Color := clBlack;
				Font.Height := 12;
				Pen.Color := Font.Color;
				fromMarker := '~';
				toMarker := '~';
			end;
			markerSize := Font.Height div 2;

			LineTo( toX, toY);
			SetBkMode( Handle, TRANSPARENT);
			TextOut( fromX - markerSize, fromY - markerSize, fromMarker);
			TextOut( toX - markerSize, toY - markerSize, toMarker);
			MoveTo( toX, toY);
		end;

		Font.Color := TColor( $0000C0);
		Font.Height := 16;
		markerSize := Font.Height div 2;
		SetBkMode( Handle, OPAQUE);
		with FRoutingPoints[ primaryIndex] do
		begin
			if primaryIndex > 0 then
				fromLabel := Format( '(%d) ', [ minutesFrom - FRoutingPoints[ primaryIndex - 1].minutesTo])
			else
				fromLabel := '';

			if primaryIndex < High( FRoutingPoints) then
				toLabel := Format( ' (%d)', [ FRoutingPoints[ primaryIndex + 1].minutesFrom - minutesTo])
			else
				toLabel := '';

			if minutesFrom = minutesTo then
				atLabel := fromLabel + MinutesToText( minutesFrom) + toLabel
			else
				atLabel := fromLabel + PeriodsToText( minutesFrom, minutesTo) + toLabel;

			TextOut( ( fromX + toX) div 2 + markerSize, ( fromY + toY) div 2 + markerSize, atLabel);
		end;
	end;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TRouteBrowser.WMEraseBkgnd( var Message: TWMERASEBKGND);
begin
	Message.Result := 1;
end;

end.
