unit Misc_WinFax;

interface

uses Classes, SysUtils, StrUtils, DateUtils, Windows, Graphics, Printers;

procedure InitializeWinFax();
procedure CleanupWinFax();

function IsWinFaxAvailable() : Boolean;

function CreateFaxCanvas( DocName, RecpName, RecpNumber, SenderCompany : String; Orientation : TPrinterOrientation = poPortrait) : TCanvas;
procedure DeleteFaxCanvas( Fax : TCanvas);

implementation

type
  FAX_PRINT_INFO = packed record
    SizeOfStruct : DWORD;	// Size of this structure
    DocName : LPCSTR;		// Document name that appears in the spooler
    RecipientName : LPCSTR;	// Recipient name
    RecipientNumber : LPCSTR;	// Recipient fax number (non-canonical number)
    SenderName : LPCSTR;	// Sender name
    SenderCompany : LPCSTR;	// Sender company (optional)
    SenderDept : LPCSTR;	// Sender department
    SenderBillingCode : LPCSTR;	// Billing code
    DrProfileName : LPCSTR;	// Profile name for delivery report    <--| mutually exclusive
    DrEmailAddress : LPCSTR;	// E.Mail address for delivery report  <--|
    OutputFileName : LPCSTR;	// for print to file, resulting file name
  end;
  PFAX_PRINT_INFO = ^FAX_PRINT_INFO;

  FAX_CONTEXT_INFO = packed record
    SizeOfStruct : DWORD;	// Size of this structure
    hDC : HDC;			// Device Context
    ServerName : array[ 0..15] of CHAR;	// Server name
  end;
  PFAX_CONTEXT_INFO = ^FAX_CONTEXT_INFO;

var
  g_hWinFaxLib : HMODULE;
  FaxStartPrintJob : function ( PrinterName : LPCSTR; PrintInfo : PFAX_PRINT_INFO; FaxJobId : LPDWORD; FaxContextInfo : PFAX_CONTEXT_INFO) : BOOL; stdcall;

{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure InitializeWinFax();
begin
	g_hWinFaxLib := LoadLibrary( 'winfax.dll');
	FaxStartPrintJob := GetProcAddress( g_hWinFaxLib, 'FaxStartPrintJobA');
end;

procedure CleanupWinFax();
begin
	FreeLibrary( g_hWinFaxLib);
	FaxStartPrintJob := nil;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function IsWinFaxAvailable() : Boolean;
begin
	result := ( @FaxStartPrintJob <> nil);
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function CreateFaxCanvas( DocName, RecpName, RecpNumber, SenderCompany : String; Orientation : TPrinterOrientation) : TCanvas;
var
	PrintInfo : FAX_PRINT_INFO;
	ContextInfo : FAX_CONTEXT_INFO;
	FaxJobID : DWORD;
	Transform : XFORM;
begin
	if @FaxStartPrintJob = nil then
	begin
		result := nil;
		exit;
	end;

	PrintInfo.SizeOfStruct := SizeOf( PrintInfo);
	PrintInfo.DocName := PChar( DocName);
	PrintInfo.RecipientName := PChar( RecpName);
	PrintInfo.RecipientNumber := PChar( RecpNumber);
	PrintInfo.SenderName := '';
	PrintInfo.SenderCompany := PChar( SenderCompany);
	PrintInfo.SenderDept := '';
	PrintInfo.SenderBillingCode := '';
	PrintInfo.DrProfileName := nil;
	PrintInfo.DrEmailAddress := '';
	PrintInfo.OutputFileName := nil;

	ContextInfo.SizeOfStruct := SizeOf( ContextInfo);

	if FaxStartPrintJob( 'FAX', @PrintInfo, @FaxJobId, @ContextInfo) then
	begin
		result := TCanvas.Create;
		result.Handle := ContextInfo.hDC;

		if Orientation = poLandscape then
		begin
			SetGraphicsMode( result.Handle, GM_ADVANCED);
			with Transform do
			begin
				eM11 := 0;
				eM21 := -( GetDeviceCaps( result.Handle, PHYSICALWIDTH) / GetDeviceCaps( result.Handle, PHYSICALHEIGHT));
				eM12 := ( GetDeviceCaps( result.Handle, PHYSICALHEIGHT) / GetDeviceCaps( result.Handle, PHYSICALWIDTH));
				eM22 := 0;
				eDx := GetDeviceCaps( result.Handle, PHYSICALWIDTH) - GetDeviceCaps( result.Handle, PHYSICALOFFSETX);
				eDy := 0;
			end;
			SetWorldTransform( result.Handle, Transform);
		end;
	end
	else
	begin
		result := nil;
	end;
end;

procedure DeleteFaxCanvas( Fax : TCanvas);
begin
	DeleteDC( Fax.Handle);
	Fax.Free;
end;

end.
