unit Piece_AddressSelector;

interface

uses Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  DB, IBCustomDataSet, IBQuery, StdCtrls, Math, DateUtils, IBDataBase,
  Misc_Constants, Misc_Utilities, DBCtrls;

const
  ADDRESS_OF_CUSTOM = 0;
  ADDRESS_OF_CLIENT = 1;
  ADDRESS_OF_SERVER = 2;

type
  TAddressSelector = class(TFrame)
    Address: TIBQuery;
    DataSourceOf_Address: TDataSource;

    cboAddress: TDBLookupComboBox;
    btnBrowseAddress: TButton;
    btnChooseAddress: TButton;

    procedure Address_AfterOpen(DataSet: TDataSet);
    procedure Address_FilterRecord(DataSet: TDataSet; var Accept: Boolean);
    procedure cboAddress_Click(Sender: TObject);
    procedure btnBrowseAddress_Click(Sender: TObject);
    procedure btnChooseAddress_Click(Sender: TObject);
  private
    { Private 錾 }
    FReadOnly : Boolean;

    FOnChange : TNotifyEvent;

    function GetTransaction() : TIBTransaction;
    procedure SetTransaction( tran : TIBTransaction);
    function GetAvailableAt() : TDate;
    procedure SetAvailableAt( datAt : TDate);
    function GetAddressKey() : Integer;
    procedure SetAddressKey( keyAddress : Integer);
    function GetClientKey() : Integer;
    procedure SetClientKey( keyClient : Integer);
    function GetServerKey() : Integer;
    procedure SetServerKey( keyServer : Integer);
    function GetKeyValue() : Integer;
    procedure SetKeyValue( keySelected : Integer);
    function GetSelected() : Boolean;
    procedure SetReadOnly( AReadOnly : Boolean);
    function GetUnclear() : Boolean;
  public
    { Public 錾 }
    constructor Create(AOwner: TComponent); override;

    property Transaction : TIBTransaction read GetTransaction write SetTransaction;
    property AvailableAt : TDate read GetAvailableAt write SetAvailableAt;
    property AddressKey : Integer read GetAddressKey write SetAddressKey;
    property ClientKey : Integer read GetClientKey write SetClientKey;
    property ServerKey : Integer read GetServerKey write SetServerKey;
    property KeyValue : Integer read GetKeyValue write SetKeyValue;
    property Selected : Boolean read GetSelected;
    property ReadOnly : Boolean read FReadOnly write SetReadOnly;
    property Unclear : Boolean read GetUnclear;

    property OnChange : TNotifyEvent read FOnChange write FOnChange;

    procedure Reset( _tran : TIBTransaction; datAvailableAt : TDate; keyAddress, keyClient, keyServer, keySelected : Integer); overload;
    procedure Reset( _tran : TIBTransaction; datAvailableAt : TDate; keyAddress, keyClient, keyServer : Integer); overload;
    procedure Reset( keyClient, keyServer : Integer); overload;

    procedure Select( AddressType : Integer);

    procedure Choose();
  end;

implementation

{$R *.dfm}

uses Chooser_Address, Set_Address;

{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

constructor TAddressSelector.Create(AOwner: TComponent);
begin
	inherited Create( AOwner);

	FReadOnly := true;

	FOnChange := nil;

	Address.ParamByName( 'datAvailableAt').AsDate := Today;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TAddressSelector.GetTransaction() : TIBTransaction;
begin
	result := Address.Transaction;
end;

procedure TAddressSelector.SetTransaction( tran : TIBTransaction);
begin
	Reset( tran, AvailableAt, AddressKey, ClientKey, ServerKey, KeyValue);
end;

function TAddressSelector.GetAvailableAt() : TDate;
begin
	result := Address.ParamByName( 'datAvailableAt').AsDate;
end;

procedure TAddressSelector.SetAvailableAt( datAt : TDate);
begin
	Reset( Transaction, datAt, AddressKey, ClientKey, ServerKey, KeyValue);
end;

function TAddressSelector.GetAddressKey() : Integer;
begin
	result := Address.ParamByName( 'keyInstance').AsInteger;
end;

procedure TAddressSelector.SetAddressKey( keyAddress : Integer);
begin
	Reset( Transaction, AvailableAt, keyAddress, ClientKey, ServerKey, KeyValue);
end;

function TAddressSelector.GetClientKey() : Integer;
begin
	result := Address.ParamByName( 'keyClient').AsInteger;
end;

procedure TAddressSelector.SetClientKey( keyClient : Integer);
begin
	Reset( Transaction, AvailableAt, AddressKey, keyClient, ServerKey, KeyValue);
end;

function TAddressSelector.GetServerKey() : Integer;
begin
	result := Address.ParamByName( 'keyServer').AsInteger;
end;

procedure TAddressSelector.SetServerKey( keyServer : Integer);
begin
	Reset( Transaction, AvailableAt, AddressKey, ClientKey, keyServer, KeyValue);
end;

function TAddressSelector.GetKeyValue() : Integer;
begin
	result := VarToInt( cboAddress.KeyValue);
end;

procedure TAddressSelector.SetKeyValue( keySelected : Integer);
begin
	SelectComboWithoutFail( cboAddress, keySelected);
	cboAddress_Click( cboAddress);
end;

function TAddressSelector.GetSelected() : Boolean;
begin
	result := ( KeyValue > 0);
end;

procedure TAddressSelector.SetReadOnly( AReadOnly : Boolean);
begin
	FReadOnly := AReadOnly;

	cboAddress.Enabled := not ReadOnly;
	btnChooseAddress.Enabled := not ReadOnly;
end;

function TAddressSelector.GetUnclear() : Boolean;
begin
	result := ( KeyValue <= 0) or ( Address.FieldByName( 'strAddress').AsString = '');
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TAddressSelector.Reset( _tran : TIBTransaction; datAvailableAt : TDate; keyAddress, keyClient, keyServer, keySelected : Integer);
begin
	Address.Close;
	Address.Transaction := _tran;
	Address.ParamByName( 'datAvailableAt').AsDate := datAvailableAt;
	Address.ParamByName( 'keyInstance').AsInteger := keyAddress;
	Address.ParamByName( 'keyClient').AsInteger := keyClient;
	Address.ParamByName( 'keyServer').AsInteger := keyServer;
	Address.Open;

	Address.Filtered := false;
	with Address do
	begin
		First;
		Next;
		while not Eof do
		begin
			if FieldByName( 'keyInstance').AsInteger = keyAddress then
			begin
				Filtered := true;
				break;
			end;
			Next;
		end;
	end;

	Address.FetchAll;
	cboAddress.DropDownRows := Address.RecordCount - BoolToInt( Address.Filtered);

	KeyValue := keySelected;
end;

procedure TAddressSelector.Reset( _tran : TIBTransaction; datAvailableAt : TDate; keyAddress, keyClient, keyServer : Integer);
begin
	Reset( _tran, datAvailableAt, keyAddress, keyClient, keyServer, keyAddress);
end;

procedure TAddressSelector.Reset( keyClient, keyServer : Integer);
begin
	Reset( Transaction, AvailableAt, AddressKey, keyClient, keyServer, KeyValue);
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TAddressSelector.Select( AddressType : Integer);
begin
	if Address.Locate( 'typCategory', AddressType, []) then
	begin
		KeyValue := Address.FieldByName( 'keyInstance').AsInteger;
	end;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TAddressSelector.Choose();
begin
	if not ReadOnly then btnChooseAddress.Click;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TAddressSelector.Address_AfterOpen(DataSet: TDataSet);
begin
	SelectComboWithoutFail( cboAddress, KeyValue);
	cboAddress_Click( cboAddress);
end;

procedure TAddressSelector.Address_FilterRecord(DataSet: TDataSet; var Accept: Boolean);
begin
	Accept := DataSet.FieldByName( 'typCategory').AsInteger <> ADDRESS_OF_CUSTOM;
end;

procedure TAddressSelector.cboAddress_Click(Sender: TObject);
begin
	btnBrowseAddress.Enabled := KeyValue > 0;
	cboAddress.Color := IfThen( Unclear, clHazard, clWindow);
	cboAddress.Hint := Address.FieldByName( 'strName').AsString;
	if @FOnChange <> nil then FOnChange( self);
end;

procedure TAddressSelector.btnBrowseAddress_Click(Sender: TObject);
begin
	TAddressSet.Prepare.Select( KeyValue);
end;

procedure TAddressSelector.btnChooseAddress_Click(Sender: TObject);
var
	keyAddress : Integer;
begin
	if ChooseAddress( Transaction, AvailableAt, keyAddress) then
	begin
		AddressKey := keyAddress;
		KeyValue := keyAddress;
	end;
end;

end.
