unit Piece_JobPalette;

interface

uses Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Grids, Provider, IBSQL, IBDatabase, DB, DBClient, DBLocal, DBLocalI,
  DateUtils, Math, StrUtils, Misc_Constants, Misc_Utilities, DBCtrls, ExtCtrls,
  ImgList, StdCtrls;

type
  TJobPalette = class(TFrame)
    Job: TIBClientDataSet;
    iglCheckMark: TImageList;

    cboCondition: TComboBox;

    pnlLocation: TPanel;
    pnlAddressTo: TPanel;
    pnlAddressFrom: TPanel;
    shapeFrom: TShape;
    shapeTo: TShape;

    gridJob: TDrawGrid;

    procedure cboCondition_Click(Sender: TObject);

    procedure gridJob_DrawCell(Sender: TObject; ACol, ARow: Integer; Rect: TRect; State: TGridDrawState);
    procedure gridJob_MouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure gridJob_DblClick(Sender: TObject);
    procedure gridJob_SelectCell(Sender: TObject; ACol, ARow: Integer; var CanSelect: Boolean);

    procedure gridJob_StartDrag(Sender: TObject; var DragObject: TDragObject);
    procedure gridJob_EndDrag(Sender, Target: TObject; X, Y: Integer);

  private
    { Private 錾 }
    FScheduleAt : TDate;

    FOnJobSelected : TNotifyEvent;

    function GetJobKey() : Integer;
    function GetJobType() : Integer;
    function GetJobMinutesFrom() : Integer;
    function GetJobMinutesTo() : Integer;
    function GetJobMask() : Integer;
    function GetJobMaskName() : String;
  protected
    { Protected 錾 }
    procedure DrawHeader( Rect : TRect; clTitle : TColor; strTitle, strMinutes : String; State: TGridDrawState);
    procedure DrawFooter( Rect : TRect; DataSet : TDataSet; State: TGridDrawState);
    procedure DrawTransferJob( Rect : TRect; DataSet : TDataSet; State: TGridDrawState);
    procedure DrawCareJob( Rect : TRect; DataSet : TDataSet; State: TGridDrawState);
    procedure DrawMenageJob( Rect : TRect; DataSet : TDataSet; State: TGridDrawState);
    procedure DrawMigrationJob( Rect : TRect; DataSet : TDataSet; State: TGridDrawState);
  public
    { Public 錾 }
    property ScheduleAt : TDate read FScheduleAt;

    property JobKey : Integer read GetJobKey;
    property JobType : Integer read GetJobType;
    property JobMinutesFrom : Integer read GetJobMinutesFrom;
    property JobMinutesTo : Integer read GetJobMinutesTo;
    property JobMask : Integer read GetJobMask;
    property JobMaskName : String read GetJobMaskName;

    procedure Reset( newDate : TDate);

    property OnJobSelected : TNotifyEvent read FOnJobSelected write FOnJobSelected;
  end;

function DragSourceToJobPalette( Source : TObject) : TJobPalette;

implementation

{$R *.dfm}

uses Set_Transfer, Set_Care, Set_Menage, Set_Migration, Misc_RouteBrowser;

{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function DragSourceToJobPalette( Source : TObject) : TJobPalette;
begin
	result := ( Source as TComponent).Owner as TJobPalette;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TJobPalette.GetJobKey() : Integer;
begin
	if Job.IsEmpty then
		result := -1
	else
		result := Job.FieldByName( 'keyInstance').AsInteger;
end;

function TJobPalette.GetJobType() : Integer;
begin
	if Job.IsEmpty then
		result := -1
	else
		result := Job.FieldByName( 'typSchedule').AsInteger;
end;

function TJobPalette.GetJobMinutesFrom() : Integer;
begin
	if Job.IsEmpty then
		result := -1
	else
		result := Job.FieldByName( 'intMinutesFrom').AsInteger;
end;

function TJobPalette.GetJobMinutesTo() : Integer;
begin
	if Job.IsEmpty then
		result := -1
	else
		result := Job.FieldByName( 'intMinutesTo').AsInteger;
end;

function TJobPalette.GetJobMask() : Integer;
begin
	result := aServiceMask[ cboCondition.ItemIndex];
end;

function TJobPalette.GetJobMaskName() : String;
begin
	result := cboCondition.Text;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TJobPalette.DrawHeader( Rect : TRect; clTitle : TColor; strTitle, strMinutes : String; State: TGridDrawState);
var
	headerRect : TRect;
begin
	with gridJob.Canvas do
	begin
		Brush.Color := clTitle;
		Brush.Style := bsSolid;
		Font.Color := clWhite;
		Font.Size := 9;
		headerRect := Rect;
		headerRect.Bottom := headerRect.Top + 14;
		TextRect( headerRect, Rect.Left + 2, Rect.Top + 1, strTitle);
		TextOut( Rect.Right - 2 - TextWidth( strMinutes), Rect.Top + 1, strMinutes);

		if gdSelected in State then
			Brush.Color := clSilver
		else
			Brush.Color := TColor( $00F0F0F0);
		Brush.Style := bsSolid;
		Rect.Top := Rect.Top + 14;
		FillRect( Rect);
		Rect.Top := Rect.Top - 14;
	end;
end;

procedure TJobPalette.DrawFooter( Rect : TRect; DataSet : TDataSet; State: TGridDrawState);
var
	intScore : Integer;
	strScore : String;
	intLeft : Integer;
begin
	with gridJob.Canvas do
	begin
		if DataSet.FieldByName( 'strNotice').AsString <> '' then
		begin
			if gdSelected in State then
				iglCheckMark.Draw( gridJob.Canvas, Rect.Left + 2, Rect.Top + 78, 0)
			else
				iglCheckMark.Draw( gridJob.Canvas, Rect.Left + 2, Rect.Top + 78, 1);
			TextOut( Rect.Left + 18, Rect.Top + 80, DataSet.FieldByName( 'strNotice').AsString);
		end;

		intScore := Round( AmountOf(
			DataSet.FieldByName( 'typExchange').AsInteger,
			DataSet.FieldByName( 'curAmountPerCount').AsCurrency,
			DataSet.FieldByName( 'curAmountPerHour').AsCurrency,
			DataSet.FieldByName( 'intMinutesTo').AsInteger - DataSet.FieldByName( 'intMinutesFrom').AsInteger
		));

		if intScore > 0 then
		begin
			Font.Size := 12;
			strScore := Format( '%d', [ intScore]);
			intLeft := Rect.Right - 2 - TextWidth( strScore);
			SetBkMode( Handle, TRANSPARENT);
			TextOut( intLeft, Rect.Top + 16, strScore);
			TextOut( intLeft - 1, Rect.Top + 16, strScore);
			TextOut( intLeft, Rect.Top + 17, strScore);
			TextOut( intLeft - 1, Rect.Top + 17, strScore);
		end;
	end;
end;

procedure TJobPalette.DrawTransferJob( Rect : TRect; DataSet : TDataSet; State: TGridDrawState);
begin
	DrawHeader( Rect, clTransfer, 'ڑ', MinutesToText( DataSet.FieldByName( 'intMinutesAt').AsInteger), State);

	with gridJob.Canvas do
	begin
		Font.Color := clBlack;
		TextOut( Rect.Left + 2, Rect.Top + 16, DataSet.FieldByName( 'strService').AsString);
		TextOut( Rect.Left + 2, Rect.Top + 32, 'pҁF' + DataSet.FieldByName( 'strClient').AsString);
		TextOut( Rect.Left + 2, Rect.Top + 48, 'onF' + DataSet.FieldByName( 'strPlaceFrom').AsString);
		TextOut( Rect.Left + 2, Rect.Top + 64, 'ړInF' + DataSet.FieldByName( 'strPlaceTo').AsString);
	end;

	DrawFooter( Rect, DataSet, State);
end;

procedure TJobPalette.DrawCareJob( Rect : TRect; DataSet : TDataSet; State: TGridDrawState);
begin
	if DataSet.FieldByName( 'refSubject').IsNull then
		DrawHeader( Rect, clCare, '(P)', PeriodsToText( DataSet.FieldByName( 'intMinutesFrom').AsInteger, DataSet.FieldByName( 'intMinutesTo').AsInteger), State)
	else
		DrawHeader( Rect, clCare, '(Q)', PeriodsToText( DataSet.FieldByName( 'intMinutesFrom').AsInteger, DataSet.FieldByName( 'intMinutesTo').AsInteger), State);

	with gridJob.Canvas do
	begin
		Font.Color := clBlack;
		TextOut( Rect.Left + 2, Rect.Top + 16, DataSet.FieldByName( 'strService').AsString);
		TextOut( Rect.Left + 2, Rect.Top + 32, 'pҁF' + DataSet.FieldByName( 'strClient').AsString);
		TextOut( Rect.Left + 2, Rect.Top + 48, ' @F' + DataSet.FieldByName( 'strPlaceFrom').AsString);
	end;

	DrawFooter( Rect, DataSet, State);
end;

procedure TJobPalette.DrawMenageJob( Rect : TRect; DataSet : TDataSet; State: TGridDrawState);
begin
	if DataSet.FieldByName( 'refSubject').IsNull then
		DrawHeader( Rect, clMenage, 'Ǝ(P)', PeriodsToText( DataSet.FieldByName( 'intMinutesFrom').AsInteger, DataSet.FieldByName( 'intMinutesTo').AsInteger), State)
	else
		DrawHeader( Rect, clMenage, 'Ǝ(Q)', PeriodsToText( DataSet.FieldByName( 'intMinutesFrom').AsInteger, DataSet.FieldByName( 'intMinutesTo').AsInteger), State);

	with gridJob.Canvas do
	begin
		Font.Color := clBlack;
		TextOut( Rect.Left + 2, Rect.Top + 16, DataSet.FieldByName( 'strService').AsString);
		TextOut( Rect.Left + 2, Rect.Top + 32, 'pҁF' + DataSet.FieldByName( 'strClient').AsString);
		TextOut( Rect.Left + 2, Rect.Top + 48, ' @F' + DataSet.FieldByName( 'strPlaceFrom').AsString);
	end;

	DrawFooter( Rect, DataSet, State);
end;

procedure TJobPalette.DrawMigrationJob( Rect : TRect; DataSet : TDataSet; State: TGridDrawState);
begin
	DrawHeader( Rect, clMigration, 'wp[}', MinutesToText( DataSet.FieldByName( 'intMinutesAt').AsInteger), State);

	with gridJob.Canvas do
	begin
		Font.Color := clBlack;
		TextOut( Rect.Left + 2, Rect.Top + 16, DataSet.FieldByName( 'strService').AsString);
		TextOut( Rect.Left + 2, Rect.Top + 32, 'pҁF' + DataSet.FieldByName( 'strClient').AsString);
		TextOut( Rect.Left + 2, Rect.Top + 48, 'onF' + DataSet.FieldByName( 'strPlaceFrom').AsString);
		TextOut( Rect.Left + 2, Rect.Top + 64, 'ړInF' + DataSet.FieldByName( 'strPlaceTo').AsString);
	end;

	DrawFooter( Rect, DataSet, State);
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TJobPalette.Reset( newDate : TDate);
var
	keyInstance, defaultRecNo : Integer;
begin
	FScheduleAt := newDate;
	// property update

	if Job.IsEmpty then
	begin
		keyInstance := -1;
		defaultRecNo := -1;
	end
	else
	begin
		keyInstance := Job.FieldByName( 'keyInstance').AsInteger;
		defaultRecNo := Job.RecNo;
	end;

	Job.DisableControls;
	Job.Close;
	Job.Params.ParamByName( 'datSchedule').AsDate := FScheduleAt;
	Job.Params.ParamByName( 'optService').AsInteger := aServiceMask[ cboCondition.ItemIndex];
	Job.Open;
	Job.IndexFieldNames := 'intMinutesAt;typSchedule;strClientRead;keyInstance';

	gridJob.RowCount := Job.RecordCount;
	if not Job.IsEmpty then
	begin
		if not Job.Locate( 'keyInstance', keyInstance, []) then
			if defaultRecNo > 0 then
				Job.RecNo := Min( Job.RecordCount, defaultRecNo);
		gridJob.Row := Job.RecNo - 1;
		gridJob.RowHeights[ 0] := gridJob.DefaultRowHeight;

		pnlAddressFrom.Caption := '   ' + MidStr( Job.FieldByName( 'strAddressFrom').AsString, 4, 999);
		pnlAddressTo.Caption := MidStr( Job.FieldByName( 'strAddressTo').AsString, 4, 999) + '   ';
	end
	else
	begin
		gridJob.RowHeights[ 0] := 0;
		pnlAddressFrom.Caption := '';
		pnlAddressTo.Caption := '';
	end;
	Job.EnableControls;

	gridJob.Invalidate;

	if @FOnJobSelected <> nil then FOnJobSelected( self);
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TJobPalette.cboCondition_Click(Sender: TObject);
begin
	Reset( ScheduleAt);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TJobPalette.gridJob_DrawCell(Sender: TObject; ACol, ARow: Integer; Rect: TRect; State: TGridDrawState);
begin
	if Job.IsEmpty then exit;
	Job.RecNo := ARow + 1;
	case Job.FieldByName( 'typSchedule').AsInteger of
	     CARE_SCHEDULE : DrawCareJob( Rect, Job, State);
	   MENAGE_SCHEDULE : DrawMenageJob( Rect, Job, State);
	MIGRATION_SCHEDULE : DrawMigrationJob( Rect, Job, State);
	 TRANSFER_SCHEDULE : DrawTransferJob( Rect, Job, State);
	end;
end;

procedure TJobPalette.gridJob_MouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
	if Job.IsEmpty then exit;
	if Y mod ( gridJob.DefaultRowHeight + 1) < 14 then gridJob.BeginDrag( false, 4);
end;

procedure TJobPalette.gridJob_DblClick(Sender: TObject);
begin
	if Job.IsEmpty then exit;
	gridJob.EndDrag( false);
	if gridJob.ScreenToClient( Mouse.CursorPos).Y mod gridJob.DefaultRowHeight < 14 then exit;
	case Job.FieldByName( 'typSchedule').AsInteger of
	     CARE_SCHEDULE : TCareSet.Prepare.Select( Job.FieldByName( 'keyInstance').AsInteger);
	   MENAGE_SCHEDULE : TMenageSet.Prepare.Select( Job.FieldByName( 'keyInstance').AsInteger);
	MIGRATION_SCHEDULE : TMigrationSet.Prepare.Select( Job.FieldByName( 'keyInstance').AsInteger);
	 TRANSFER_SCHEDULE : TTransferSet.Prepare.Select( Job.FieldByName( 'keyInstance').AsInteger);
	end;
end;

procedure TJobPalette.gridJob_SelectCell(Sender: TObject; ACol, ARow: Integer; var CanSelect: Boolean);
begin
	if Job.IsEmpty then exit;
	if not Job.ControlsDisabled then Job.RecNo := ARow + 1;
	pnlAddressFrom.Caption := '   ' + MidStr( Job.FieldByName( 'strAddressFrom').AsString, 4, 999);
	pnlAddressTo.Caption := MidStr( Job.FieldByName( 'strAddressTo').AsString, 4, 999) + '   ';
	if @FOnJobSelected <> nil then FOnJobSelected( self);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TJobPalette.gridJob_StartDrag(Sender: TObject; var DragObject: TDragObject);
begin
	if @OnStartDrag <> nil then OnStartDrag( Sender, DragObject);
	RouteBrowser.StartHovering(
		Job.FieldByName( 'intMinutesAt').AsInteger,
		Job.FieldByName( 'intMinutesFor').AsInteger,
		Job.FieldByName( 'intFromX').AsInteger,
		Job.FieldByName( 'intFromY').AsInteger,
		Job.FieldByName( 'intToX').AsInteger,
		Job.FieldByName( 'intToY').AsInteger
	);
end;

procedure TJobPalette.gridJob_EndDrag(Sender, Target: TObject; X, Y: Integer);
begin
	RouteBrowser.FinishHovering;
	if @OnEndDrag <> nil then OnEndDrag( Sender, Target, X, Y);
end;

end.
