unit Set_Care;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, Base_Set, IBSQL, IBDatabase, StdCtrls, ExtCtrls, DB,
  IBCustomDataSet, IBQuery, ComCtrls, DBCtrls, Math, StrUtils, DateUtils,
  ActnList, Mask,  IBUpdateSQL, Misc_Constants, Misc_Utilities,
  Piece_ServiceSelector, Piece_MinutesSelector, Piece_AddressSelector,
  Piece_ProfileSelector, Piece_ClientSelector, Piece_ServerSelector,
  Piece_PatternSelector, MainDatastore, Misc_Scheduler;

type
  TCareSet = class(TSetTemplate)
    Schedule: TIBQuery;
    UpdaterOf_Schedule: TIBUpdateSQL;

    Subject: TIBQuery;

    Pattern: TPatternSelector;
    Service: TServiceSelector;

    MinutesFrame: TShape;
    pnlMinutes: TPanel;
    lblMinutes: TLabel;
    MinutesFrom: TMinutesSelector;
    lblMinutesOver: TLabel;
    MinutesTo: TMinutesSelector;

    AddressFrame: TShape;
    pnlAddress: TPanel;
    lblAddress: TLabel;
    AddressAt: TAddressSelector;

    ClientFrame: TShape;
    Client: TClientSelector;

    ServerFrame: TShape;
    Server: TServerSelector;

    btnQuoteFromPrevMonth: TButton;
    btnQuoteFromThisMonth: TButton;

    chkCopyNotice: TCheckBox;
    chkCopyServer: TCheckBox;
    btnBrowseSubject: TButton;

    ShortCuts: TActionList;
    Shortcut_ChooseAddressAt: TAction;
    Shortcut_ChooseServer: TAction;

    procedure MinutesFrom_Change(Sender: TObject);
    procedure MinutesTo_Change(Sender: TObject);
    procedure AddressAt_Change(Sender: TObject);
    procedure Client_Change(Sender: TObject);
    procedure Server_Change(Sender: TObject);

    procedure btnChooseServer_Click(Sender: TObject);

    procedure btnQuoteSchedule_Click(Sender: TObject);

    procedure btnBrowseSubject_Click(Sender: TObject);

    procedure ChooseAddressAt_Execute(Sender: TObject);
    procedure ChooseServer_Execute(Sender: TObject);

  private
    { Private 錾 }
    FSubjectKey : Integer;
    FSchedule : TDate;
    FMinutesAt : Integer;
    FMinutesFor : Integer;
    FClientKey : Integer;
    FServerKey : Integer;
  protected
    { Protected 錾 }
    procedure DefineTable( var nameOfTable : String; var modifyFlag, cancelFlag : Boolean); override;

    procedure BeforeInput(); override;
    procedure AfterInput(); override;

    procedure BeforeSelect(); override;
    procedure AfterSelect(); override;

    procedure BeforeAppend(); override;
    procedure AfterAppend(); override;

    procedure BeforePopup(); override;
    procedure AfterPopup(); override;
  public
    { Public 錾 }
    constructor Prepare(); overload; override;
    constructor Prepare( datSchedule : TDate; intMinutesAt, keyClient : Integer); overload;
    constructor Prepare( keySubject, intMinutesAt : Integer); overload;
    constructor Prepare( keySubject : Integer); overload;
    constructor Arrange( keyServer : Integer); overload;
  end;

var
  CareSet: TCareSet;

implementation

{$R *.dfm}

uses Chooser_Candidate, Chooser_Schedule;

{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

constructor TCareSet.Prepare();
begin
	inherited Prepare;

	FSubjectKey := -1;
	FSchedule := Today;
	FMinutesAt := 0;
	FMinutesFor := 0;
	FClientKey := -1;
	FServerKey := -1;
end;

constructor TCareSet.Prepare( datSchedule : TDate; intMinutesAt, keyClient : Integer);
begin
	inherited Prepare;

	FSubjectKey := -1;
	FSchedule := datSchedule;
	FMinutesAt := intMinutesAt;
	FMinutesFor := 0;
	FClientKey := keyClient;
	FServerKey := -1;
end;

constructor TCareSet.Prepare( keySubject, intMinutesAt : Integer);
begin
	inherited Prepare;

	FSubjectKey := keySubject;
	FSchedule := Today;
	FMinutesAt := intMinutesAt;
	FMinutesFor := 0;
	FClientKey := -1;
	FServerKey := -1;
end;

constructor TCareSet.Prepare( keySubject : Integer);
begin
	inherited Prepare;

	FSubjectKey := keySubject;
	FSchedule := Today;
	FMinutesAt := 9999;
	FMinutesFor := 0;
	FClientKey := -1;
	FServerKey := -1;
end;

constructor TCareSet.Arrange( keyServer : Integer);
begin
	inherited Prepare;

	FSubjectKey := -1;
	FSchedule := Today;
	FMinutesAt := 0;
	FMinutesFor := 0;
	FClientKey := -1;
	FServerKey := keyServer;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TCareSet.DefineTable( var nameOfTable : String; var modifyFlag, cancelFlag : Boolean);
begin
	nameOfTable := 'Schedule';
	modifyFlag := true;
	cancelFlag := true;

	MinutesFrom.OnChange := MinutesFrom_Change;
	MinutesTo.OnChange := MinutesTo_Change;
	AddressAt.OnChange := AddressAt_Change;
	Client.OnChange := Client_Change;
	Server.OnChange := Server_Change;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TCareSet.BeforeInput();
begin
	Schedule.ParamByName( 'keyInstance').AsInteger := PrimaryKey;
	Schedule.Open;
end;

procedure TCareSet.AfterInput();
var
	PatternKey : Integer;
begin
	if Appending and Pattern.Extendable then
		PatternKey := RealizedKey
	else
		PatternKey := Schedule.FieldByName( 'keyPattern').AsInteger;

	with Schedule do
	begin
		if Appending then Append else Edit;
		FieldByName( 'keyInstance').AsInteger := RealizedKey;
		FieldByName( 'datSchedule').AsDateTime := FSchedule;
		FieldByName( 'keyPattern').AsInteger := PatternKey;
		FieldByName( 'typPattern').AsInteger := Pattern.TypeCode;
		FieldByName( 'optPattern').AsInteger := Pattern.MaskCode;
		FieldByName( 'optElement').AsInteger := CheckBoxesToOptions( [ chkCopyServer, chkCopyNotice]);
		FieldByName( 'typProgress').AsInteger := Service.Progress;
		FieldByName( 'refSubject').AsVariant := ZeroToNull( FSubjectKey);
		FieldByName( 'refService').AsInteger := Service.ServiceKey;
		FieldByName( 'refExchange').AsInteger := Service.ExchangeKey;
		FieldByName( 'intMinutesAt').AsInteger := MinutesFrom.Value;
		FieldByName( 'intMinutesFor').AsInteger := MinutesTo.Value - MinutesFrom.Value;
		FieldByName( 'refAddressFrom').AsInteger := AddressAt.KeyValue;
		FieldByName( 'refAddressTo').AsInteger := AddressAt.KeyValue;
		FieldByName( 'strNotice').AsString := Service.Notice;
		FieldByName( 'refClient').AsVariant := ZeroToNull( Client.KeyValue);
		FieldByName( 'refServer').AsVariant := ZeroToNull( Server.KeyValue);
		Post;
		ApplyUpdates;
	end;

	if Appending and Pattern.Extendable then
	begin
		with Scheduler do
		begin
			Transaction := self.Transaction;

			datAnchor := FSchedule;

			keyPattern := PatternKey;
			typPattern := Pattern.TypeCode;
			optPattern := Pattern.MaskCode;
			optElement := CheckBoxesToOptions( [ chkCopyServer, chkCopyNotice]);

			typProgress := Service.Progress;
			refService := Service.ServiceKey;
			refExchange := Service.ExchangeKey;

			intMinutesAt := MinutesFrom.Value;
			intMinutesFor := MinutesTo.Value - MinutesFrom.Value;
			refAddressFrom := AddressAt.KeyValue;
			refAddressTo := AddressAt.KeyValue;
			strNotice := IfThen( chkCopyNotice.Checked, Service.Notice, '');

			refClient := Client.KeyValue;
			refServer := IfThen( chkCopyServer.Checked, Server.KeyValue, 0);
		end;

		if Subject.IsEmpty then
		begin
			Scheduler.CopyAsSubject;
		end
		else if ShowConfirmation( '\Ƃĕʂ܂H', 'ʉ\ł', MB_YESNO) = IDYES then
		begin
			Scheduler.CopyAsLemma( Subject.FieldByName( 'keyPattern').AsInteger);
		end;
	end;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TCareSet.BeforeSelect();
begin
	with Schedule do
	begin
		ParamByName( 'keyInstance').AsInteger := PrimaryKey;
		Open;

		FSubjectKey := FieldByName( 'refSubject').AsInteger;
		FSchedule := FieldByName( 'datSchedule').AsDateTime;
		FMinutesAt := FieldByName( 'intMinutesAt').AsInteger;
		FMinutesFor := FieldByName( 'intMinutesFor').AsInteger;
		FClientKey := FieldByName( 'refClient').AsInteger;
		if ( FServerKey <= 0) or Browsing then FServerKey := FieldByName( 'refServer').AsInteger;

		Subject.ParamByName( 'keyInstance').AsInteger := FSubjectKey;
		Subject.Open;
		// value setup

		Pattern.Reset( FSchedule, FieldByName( 'typPattern').AsInteger, FieldByName( 'optPattern').AsInteger);

		Service.Reset( Transaction, FSchedule, CARE_SERVICE,
			FieldByName( 'typProgress').AsInteger,
			FieldByName( 'refService').AsInteger,
			FieldByName( 'refExchange').AsInteger,
			FieldByName( 'strNotice').AsString
		);

		MinutesFrom.Value := FMinutesAt;
		MinutesTo.Value := FMinutesAt + FMinutesFor;

		AddressAt.Reset( Transaction, FSchedule, FieldByName( 'refAddressFrom').AsInteger, FClientKey, FServerKey);

		Client.Reset( Transaction, FSchedule, FClientKey);
		Server.Reset( Transaction, FSchedule, FServerKey);

		OptionsToCheckBoxes( FieldByName( 'optElement').AsInteger, [ chkCopyServer, chkCopyNotice]);
	end;
end;

procedure TCareSet.AfterSelect();
begin
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TCareSet.BeforeAppend();
var
	keyAddressAt : Integer;
begin
	with Subject do
	begin
		ParamByName( 'keyInstance').AsInteger := FSubjectKey;
		Open;

		if not IsEmpty then
		begin
			if FieldByName( 'typStatus').AsInteger <> DATA_ENABLED then
			begin
				ErrorMessages.Add( 'w肳ꂽ\݂͑܂B');
				exit;
			end;

			FSchedule := FieldByName( 'datSchedule').AsDateTime;
			FClientKey := FieldByName( 'refClient').AsInteger;

			Pattern.Reset( FSchedule, FieldByName( 'typPattern').AsInteger, FieldByName( 'optPattern').AsInteger);

			FMinutesAt := FieldByName( 'intMinutesAt').AsInteger;
			FMinutesFor := FieldByName( 'intMinutesFor').AsInteger;

			keyAddressAt := FieldByName( 'refAddressTo').AsInteger;
		end
		else
		begin
			Pattern.Reset( FSchedule, PATTERN_SINGLETON, 0);

			FMinutesFor := 60;

			keyAddressAt := -1;
		end;
	end;

	Service.Reset( Transaction, FSchedule, CARE_SERVICE, UNCLEAR_PROGRESS, -1, -1, '');

	MinutesFrom.Value := FMinutesAt;
	MinutesTo.Value := FMinutesAt + FMinutesFor;

	AddressAt.Reset( Transaction, FSchedule, keyAddressAt, FClientKey, FServerKey);

	Client.Reset( Transaction, FSchedule, FClientKey);
	Server.Reset( Transaction, FSchedule, FServerKey);

	Client.btnChooseProfile.Visible := Subject.IsEmpty;
end;

procedure TCareSet.AfterAppend();
begin
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TCareSet.BeforePopup();
begin
	Pattern.ReadOnly := ( not Appending) or ( not Subject.IsEmpty);
	Service.ReadOnly := Browsing;

	MinutesFrom.ReadOnly := Browsing;
	MinutesTo.ReadOnly := Browsing;

	AddressAt.ReadOnly := Browsing;

	Client.ReadOnly := Browsing;
	Server.ReadOnly := Browsing;

	btnQuoteFromPrevMonth.Enabled := not Browsing;
	btnQuoteFromThisMonth.Enabled := not Browsing;

	chkCopyServer.Enabled := not Browsing;
	chkCopyNotice.Enabled := not Browsing;

	btnBrowseSubject.Visible := not Subject.IsEmpty;

	if Appending then Service.SetFocus;
	if Selecting then MinutesFrom.SetFocus;
	// enable controls

	if Subject.IsEmpty then
		Caption := FormatDateTime( '"쁗"yyyy"N"MM""dd"("ddd")"', FSchedule)
	else
		Caption := FormatDateTime( '"̕⏕"yyyy"N"MM""dd"("ddd")"', FSchedule);

	if not Appending then Caption := Format( '%s(%d)', [ Caption, PrimaryKey]);
	// update caption
end;

procedure TCareSet.AfterPopup();
var
	Verifications : TStringList;
begin
	if Pattern.IsInvalid then
		ErrorMessages.Add( '\tɂȂĂ܂B');

	if Service.ServiceKey <= 0 then
		ErrorMessages.Add( 'T[rXIĂ܂B');

	if Service.ExchangeKey <= 0 then
		ErrorMessages.Add( 'ӗ炪IĂ܂B');

	if MinutesFrom.Value > MinutesTo.Value then
		ErrorMessages.Add( 'tɂȂĂ܂B');

	if MinutesFrom.Value = MinutesTo.Value then
		ErrorMessages.Add( '{Ԃ[ɂȂĂ܂B');

	if not AddressAt.Selected then
		ErrorMessages.Add( '{ꏊIĂ܂B');

	if not Client.Selected then
		ErrorMessages.Add( 'p҂IĂ܂B');

	if Client.KeyValue = Server.KeyValue then
		ErrorMessages.Add( 'wp[Ɨp҂ɓ̓o^҂IԂƂ͂ł܂B');

	if ErrorMessages.Count > 0 then exit;
	// error check

	Verifications := TStringList.Create;
	try
		if AddressAt.Unclear then Verifications.Add( '{ꏊ͂ɂȂĂ܂B');

		if Verifications.Count = 0 then
		begin
			if Service.Progress = UNCLEAR_PROGRESS then
			begin
				if ShowConfirmation( 'Kvȏ񂪑Sē͂܂B̗\m肵܂H', '\̊m', MB_YESNO) = IDYES then
				begin
					Service.Progress := DEFINED_PROGRESS;
				end;
			end;
		end
		else
		begin
			if Service.Progress <> UNCLEAR_PROGRESS then
			begin
				ShowNotify( Verifications.Text + 'L̂悤ȕs邽߁A̗\͊m/~܂B', '\̊m');
				Service.Progress := UNCLEAR_PROGRESS;
			end;
		end;
	finally
		Verifications.Free;
	end;
	// progress check
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TCareSet.MinutesFrom_Change(Sender: TObject);
begin
	MinutesTo.Value := MinutesFrom.Value + MinutesTo.Value - MinutesFrom.PrevValue;
	Service.Minutes := MinutesTo.Value - MinutesFrom.Value;
end;

procedure TCareSet.MinutesTo_Change(Sender: TObject);
begin
	Service.Minutes := MinutesTo.Value - MinutesFrom.Value;
end;

procedure TCareSet.AddressAt_Change(Sender: TObject);
begin
	if ( not Preparing) and btnOK.Enabled then btnOK.SetFocus;
end;

procedure TCareSet.Client_Change(Sender: TObject);
begin
	AddressAt.ClientKey := Client.KeyValue;
end;

procedure TCareSet.Server_Change(Sender: TObject);
begin
	AddressAt.ServerKey := Server.KeyValue;
	if ( not Preparing) and btnOK.Enabled then btnOK.SetFocus;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TCareSet.btnChooseServer_Click(Sender: TObject);
var
	keyServer : Integer;
begin
	if ChooseCandidate(
		Transaction, FSchedule,
		Pattern.TypeCode, Pattern.MaskCode,
		MinutesFrom.Value, MinutesTo.Value,
		keyServer
	) then
		Server.KeyValue := keyServer;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TCareSet.btnQuoteSchedule_Click(Sender: TObject);
var
	info : TScheduleInfo;
begin
	if Sender = btnQuoteFromThisMonth then
		QuoteSchedule( Transaction, FSchedule, FClientKey, CARE_SERVICE, info)
	else
		QuoteSchedulePrevMonth( Transaction, FSchedule, FClientKey, CARE_SERVICE, info);

	if info.refService > 0 then
	begin
		Service.Reset( info.refService, info.refExchange, info.strNotice);
	end;

	if info.intMinutesAt >= 0 then
	begin
		MinutesFrom.Value := info.intMinutesAt;
		MinutesTo.Value := info.intMinutesAt + info.intMinutesFor;
	end;

	if info.refAddressFrom > 0 then
	begin
		AddressAt.KeyValue := info.refAddressFrom;
	end;

	if info.refServer > 0 then
	begin
		Server.KeyValue := info.refServer;
	end;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TCareSet.btnBrowseSubject_Click(Sender: TObject);
begin
	TCareSet.Prepare.Browse( Subject.FieldByName( 'keyInstance').AsInteger);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TCareSet.ChooseAddressAt_Execute(Sender: TObject);
begin
	if not Browsing then AddressAt.Choose;
end;

procedure TCareSet.ChooseServer_Execute(Sender: TObject);
begin
	if not Browsing then Server.Choose;
end;

end.
