/* 
 * Copyright 2007 Tatooine Project <http://tatooine.sourceforge.jp/> 
 *  
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *  
 *     http://www.apache.org/licenses/LICENSE-2.0
 *  
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.tatooine.gtx;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * LoopRange.
 *
 * @author  Tooru Noda
 * @version 1.0 2007/04/11
 * @since   JDK5.0 Tiger
 */
public class LoopRange {
	
	/** n. */
	public static final Pattern PTN_N = Pattern.compile("^([1-9][0-9]*|0)$");
	
	/** a..b. */
	public static final Pattern PTN_S2E1 = 
			Pattern.compile("^([1-9][0-9]*|0)\\.\\.([1-9][0-9]*|0)$");
	
	/** a...b. */
	public static final Pattern PTN_S2E2 = 
			Pattern.compile("^([1-9][0-9]*|0)\\.\\.\\.([1-9][0-9]*|0)$");
	
	/** a, b. */
	public static final Pattern PTN_A_LEN = Pattern.compile("^([1-9][0-9]*|0)\\s*,\\s*([1-9][0-9]*|0)$");
	
	/** 開始値. */
	private int _begin = 0;
	
	/** 終了値. */
	private int _end = 0;
	
	/** 刻み値. */
	private int _step = 1;

	/**
	 * <code>LoopRange</code>を構築する.
	 * 
	 * @param begin	開始値
	 * @param end		終了値
	 */
	public LoopRange(int begin, int end) {
		_begin = begin;
		_end = end;
	}

	/**
	 * <code>LoopRange</code>を構築する.
	 * 
	 * @param begin	開始値
	 * @param end		終了値
	 * @param step	刻み値
	 */
	public LoopRange(int begin, int end, int step) {
		_begin = begin;
		_end = end;
		_step = step;
	}

	/**
	 * 開始値を返す.
	 * 
	 * @return 開始値
	 */
	public final int getBegin() {
		return _begin;
	}

	/**
	 * 開始値を設定する.
	 * 
	 * @param begin 開始値
	 */
	public final void setBegin(int begin) {
		_begin = begin;
	}

	/**
	 * 終了値を返す.
	 * 
	 * @return 終了値
	 */
	public final int getEnd() {
		return _end;
	}

	/**
	 * 終了値を設定する.
	 * 
	 * @param end 終了値
	 */
	public final void setEnd(int end) {
		_end = end;
	}

	/**
	 * 刻み値を設定する.
	 * 
	 * @param step 刻み値
	 */
	public final void setStep(int step) {
		_step = step;
	}
	
	/**
	 * 次のインデックス値を取得する.
	 * 
	 * @param index	現在のインデックス
	 * @return	次のインデックス
	 */
	public final int getNext(int index) {
		if (_begin > _end) {
			return index - _step;
		}
		return index + _step;
	}
	
	/**
	 * <code>LoopRange</code>オブジェクトを生成する.
	 * 
	 * @param value	ループの範囲を表す式
	 * @return <code>LoopRange</code>オブジェクト
	 * @throws LoopRangeSyntaxException	構文エラー時に送出される。
	 */
	public static LoopRange parse(String value) throws LoopRangeSyntaxException {
		
		// <ARG>   ::= ^([1-9][0-9]*|0)$
		// <RANGE> ::= <ARG>
		//           | <ARG> '..'  <ARG>
		//           | <ARG> '...' <ARG>
		//           | <ARG> ','   <ARG>
		
		/* 指定がない場合、初期値を返す */
		if (value == null || value.trim().length() == 0) {
			return new LoopRange(0, 0);
		}
		/* n */
		if (PTN_N.matcher(value).matches()) {
			return new LoopRange(0, Integer.parseInt(value));
		}
		/* a..b */
		else if (PTN_S2E1.matcher(value).matches()) {
			Matcher m = PTN_S2E1.matcher(value);
			if (m.find()) {
				return new LoopRange(
						Integer.parseInt(m.group(1)), 
						Integer.parseInt(m.group(2)) + 1);
			}
		}
		/* a...b */
		else if (PTN_S2E2.matcher(value).matches()) {
			Matcher m = PTN_S2E2.matcher(value);
			if (m.find()) {
				return new LoopRange(
						Integer.parseInt(m.group(1)), 
						Integer.parseInt(m.group(2)));
			}
		}
		/* a, n */
		else if (PTN_A_LEN.matcher(value).matches()) {
			Matcher m = PTN_A_LEN.matcher(value);
			if (m.find()) {
				int begin = Integer.parseInt(m.group(1));
				return new LoopRange(begin, 
						Integer.parseInt(m.group(2)) + begin - 1);
			}
		}
		throw new LoopRangeSyntaxException(value);
	}

}
