package jp.sf.tatooine.gtx.node;

import static jp.sf.tatooine.base.OftenUsedStrings.COMMA;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import jp.sf.tatooine.base.utils.NameUtils;
import jp.sf.tatooine.gtx.Filter;
import jp.sf.tatooine.gtx.GtxContext;
import jp.sf.tatooine.gtx.GtxSyntaxException;
import jp.sf.tatooine.gtx.filter.Abs;
import jp.sf.tatooine.gtx.filter.Append;
import jp.sf.tatooine.gtx.filter.Binary;
import jp.sf.tatooine.gtx.filter.Capitalize;
import jp.sf.tatooine.gtx.filter.Ceil;
import jp.sf.tatooine.gtx.filter.Center;
import jp.sf.tatooine.gtx.filter.Chomp;
import jp.sf.tatooine.gtx.filter.Chop;
import jp.sf.tatooine.gtx.filter.Classify;
import jp.sf.tatooine.gtx.filter.Default;
import jp.sf.tatooine.gtx.filter.Delete;
import jp.sf.tatooine.gtx.filter.Escape;
import jp.sf.tatooine.gtx.filter.Floor;
import jp.sf.tatooine.gtx.filter.Format;
import jp.sf.tatooine.gtx.filter.Hex;
import jp.sf.tatooine.gtx.filter.Left;
import jp.sf.tatooine.gtx.filter.Length;
import jp.sf.tatooine.gtx.filter.Lowercase;
import jp.sf.tatooine.gtx.filter.Lpad;
import jp.sf.tatooine.gtx.filter.Mid;
import jp.sf.tatooine.gtx.filter.Octal;
import jp.sf.tatooine.gtx.filter.RegexReplace;
import jp.sf.tatooine.gtx.filter.Replace;
import jp.sf.tatooine.gtx.filter.Right;
import jp.sf.tatooine.gtx.filter.Rpad;
import jp.sf.tatooine.gtx.filter.Substring;
import jp.sf.tatooine.gtx.filter.Swapcase;
import jp.sf.tatooine.gtx.filter.Trim;
import jp.sf.tatooine.gtx.filter.Truncate;
import jp.sf.tatooine.gtx.filter.Uncapitalize;
import jp.sf.tatooine.gtx.filter.Underscore;
import jp.sf.tatooine.gtx.filter.Uppercase;

import org.apache.commons.jexl.Expression;
import org.apache.commons.jexl.ExpressionFactory;
import org.apache.commons.jexl.JexlContext;
import org.apache.commons.jexl.JexlHelper;
import org.apache.commons.lang.StringUtils;

public final class ELUtils {
	
	/** EL書式:${el}. */
	private static final Pattern PTN_EL = Pattern.compile("\\$\\{(.*?)\\}");
	
	/** 変数名規約. */
	private static final String REGEXP_IDENTIFIER = "[a-zA-Z_$][a-zA-Z0-9_\\-#$]*";
	
	/** フィルタの区切り文字 */
	private static final String FILTER_SEPARATOR = "\\|";
	
	/** 関数規約. */
	private static final String REGEXP_FUNC = 
		String.format("(%s)\\s*(\\((.*)\\))?", REGEXP_IDENTIFIER);
	
	/** 関数書式：func_name(args). */
	private static final Pattern PTN_FUNC = Pattern.compile(REGEXP_FUNC);
	
	/** 関数名位置 */
	private static final int POS_IDENTIFIER = 1;
	
	/** パラメータリスト位置 */
	private static final int POS_PARAM_LIST = 3;
	
	/** フィルタマップ */
	private static final HashMap<String, Filter> _filterMap = new HashMap<String, Filter>();
	static {
		addFilter(new Abs());
		addFilter(new Append());
		addFilter(new Binary());
		addFilter(new Capitalize());
		addFilter(new Ceil());
		addFilter(new Center());
		addFilter(new Chomp());
		addFilter(new Chop());
		addFilter(new Classify());
		addFilter(new Default());
		addFilter(new Delete());
		addFilter(new Escape());
		addFilter(new Floor());
		addFilter(new Format());
		addFilter(new Hex());
		addFilter(new Left());
		addFilter(new Length());
		addFilter(new Lowercase());
		addFilter(new Lpad());
		addFilter(new Mid());
		addFilter(new Octal());
		addFilter(new RegexReplace());
		addFilter(new Replace());
		addFilter(new Right());
		addFilter(new Rpad());
		addFilter(new Substring());
		addFilter(new Swapcase());
		addFilter(new Trim());
		addFilter(new Truncate());
		addFilter(new Uncapitalize());
		addFilter(new Underscore());
		addFilter(new Uppercase());
	}

	private ELUtils() { }
	
	/**
	 * カスタムフィルタをパーサに登録する.
	 * 
	 * @param filter	カスタムフィルタ
	 */
	public static final void addFilter(Filter filter) {
		_filterMap.put(NameUtils.underscore(filter.getClass().getSimpleName()), filter);
	}
	
	public static final List<GtxEvaluateable> parse(String value) {
		return null;
	}
	
	/* EL式を評価する */
	public static final String evaluateEL(GtxContext context, String text) throws GtxSyntaxException {
		
		StringBuffer buf = new StringBuffer();
		try {
			Matcher m = PTN_EL.matcher(text);
			while (m.find()) {
				String el = m.group(1);
				String[] array = el.split(FILTER_SEPARATOR);
				String result = null;
				if (array.length != 0) {
					Expression e = ExpressionFactory.createExpression(array[0].trim());
					JexlContext jexlContext = JexlHelper.createContext();
					jexlContext.setVars(context.flatten());
					Object o = e.evaluate(jexlContext);
					if (o == null) {
						result = "";
					}
					else {
						result = o.toString();
						for (int i = 1; i < array.length; i++) {
							String filter = array[i].trim();
							if (StringUtils.isEmpty(filter)) {
								throw new GtxSyntaxException(text);
							}
							result = evaluateFilter(result, filter);
						}
						Map map = jexlContext.getVars();
						for (Object entry : map.entrySet()) {
							Map.Entry tmp = (Map.Entry) entry;
							context.put((String) tmp.getKey(), tmp.getValue());
						}
					}
				}
				m.appendReplacement(buf, result);
			}
			m.appendTail(buf);
		}
		catch (Exception e) {
			throw new GtxSyntaxException(e);
		}
		return buf.toString();
	}
	
	/* Filterを評価する。 */
	private static final String evaluateFilter(String value, String filter) throws GtxSyntaxException {
		
		//【フィルタ文法規約】
		// method::= identifier [ "(" [ parameter_list ] ")" ]
		// identifier ::= [a-zA-Z_$][a-zA-Z0-9_\-#$]*
		// parameter_list ::= parameter { "," parameter }
		// parameter ::= .+
		
		try {
			Matcher m = PTN_FUNC.matcher(filter);
			if (!m.matches()) {
				throw new GtxSyntaxException(filter);
			}
			String id = m.group(POS_IDENTIFIER);
			String[] params = new String[0];
			if (m.group(POS_PARAM_LIST) != null) {
				params = m.group(POS_PARAM_LIST).split(COMMA);
				for (int i = 0; i < params.length; i++) {
					params[i] = params[i].trim();
				}
			}
			if (!_filterMap.containsKey(id)) {
				throw new GtxSyntaxException("Unknown Filter name. : " + id);
			}
			Filter filterObj = _filterMap.get(id);
			return filterObj.invoke(value, params);
		}
		catch (Exception e) {
			throw new GtxSyntaxException(e);
		}
	}
}
