#ifndef _TMSAFETY_MATRIX_HPP
#define _TMSAFETY_MATRIX_HPP
/** 
	@file safety_matrix.hpp
	@brief Defines tempest::safety_matrix<T,RowSz,ColumnSz,Al>.
	@author ototoi / Toru Matsuoka
	@date 2004/05/17 
*/

#include"safety_vector.hpp"
#include"matrix_base.hpp"

namespace tempest{

/** 
 *	@class safety_matrix
 *	@brief Safety_matrix is more safe than matrix. 
 *	@todo 
 *	@code
 *		tempest::safety_matrix<double,3,4> m;
 *	@endcode	
 *	@bug	
 *	@attention This class is slow.
 */
template<class,std::size_t,std::size_t,class>class safety_matrix;
	
template<class T, std::size_t RowSz, std::size_t ColumnSz = RowSz, class Al = std::allocator<T> >
class safety_matrix:public matrix_base< safety_matrix<T,RowSz,ColumnSz,Al>, T, RowSz, ColumnSz > {
	
public:
	//-----------------------------------------------
	//type defines
	typedef T	 										value_type;
	typedef safety_matrix<T,RowSz,ColumnSz,Al>			this_type;

	typedef safety_vector<T,RowSz * ColumnSz,Al>					array_type;
	
	typedef typename array_type::allocator_type			Allocator;
	typedef Allocator									allocator_type;
	
	
	typedef typename array_type::pointer 				pointer;
	typedef typename array_type::const_pointer 			const_pointer;
	typedef typename array_type::reference				reference;
	typedef typename array_type::const_reference		const_reference;
	typedef typename array_type::size_type				size_type;
	typedef typename array_type::difference_type		difference_type;
	
	typedef typename array_type::iterator				iterator;		//std::iterator<std::random_access_iterator_tag,T,difference_type>
	typedef typename array_type::const_iterator			const_iterator;	//std::iterator<std::random_access_iterator_tag,const T,difference_type>
	typedef typename array_type::reverse_iterator		reverse_iterator;
	typedef typename array_type::const_reverse_iterator	const_reverse_iterator;
	
public:
	static const size_type row_size = RowSz;
	static const size_type col_size = ColumnSz;
	static const size_type c_size   = RowSz * ColumnSz;
	
public:	
	//-----------------------------------------------
	//functions for iterator
	iterator		begin()			{return _m.begin();}
	iterator		end()			{return _m.end();}	
	const_iterator	begin()	const	{return _m.begin();}
	const_iterator	end()	const	{return _m.end();}
	
	reverse_iterator		rbegin()		{return _m.rbegin();}
	reverse_iterator		rend() 			{return _m.rend();}
	const_reverse_iterator	rbegin()const	{return _m.rbegin();}
	const_reverse_iterator	rend() 	const 	{return _m.rend();}
	
	void swap(this_type & rhs){
		_m.swap(rhs._m);
	}
	
	
	//constructor
	//
	explicit safety_matrix(const Allocator& al = Allocator() ):_m(al){}
	safety_matrix(const this_type& rhs):_m(rhs._m){}
	safety_matrix(const this_type& rhs, const Allocator& al):_m(rhs._m,al){}
	
	template<class IT>
	safety_matrix(IT first,IT last,const Allocator& al = Allocator()):_m(first,last,al){}
	
	explicit safety_matrix(T const (&rhs)[c_size], const Allocator& al = Allocator()):_m(rhs,al){}
	
	template<class Self,class Type>
	explicit safety_matrix(const matrix_base<Self,Type,row_size,col_size> & rhs, const Allocator& al = Allocator()):_m(static_cast<const Self &>(rhs).begin(),static_cast<const Self &>(rhs).end(),al){}
	
	//-----------------------------------------------
	//inserters
	this_type& operator= (const this_type &rhs){
		_m.swap( array_type(rhs._m) );
		return *this;
	}
	
	template<class Self,class Type>
	this_type& operator= (const matrix_base<Self,Type,row_size,col_size> & rhs){
		std::copy(static_cast<const Self &>(rhs).begin(),static_cast<const Self &>(rhs).end(),begin());
		return *this;
	}
	
	//-----------------------------------------------
	//capacity
	size_type size ()	  const { return c_size; }
	size_type max_size () const { return c_size; }
	bool	  empty ()	  const { return false; }
	
	//operators
	//-----------------------------------------------
	template<class AX>
	this_type& operator+=(const safety_matrix<T,RowSz,ColumnSz,AX> &rhs){
		this_type temp(*this);
		typename safety_matrix<T,RowSz,ColumnSz,AX>::const_iterator ri = rhs.begin();
		iterator ii = temp.begin();
		iterator ei = temp.end();
		while(ii != ei){
			*ii += *ri;
			++ii;++ri;
		}
		swap(temp);
		return *this;
	}	
	
	template<class AX>
	this_type& operator-=(const safety_matrix<T,RowSz,ColumnSz,AX> &rhs){
		this_type temp(*this);
		typename safety_matrix<T,RowSz,ColumnSz,AX>::const_iterator ri = rhs.begin();
		iterator ii = temp.begin();
		iterator ei = temp.end();
		while(ii != ei){
			*ii -= *ri;
			++ii;++ri;
		}
		swap(temp);
		return *this;
	}	
	
	//This functon is NOT recommended.
	template<class AX>
	this_type& operator*= (const safety_matrix<T,ColumnSz,ColumnSz,AX> &rhs){
		this_type temp(*this);
		T sum;
		for(size_type i =0;i<RowSz;++i){
			for(size_type j =0;j<ColumnSz;++j){				
				sum = T();
				for(size_type k=0;k<ColumnSz;k++){
					sum += (*this).at(i,k)*rhs.at(k,j);	
				}
				temp.at(i,j) = sum;
			}
		}
		swap(temp);
		return *this;
	}
	
	template<class X>
	this_type& operator*= (const X& rhs){
		this_type temp(*this);
		iterator ii = temp.begin();
		iterator ei = temp.end();
		while(ii != ei){
			*ii *= rhs;
			++ii;
		}
		swap(temp);
		return *this;	
	}
	template<class X>
	this_type& operator/= (const X& rhs){
		this_type temp(*this);
		iterator ii = temp.begin();
		iterator ei = temp.end();
		while(ii != ei){
			*ii /= rhs;
			++ii;
		}
		swap(temp);
		return *this;
	}
	
	//-------------------------------------------
	//indexer
	T * operator[](size_type i){
		return &(_m[i*RowSz]);	
	}
	const T * operator[](size_type i)const{
		return &(_m[i*RowSz]);	
	}
	
	T* at(size_type i){
		size_type idx = RowSz*i;
		if(c_size<=idx){throw std::out_of_range("tempest::safety_matrix");}
    	return &(_m[idx]);	
	}	
	const T* at(size_type i)const{
		size_type idx = RowSz*i;
		if(c_size<=idx){throw std::out_of_range("tempest::safety_matrix");}
    	return &(_m[idx]);	
	}
	
	T& at(size_type i,size_type j){
		size_type idx = RowSz*i+j;
		if(c_size<=idx){throw std::out_of_range("tempest::safety_matrix");}
    	return _m[idx];
	}
	const T& at(size_type i,size_type j)const{
		size_type idx = RowSz*i+j;
		if(c_size<=idx){throw std::out_of_range("tempest::safety_matrix");}
    	return _m[idx];
	}
	//
	
	const char* debug()const{return "tempest::safety_matrix<T,RowSz,ColumnSz,Al>"; }	

	array_type&       get_array()     {return _m;}
	const array_type& get_array()const{return _m;}
private:
	safety_vector<T,c_size,Al> _m;//member;

};//End of class.

	
//-----------------------------------------------
//binary operators
template<class T,std::size_t Sz1,std::size_t Sz2,class Al>
inline safety_matrix<T,Sz1,Sz2,Al> operator- (const safety_matrix<T,Sz1,Sz2,Al> & rhs){
	return safety_matrix<T,Sz1,Sz2,Al>(rhs) *= -1;
}

#define DACLARE_BINARY_OPERATOR(OP)																\
template<class T,std::size_t Sz1,std::size_t Sz2,class Al>										\
inline safety_matrix<T,Sz1,Sz2,Al>																\
operator OP (const safety_matrix<T,Sz1,Sz2,Al> &lhs, const safety_matrix<T,Sz1,Sz2,Al> &rhs){	\
	return safety_matrix<T,Sz1,Sz2,Al>(lhs) OP ## = rhs;												\
}

	DACLARE_BINARY_OPERATOR(+)	
	DACLARE_BINARY_OPERATOR(-)	
	
#undef DACLARE_BINARY_OPERATOR	
		
	
	
template<class T, std::size_t RowSz, std::size_t ColumnSz,class Al>
inline bool operator==(const safety_matrix<T,RowSz,ColumnSz,Al>& lhs, const safety_matrix<T,RowSz,ColumnSz,Al>& rhs){
	return lhs.get_array() == rhs.get_array();		
}
	
template<class T, std::size_t RowSz, std::size_t ColumnSz,class Al>
inline bool operator!=(const safety_matrix<T,RowSz,ColumnSz,Al>& lhs, const safety_matrix<T,RowSz,ColumnSz,Al>& rhs){
	return !(lhs==rhs);		
}
	
template<typename T, std::size_t RowSz, std::size_t ColumnSz, class Al, typename _CharT, class _Traits>
std::basic_ostream<_CharT, _Traits>& operator<<(std::basic_ostream<_CharT, _Traits>& os,  const safety_matrix<T,RowSz,ColumnSz,Al>& rhs){
	
	std::basic_ostringstream<_CharT, _Traits> s;
	s.flags(os.flags());
	s.imbue(os.getloc());
	s.precision(os.precision());
	for(std::size_t i=0;i<RowSz;i++){
		s <<"(";
		for(std::size_t j = 0;j < ColumnSz-1;++j){
			s << rhs[i][j] <<",";
		}
		s <<rhs[i][ColumnSz-1];
		s <<")";
		if(i != RowSz-1){s << "\n";}
	}
	return os << s.str();
}
	
}//End of namespace.
#endif


