#ifndef _TMTRANSFORM_HPP
#define _TMTRANSFORM_HPP
/** 
	@file transform.hpp
	@brief Defines functions for vector VS. matrix.
	@author ototoi / Toru Matsuoka
	@date 2004/05/13 
*/

#include<cstddef>

#include"vector_base.hpp"
#include"matrix_base.hpp"


namespace tempest{
	
template<class T, std::size_t Sz>class vector;							//pre
template<class T, std::size_t RowSz, std::size_t ColumnSz>class matrix;	//pre
template<class T>class quaternion;										//pre


// Vout = v * m;
template<class T, std::size_t RowSz, std::size_t ColumnSz>
inline vector<T,ColumnSz> operator* (const vector<T,RowSz> & lhs,const matrix<T,RowSz,ColumnSz> &rhs){
	vector<T,ColumnSz> temp;
	
	for(std::size_t i =0;i<ColumnSz;++i){			
		T sum = T();
		for(std::size_t j=0;j<RowSz;j++){
			sum += lhs[j]*rhs[j][i];	
		}
		temp[i] = sum;		
	}
	return temp;
}
// Vout = m * v;
template<class T, std::size_t RowSz, std::size_t ColumnSz>
inline vector<T,RowSz> operator* (const matrix<T,RowSz,ColumnSz> & lhs, const vector<T,ColumnSz> &rhs){
	vector<T,RowSz> temp;
	for(std::size_t i =0;i<RowSz;++i){			
		T sum = T();
		for(std::size_t j=0;j<ColumnSz;j++){
			sum += lhs[i][j]*rhs[j];	
		}
		temp[i] = sum;		
	}
	return temp;	
}
// v *= m;
//(v = v * m)
//template<class T>

// transform(&out,m,v);
//(v = m * v)
template<class T, std::size_t RowSz, std::size_t ColumnSz>
inline void transform(
	vector<T,RowSz> *out,
	const matrix<T,RowSz,ColumnSz> &lhs,
	const vector<T,ColumnSz> &rhs
){
	for(std::size_t i =0;i<RowSz;++i){			
		T sum = T();
		for(std::size_t j=0;j<ColumnSz;j++){
			sum += lhs[i][j]*rhs[j];	
		}
		(*out)[i] = sum;		
	}
	return;
}
	

	
	
//-----------------------------------------------
//vector<T,2> VS matrix<T,2>

// Vout = v * m;
template<class T>
inline vector<T,2> operator* (const vector<T,2> & lhs,const matrix<T,2> &rhs){
	return vector<T,2>( lhs[0]*rhs[0][0] + lhs[1]*rhs[1][0],lhs[0]*rhs[0][1] + lhs[1]*rhs[1][1] );
}
// Vout = m * v;
template<class T>
inline vector<T,2> operator* (const matrix<T,2> & lhs, const vector<T,2> &rhs){
	return vector<T,2>(lhs[0][0]*rhs[0] + lhs[0][1]*rhs[1],lhs[1][0]*rhs[0] + lhs[1][1]*rhs[1] );
}
// v *= m;
//(v = v * m)

// transform(m,v);
//(v = m * v)
template<class T>
inline void transform(
	vector<T,2> *out,
	const matrix<T,2> &lhs,
	const vector<T,2> &rhs
){
	(*out)[0] = lhs[0][0]*rhs[0] + lhs[0][1]*rhs[1];
    (*out)[1] = lhs[1][0]*rhs[0] + lhs[1][1]*rhs[1];
	return;
}

//-----------------------------------------------

//-----------------------------------------------

/*
  vector<T,2> VS matrix<T,3>
*/
// Vout = v * m;
template<class T>
inline vector<T,2> operator* (const vector<T,2> & lhs,const matrix<T,3> &rhs){
	return
    	vector<T,2>(
        	lhs[0]*rhs[0][0] + lhs[1]*rhs[1][0] + rhs[2][0],
            lhs[0]*rhs[0][1] + lhs[1]*rhs[1][1] + rhs[2][1]
        );
}
// Vout = m * v;
template<class T>
inline vector<T,2> operator* (const matrix<T,3> & lhs, const vector<T,2> &rhs){
	return
    	vector<T,2>(
        	lhs[0][0]*rhs[0] + lhs[0][1]*rhs[1] + lhs[0][2],
            lhs[1][0]*rhs[0] + lhs[1][1]*rhs[1] + lhs[1][2]
        );
}
// v *= m;
//(v = v * m)
template<class T>
inline vector<T,2>& operator*= (vector<T,2> & lhs, const matrix<T,3> &rhs){
	T tmp = lhs[0];

    lhs[0] = tmp*rhs[0][0] + lhs[1]*rhs[1][0] + rhs[2][0];
    lhs[1] = tmp*rhs[0][1] + lhs[1]*rhs[1][1] + rhs[2][1];

	return lhs;
}
// transform(m,v);
//(v = m * v)
template<class T>
inline void transform(
	vector<T,2> * out,
	const matrix<T,3> &lhs,
	const vector<T,2> &rhs
){
	(*out)[0] = lhs[0][0]*rhs[0] + lhs[0][1]*rhs[1] + lhs[0][2];
	(*out)[1] = lhs[1][0]*rhs[0] + lhs[1][1]*rhs[1] + lhs[1][2];
}


//-----------------------------------------------

/*
  vector<T,3> VS matrix<T,3>
*/
// Vout = v * m;
template<class T>
inline vector<T,3> operator* (const vector<T,3> &lhs,const matrix<T,3> &rhs){
	return
    	vector<T,3>(
        	lhs[0]*rhs[0][0] + lhs[1]*rhs[1][0] + lhs[2]*rhs[2][0],
            lhs[0]*rhs[0][1] + lhs[1]*rhs[1][1] + lhs[2]*rhs[2][1],
            lhs[0]*rhs[0][2] + lhs[1]*rhs[1][2] + lhs[2]*rhs[2][2]
        );
}
// Vout = m * v;
template<class T>
inline vector<T,3> operator* (const matrix<T,3> &lhs, const vector<T,3> &rhs){
	return
    	vector<T,3>(
        	lhs[0][0]*rhs[0] + lhs[0][1]*rhs[1] + lhs[0][2]*rhs[2],
            lhs[1][0]*rhs[0] + lhs[1][1]*rhs[1] + lhs[1][2]*rhs[2],
            lhs[2][0]*rhs[0] + lhs[2][1]*rhs[1] + lhs[2][2]*rhs[2]
        );
}
// v *= m;
//(v = v * m)
template<class T>
inline vector<T,3>& operator*= (vector<T,3> &lhs, const matrix<T,3> &rhs){
	T tmp_0 = lhs[0];
    T tmp_1 = lhs[1];

    lhs[0] = tmp_0*rhs[0][0] + tmp_1*rhs[1][0] + lhs[2]*rhs[2][0];
    lhs[1] = tmp_0*rhs[0][1] + tmp_1*rhs[1][1] + lhs[2]*rhs[2][1];
    lhs[2] = tmp_0*rhs[0][2] + tmp_1*rhs[1][2] + lhs[2]*rhs[2][2];

	return lhs;
}
// transform(m,v);
//(v = m * v)
template<class T>
inline void transform(
	vector<T,3> *out,
	const matrix<T,3> &lhs,
	const vector<T,3> &rhs
){
    (*out)[0] = lhs[0][0]*rhs[0] + lhs[0][1]*rhs[1] + lhs[0][2]*rhs[2];
    (*out)[1] = lhs[1][0]*rhs[0] + lhs[1][1]*rhs[1] + lhs[1][2]*rhs[2];
    (*out)[2] = lhs[2][0]*rhs[0] + lhs[2][1]*rhs[1] + lhs[2][2]*rhs[2];
}

//-----------------------------------------------

//-----------------------------------------------
/*
  vector<T,3> VS matrix<T,4>
*/
// Vout = v * m;
template<class T>
inline vector<T,3> operator* (const vector<T,3> &lhs,const matrix<T,4> &rhs){
	return
		vector<T,3>(
        	lhs[0]*rhs[0][0] + lhs[1]*rhs[1][0] + lhs[2]*rhs[2][0]  + rhs[3][0],
            lhs[0]*rhs[0][1] + lhs[1]*rhs[1][1] + lhs[2]*rhs[2][1]  + rhs[3][1],
            lhs[0]*rhs[0][2] + lhs[1]*rhs[1][2] + lhs[2]*rhs[2][2]  + rhs[3][2]
        );
}
// Vout = m * v;
template<class T>
inline vector<T,3> operator* (const matrix<T,4> &lhs, const vector<T,3> &rhs){
	return
    	vector<T,3>(
        	lhs[0][0]*rhs[0] + lhs[0][1]*rhs[1] + lhs[0][2]*rhs[2] + lhs[0][3],
            lhs[1][0]*rhs[0] + lhs[1][1]*rhs[1] + lhs[1][2]*rhs[2] + lhs[1][3],
            lhs[2][0]*rhs[0] + lhs[2][1]*rhs[1] + lhs[2][2]*rhs[2] + lhs[2][3]
        );
}
// v *= m;
//(v = v * m)
template<class T>
inline vector<T,3>& operator*= (vector<T,3> &lhs, const matrix<T,4> &rhs){
	T tmp_0 = lhs[0];
    T tmp_1 = lhs[1];

    lhs[0] = tmp_0*rhs[0][0] + tmp_1*rhs[1][0] + lhs[2]*rhs[2][0] + rhs[3][0];
    lhs[1] = tmp_0*rhs[0][1] + tmp_1*rhs[1][1] + lhs[2]*rhs[2][1] + rhs[3][1];
    lhs[2] = tmp_0*rhs[0][2] + tmp_1*rhs[1][2] + lhs[2]*rhs[2][2] + rhs[3][2];
	
	return lhs;
}


// transform(m,v);
//(v = m * v)
template<class T>
inline void transform(
	vector<T,3> *out,
	const matrix<T,4> &lhs,
	const vector<T,3> &rhs
){
	(*out)[0] = (lhs[0][0]*rhs[0] + lhs[0][1]*rhs[1] + lhs[0][2]*rhs[2] + lhs[0][3]);
    (*out)[1] = (lhs[1][0]*rhs[0] + lhs[1][1]*rhs[1] + lhs[1][2]*rhs[2] + lhs[1][3]);
    (*out)[2] = (lhs[2][0]*rhs[0] + lhs[2][1]*rhs[1] + lhs[2][2]*rhs[2] + lhs[2][3]);
}

//-----------------------------------------------
//-----------------------------------------------

/*
  vector<T,4> VS matrix<T,4>
*/

// Vout = v * m;
template<class T>
inline vector<T,4> operator* (const vector<T,4> &lhs,const matrix<T,4> &rhs){
	return
		vector<T,4>(
        	lhs[0]*rhs[0][0] + lhs[1]*rhs[1][0] + lhs[2]*rhs[2][0] + lhs[3]*rhs[3][0],
            lhs[0]*rhs[0][1] + lhs[1]*rhs[1][1] + lhs[2]*rhs[2][1] + lhs[3]*rhs[3][1],
            lhs[0]*rhs[0][2] + lhs[1]*rhs[1][2] + lhs[2]*rhs[2][2] + lhs[3]*rhs[3][2],
			lhs[0]*rhs[0][3] + lhs[1]*rhs[1][3] + lhs[2]*rhs[2][3] + lhs[3]*rhs[3][3]
        );
}
// Vout = m * v;
template<class T>
inline vector<T,4> operator* (const matrix<T,4> & lhs, const vector<T,4> &rhs){
	return
		vector<T,4>(
        	lhs[0][0]*rhs[0] + lhs[0][1]*rhs[1] + lhs[0][2]*rhs[2] + lhs[0][3]*rhs[3],
            lhs[1][0]*rhs[0] + lhs[1][1]*rhs[1] + lhs[1][2]*rhs[2] + lhs[1][3]*rhs[3],
            lhs[2][0]*rhs[0] + lhs[2][1]*rhs[1] + lhs[2][2]*rhs[2] + lhs[2][3]*rhs[3],
			lhs[3][0]*rhs[0] + lhs[3][1]*rhs[1] + lhs[3][2]*rhs[2] + lhs[3][3]*rhs[3]
        );
}
// v *= m;
//(v = v * m)
template<class T>
inline vector<T,4>& operator*= (vector<T,4> & lhs, const matrix<T,4> &rhs){	
	T tmp_0 = lhs[0];
    T tmp_1 = lhs[1];
	T tmp_2 = lhs[2];
	
    lhs[0] = tmp_0*rhs[0][0] + tmp_1*rhs[1][0] + tmp_2*rhs[2][0] + lhs[3]*rhs[3][0];
    lhs[1] = tmp_0*rhs[0][1] + tmp_1*rhs[1][1] + tmp_2*rhs[2][1] + lhs[3]*rhs[3][1];
    lhs[2] = tmp_0*rhs[0][2] + tmp_1*rhs[1][2] + tmp_2*rhs[2][2] + lhs[3]*rhs[3][2];
	lhs[3] = tmp_0*rhs[0][3] + tmp_1*rhs[1][3] + tmp_2*rhs[2][3] + lhs[3]*rhs[3][3];

	return lhs;
}


// transform(m,v);
//(v = m * v)
template<class T>
inline void transform(const matrix<T,4> & lhs, const vector<T,4> &rhs,vector<T,4> &out){
	(*out)[0] = lhs[0][0]*rhs[0] + lhs[0][1]*rhs[1] + lhs[0][2]*rhs[2] + lhs[0][3]*rhs[3];
    (*out)[1] = lhs[1][0]*rhs[0] + lhs[1][1]*rhs[1] + lhs[1][2]*rhs[2] + lhs[1][3]*rhs[3];
    (*out)[2] = lhs[2][0]*rhs[0] + lhs[2][1]*rhs[1] + lhs[2][2]*rhs[2] + lhs[2][3]*rhs[3];
	(*out)[3] = lhs[3][0]*rhs[0] + lhs[3][1]*rhs[1] + lhs[3][2]*rhs[2] + lhs[3][3]*rhs[3];
}

	
template<class T, std::size_t Sz>
inline void transform(
	vector<T,Sz> *out,
	const quaternion<T> & q,
	const vector<T,Sz> &v
){
	
	//out = q * v * q^(-1)
	quaternion<T> tq( q*quaternion<T>(0,v)*~q );
	
	(*out)[0] = tq[1];
	(*out)[1] = tq[2];
	(*out)[2] = tq[3];
	
	return;
}
	


}//end of namespace tempest

#endif

