#ifndef _TMSAFETY_VECTOR_HPP
#define _TMSAFETY_VECTOR_HPP
/** 
	@file safety_vector.hpp
	@brief Defines tempest::safety_vector<T,Sz>.
	@author ototoi / Toru Matsuoka
	@date 2004/05/17 
*/

#include<cmath>
#include<stdexcept>//
#include<memory>//std::allocator
#include<new>//placement new
#include<iterator>

#include<ostream>
#include<sstream>

#include"vector_base.hpp"

namespace tempest{

/** 
 *  @class safety_vector
 *  @brief Safety_vector is more safe than vector. 
 *  @todo 
 *  @code
 *  tempest::safety_vector<double,3> v;
 *  @endcode    
 *  @bug    
 *  @attention This class is slow.
 */
	
template<class T, std::size_t Sz, class Al >class safety_vector;
	
template<class T, std::size_t Sz, class Al = std::allocator<T> >
class safety_vector:public vector_base< safety_vector<T, Sz, Al> , T, Sz > {
	
public:
	
	//-----------------------------------------------
	//type defines
	typedef T	 										value_type;
	typedef safety_vector<T,Sz,Al>						this_type;

	typedef typename Al::template rebind<T>::other		Allocator;
	typedef Allocator									allocator_type;
	
	typedef typename Allocator::pointer 				pointer;
	typedef typename Allocator::const_pointer 			const_pointer;
    typedef typename Allocator::reference				reference;
    typedef typename Allocator::const_reference			const_reference;
    typedef typename Allocator::size_type				size_type;
    typedef typename Allocator::difference_type			difference_type;
	
	typedef pointer 									iterator;		//std::iterator<std::random_access_iterator_tag,T,difference_type>
	typedef const_pointer 								const_iterator;	//std::iterator<std::random_access_iterator_tag,const T,difference_type>
	typedef std::reverse_iterator<iterator>				reverse_iterator;
	typedef std::reverse_iterator<const_iterator>		const_reverse_iterator;
	
	//-----------------------------------------------
public://static members...
	static const size_type c_size = Sz;//container size	
private:
	template<class IT>
	static void destroy(IT first,IT last){
		while(first != last){first->~T();++first;}		
	}
	
	class array_releaser{
	private://members
		pointer m_first;
		pointer m_constructed;
		Allocator & allocator;
	public:
		array_releaser(Allocator & al):allocator(al),
		m_first( allocator.allocate(c_size,0) ),
		m_constructed(m_first){}
	
		~array_releaser(){
			if(m_first == 0)return;
			destroy(m_first,m_constructed);		
			allocator.deallocate(m_first,c_size);
		}
	private:	
		template<class IT>
		void construct_impl(IT a, IT b,pointer first){	
			while(a != b){
				new(static_cast<void*>(m_constructed) ) T(*a);//allocator.construct(first,*a);
				++a;m_constructed++;
			}	
		}
		
		template<class IT,class TAG>	//general
		void construct_tag_dispatch(IT a, IT b,TAG ){
			pointer last = m_first + c_size;
			while(m_constructed != last){
				if(a != b){
					new(static_cast<void*>(m_constructed) ) T(*a);//allocator.construct(first,*a);
					++a;
				}else{
					new(static_cast<void*>(m_constructed) ) T();
				}				
				m_constructed++;
			}
		}
		
		template<class IT>				//random access
		void construct_tag_dispatch(IT a, IT b,std::random_access_iterator_tag ){
			std::ptrdiff_t dist = b-a;
			if(dist < c_size){
				construct_impl(a,b,m_first);//
				construct();
			}else{
				construct_impl(a,a+c_size,m_first);
			}	
		}
	public:
		void construct(){	
			pointer last = m_first + c_size;
			while(m_constructed != last){new(m_constructed) T();m_constructed++;}	
		}
		template<class IT>
		void construct(IT a, IT b){	
			typedef typename std::iterator_traits<IT>::iterator_category tag;
			construct_tag_dispatch(a,b,tag());	
		}

		pointer release(){
			pointer p = m_first;
			m_first = m_constructed = 0;
			return p;
		}
		
};
    
public:
	//functions for iterator
	pointer			begin()			{return element;}
	pointer			end()			{return element+c_size;}
	const_pointer	begin()	const	{return element;}
	const_pointer	end()	const	{return element+c_size;}
	
	reverse_iterator		rbegin()		{return reverse_iterator(end());}
	reverse_iterator		rend() 			{return reverse_iterator(begin());}
	const_reverse_iterator	rbegin()const	{return const_reverse_iterator(end());}
	const_reverse_iterator	rend() 	const 	{return const_reverse_iterator(begin());}
	
	//front & back functions
	//
	reference		front()			{return *begin();}
	reference		back()			{return *end();}
	const_reference	front()	const	{return *begin();}
	const_reference	back()	const	{return *end();}
	
	//
	//
	Allocator get_allocator()const{
		return allocator;
	}
	
	//constructor
	//
	explicit safety_vector(const Allocator& al = Allocator() ):allocator(al){
		array_releaser tmp(allocator);
		tmp.construct();
		element = tmp.release();
	}
	
	safety_vector(const this_type& rhs):allocator(rhs.get_allocator()){
		array_releaser tmp(allocator);
		tmp.construct(rhs.begin(),rhs.end());
		element = tmp.release();
	}
	
	safety_vector(const this_type& rhs, const Allocator& al):allocator(al){
		array_releaser tmp(allocator);
		tmp.construct(rhs.begin(),rhs.end());
		element = tmp.release();
	}
	
	template<class IT>
	safety_vector(IT first,IT last,const Allocator& al = Allocator()):allocator(al){
		array_releaser tmp(allocator);
		tmp.construct(first,last);
		element = tmp.release();
	}
	
	explicit safety_vector(value_type const (&array)[c_size] , const Allocator& al = Allocator()):allocator(al){
		array_releaser tmp(allocator);
		tmp.construct(&(array[0]),&(array[c_size]));
		element = tmp.release();
	}
	
	template<class Self>
	explicit safety_vector(const vector_base<Self,T,Sz> & rhs,const Allocator& al = Allocator()):allocator(al){
		array_releaser tmp(allocator);
		tmp.construct(static_cast<const Self &>(rhs).begin(),static_cast<const Self &>(rhs).end());
		element = tmp.release();
	}	
	
	~safety_vector(){
		//if(element == 0)return;
		destroy(element,element+c_size);
		allocator.deallocate(element,c_size);
	}
	
	void swap(this_type& rhs) {
		std::swap(element,rhs.element);
	}
	
	//-----------------------------------------------
    //inserters
	this_type& operator= (const this_type &rhs){
		swap(  this_type(rhs)  );
		return *this;
	}
	
	template<class Self>
	this_type& operator= (const vector_base<Self,T,Sz> & rhs){
		this_type temp(*this);
		std::copy(static_cast<const Self &>(rhs).begin(),static_cast<const Self &>(rhs).end(),temp.begin());
		swap(temp);
		return *this;
	}
	
	template<class IT>
	void assign(IT first, IT last){
		swap( this_type(first,last) );
	}
	
	//-----------------------------------------------
	//capacity
	size_type size ()     const { return Sz; }
    size_type max_size () const { return Sz; }
    bool      empty ()    const { return false; }
	
	//-----------------------------------------------
	//operators
	
	this_type& negate(){
		this_type temp(*this);
		for(size_type i = 0;i < Sz;i++){
			temp[i] = -temp[i];
		}
		swap(temp);
		return *this;
	}

/**
  @def DECLARE_OP_EQUAL( OP )
  Defines "safety_vector<T,Sz,Al> OP= safety_vector<T,Sz,Al>".
*/	
#define DECLARE_OP_EQUAL( OP )								\
	this_type& operator OP (const this_type& rhs){			\
		this_type temp(*this);								\
		for(size_type i = 0;i < Sz;i++){					\
			temp[i] OP rhs[i];								\
		}													\
		swap(temp);											\
		return *this;										\
	}
	
	DECLARE_OP_EQUAL(+=)
	DECLARE_OP_EQUAL(-=)
	DECLARE_OP_EQUAL(*=)
	DECLARE_OP_EQUAL(/=)
	
#undef DECLARE_OP_EQUAL 

/**
  @def DECLARE_OP_EQUAL_SCALER( OP )
  Defines Defines "safety_vector<T,Sz> \OP= T".
*/

#define DECLARE_OP_EQUAL_SCALER( OP )				\
	this_type& operator OP (const value_type &rhs){	\
		this_type temp(*this);						\
		for(size_type i = 0;i < Sz;i++){			\
			temp[i] OP rhs;							\
		}											\
		swap(temp);									\
		return *this;								\
	}
	
	DECLARE_OP_EQUAL_SCALER(*=)
	DECLARE_OP_EQUAL_SCALER(/=)

#undef DECLARE_OP_EQUAL_SCALER
	
	//
	
	T & operator[](size_type i){
    	return element[i];
	}
	
	const T& operator[](size_type i) const{
		return element[i];
	}
	
	T& at(size_type i){
		if(Sz<=i){throw std::out_of_range("tempest::safety_vector");}
    	return element[i];
	}
	const T& at(size_type i)const{
		if(Sz<=i){throw std::out_of_range("tempest::safety_vector");}
    	return element[i];
	}
	
	//-----------------------------------------------
	//utilities
	
    value_type length() const {
		return std::sqrt( sqr_length() );
	}
	
	value_type sqr_length() const {
		T temp = T();
    	for(size_type i=0;i<Sz;i++){ temp += element[i] * element[i]; }
		return temp;
	}
	
	value_type sum() const {
		T temp = T();
    	for(size_type i=0;i<Sz;i++){ temp += element[i]; }
		return temp;
	}
	
	
	this_type& normalize(){
		T length = sqr_length(); 		//||V||^2
    	if (length == T()) return *this;

		length = T(1)/std::sqrt(length);	// 1 / ||V||
		//length = rsqrt(length);
		swap((this_type(*this) *= length));

    	return *this;
	}

    //-----------------------------------------------
	//debug code
//#ifdef _DEBUG
public:
    const char* debug()const{return "tempest::safety_vector<T,Sz>";}
//#endif
private:	
	Allocator allocator;
	pointer element;		
}; 

//-----------------------------------------------	
// utility functions

template<class T,std::size_t Sz, class A>
inline T length(const safety_vector<T,Sz,A> &rhs){
	return rhs.length();
}

template<class T,std::size_t Sz, class A>
inline T sqr_length(const safety_vector<T,Sz,A> &rhs){
	return rhs.sqr_length();
}
	
template<class T,std::size_t Sz, class A>
inline T sum(const safety_vector<T,Sz,A> &rhs){
	return rhs.sum();
}
	
template<class T,std::size_t Sz, class A>
inline safety_vector<T,Sz,A> normalize(const safety_vector<T,Sz,A> &rhs){
	T length = rhs.sqr_length(); 		//||V||^2
	//if (length == T()) return rhs;

	length = static_cast<T>(1)/std::sqrt(length);	// 1 / ||V||
	//length = rsqrt(length);
	return safety_vector<T,Sz,A>(rhs) *= length;
}

template<class T,std::size_t Sz, class A>
inline T dot(const safety_vector<T,Sz,A> &lhs,const safety_vector<T,Sz,A> &rhs){
	T temp = T();
	for(std::size_t i = 0;i < Sz;i++){temp += (lhs[i]*rhs[i]);}
	return temp;
}

	
//-----------------------------------------------
//unary operators
template<class T,std::size_t Sz,class Al>
inline const safety_vector<T,Sz,Al>
	operator+ (const safety_vector<T,Sz,Al> & rhs){
	return rhs;
}
template<class T,std::size_t Sz,class Al>
inline safety_vector<T,Sz,Al> operator- (const safety_vector<T,Sz,Al> & rhs){
	return safety_vector<T,Sz,Al>(rhs).negate();
}

//-----------------------------------------------
//binary operators
template<class T,std::size_t Sz,class Al> 
inline safety_vector<T,Sz,Al> operator+ (const safety_vector<T,Sz,Al> &lhs, const safety_vector<T,Sz,Al> &rhs){
	return safety_vector<T,Sz,Al>(lhs) += rhs;
}
template<class T,std::size_t Sz,class Al>
inline safety_vector<T,Sz,Al> operator- (const safety_vector<T,Sz,Al> &lhs, const safety_vector<T,Sz,Al> &rhs){
	return safety_vector<T,Sz,Al>(lhs) -= rhs;
}
template<class T,std::size_t Sz,class Al>
inline safety_vector<T,Sz,Al> operator* (const safety_vector<T,Sz,Al> &lhs, const safety_vector<T,Sz,Al> &rhs){
	return safety_vector<T,Sz,Al>(lhs) *= rhs;
}
template<class T,std::size_t Sz,class Al>
inline safety_vector<T,Sz,Al> operator/ (const safety_vector<T,Sz,Al> &lhs, const safety_vector<T,Sz,Al> &rhs){
	return safety_vector<T,Sz,Al>(lhs) /= rhs;
}
	
//-----------------------------------------------
//specific scalar
	
template<class T, class X, std::size_t Sz, class Al> 
inline safety_vector<T,Sz,Al> operator* (const X lhs,const safety_vector<T,Sz,Al> & rhs){ 
	return safety_vector<T,Sz,Al>(rhs) *= lhs ; 
}
	
template<class T, class X, std::size_t Sz, class Al> 
inline safety_vector<T,Sz,Al> operator* (const safety_vector<T,Sz,Al> &lhs,const X rhs){ 
	return safety_vector<T,Sz,Al>(lhs) *= rhs ;
}

template<class T, class X, std::size_t Sz, class Al> 
inline safety_vector<T,Sz,Al> operator/ (const safety_vector<T,Sz,Al> &lhs,const X rhs){ 
	return safety_vector<T,Sz,Al>(lhs) /= rhs ;
}
	
//--------------------------------------------------
//compare

template<class T,std::size_t Sz,class Al>
inline bool operator==(const safety_vector<T,Sz,Al> &lhs,const safety_vector<T,Sz,Al> &rhs){
	for(std::size_t i = 0;i<Sz;++i){if(lhs[i]!=rhs[i])return false;}
	return true;
}
template<class T,std::size_t Sz,class Al>
inline bool operator!=(const safety_vector<T,Sz,Al> &lhs,const safety_vector<T,Sz,Al> &rhs){
	return !(lhs==rhs);
}

#if 0
template<class T,std::size_t Sz,class Al>
inline bool operator< (const safety_vector<T,Sz,Al> &lhs,const safety_vector<T,Sz,Al> &rhs){
	return lhs.sqr_length() < rhs.sqr_length();
}
template<class T,std::size_t Sz,class Al>
inline bool operator> (const safety_vector<T,Sz,Al> &lhs,const safety_vector<T,Sz,Al> &rhs){
	return lhs.sqr_length() > rhs.sqr_length();
}
template<class T,std::size_t Sz,class Al>
inline bool operator<= (const safety_vector<T,Sz,Al> &lhs,const safety_vector<T,Sz,Al> &rhs){
	return !(lhs > rhs);
}
template<class T,std::size_t Sz,class Al>
inline bool operator>= (const safety_vector<T,Sz,Al> &lhs,const safety_vector<T,Sz,Al> &rhs){
	return !(lhs < rhs);
}
	
#endif

//-----------------------------------------------
//output

/** 
 *	ostream << 
 */
template<typename T, std::size_t Sz, class Al, typename _CharT, class _Traits>
std::basic_ostream<_CharT, _Traits>& operator<<(std::basic_ostream<_CharT, _Traits>& os, const safety_vector<T,Sz,Al>& rhs){
	
	std::basic_ostringstream<_CharT, _Traits> s;
	s.flags(os.flags());
	s.imbue(os.getloc());
	s.precision(os.precision());
	s << "(";
	for(std::size_t i = 0; i < Sz-1; ++i){
		s << rhs[i] <<",";
	}
	s <<rhs[Sz-1] <<")";
	return os << s.str();
}

	
}
#endif


